/*
gcc -framework GLUT -framework OpenGL -framework Cocoa -lm glviewmesh.c ../tpm/mesh.c -O3
gcc -framework GLUT -framework OpenGL -framework Cocoa -lm glviewmesh.c ../tpm/mesh.c ../tpm/obs.c ../tpm/ephems.c
echo ~/Proj/TPM/rq36/1999RQ36.shape1.2008dec30.obj ~/Proj/TPM/rq36/rq36.obs 0.0 -90.0 4.0 0.0 0.0 | ./a.out
 */

#ifdef __APPLE__
  #include <GLUT/glut.h>
#else
  #include <GL/glut.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h> 
#include <math.h>
#include <pthread.h> 
#include <tiffio.h>
#include <unistd.h>

#include "../tpm/mesh.h"
#include "../tpm/ephems.h"
#include "../tpm/obs.h"
#include "../tms/constants.h"
#include "../tms/macros.h"

#define CHECKVIS  1
#define WIREFRAME 2
#define TRIANGLES 4
#define AXES      8
#define TEMPFILE 16

MESH* pmesh; // point to the mesh
OBS*  pobs=NULL;  // point to the observations
int   Nobs=0, CurrentObs=0;  // number of observations and Number of current obs for RenederMesh
double pixsize=1.0; // pix size in mas
double fieldofview=-1.0;
int Npix=512, NValidObs=0;
FILE *fh=NULL, *ofh=NULL;
double* pima2d=NULL;
double glupixelarea=0.0;
int mode=0;
char visfilename[512];
FILE *visfh=NULL;
float *ptemp=NULL;
char  tempfilename[512];
double tempepoch=0;

// temperature lookup table
float ltred[512];
float ltblue[512];
float ltgreen[512];

// temperature range [min:max]
float tempmax, tempmin;

// rotated Sun Vector
VECTOR vRotatedSunN;

int whiteBG=0; // use black background by default

void display(void);

// this is a copy and paste of the function from tpm.c
int PrintSubSolarSubObserver(OBS* pobs)
{
  VECTOR vSunN, v;
  double lambda, beta, rxy;

  normalize(pobs->vs, &v); 
  memcpy(&vSunN, &v, sizeof(VECTOR));
  rxy=sqrt(v.x*v.x + v.y*v.y); lambda=atan2(v.y, v.x)*RADEG;
  if (lambda<0) lambda+=360.0;
  if (rxy!=0)
    beta=atan(v.z/rxy)*RADEG;
  else 
    { 
      if (v.z>0) beta=90.0;
      else beta=-90.0;
    }
  printf("ss> %18.10lf %6.1lf %6.1lf %14.10lf\t", pobs->JD, lambda, beta, NORMA(pobs->vs));
	  
  normalize(pobs->vo, &v);
  rxy=sqrt(v.x*v.x + v.y*v.y); lambda=atan2(v.y, v.x)*RADEG;
  if (lambda<0) lambda+=360.0;
  if (rxy!=0)
    beta=atan(v.z/rxy)*RADEG;
  else 
    { 
      if (v.z>0) beta=90.0;
      else beta=-90.0;
    }
  printf("so> %6.1lf %6.1lf %14.10lf %14.10lf\n", 
	 lambda, beta, NORMA(pobs->vo), RADEG*acos(SCALAR(v,vSunN)));
  return 0;
}


int writetiff(char *filename, char *description,
  int x, int y, int width, int height, int compression)
{
  TIFF *file;
  GLubyte *image, *p;
  int i;

  file = TIFFOpen(filename, "w");
  if (file == NULL) {
    return 1;
  }
  image = (GLubyte *) malloc(width * height * sizeof(GLubyte) * 3);

  /* OpenGL's default 4 byte pack alignment would leave extra bytes at the
     end of each image row so that each full row contained a number of bytes
     divisible by 4.  Ie, an RGB row with 3 pixels and 8-bit componets would
     be laid out like "RGBRGBRGBxxx" where the last three "xxx" bytes exist
     just to pad the row out to 12 bytes (12 is divisible by 4). To make sure
     the rows are packed as tight as possible (no row padding), set the pack
     alignment to 1. */
  glPixelStorei(GL_PACK_ALIGNMENT, 1);

  glReadPixels(x, y, width, height, GL_RGB, GL_UNSIGNED_BYTE, image);
  TIFFSetField(file, TIFFTAG_IMAGEWIDTH, (uint32) width);
  TIFFSetField(file, TIFFTAG_IMAGELENGTH, (uint32) height);
  TIFFSetField(file, TIFFTAG_BITSPERSAMPLE, 8);
  TIFFSetField(file, TIFFTAG_COMPRESSION, compression);
  TIFFSetField(file, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
  TIFFSetField(file, TIFFTAG_SAMPLESPERPIXEL, 3);
  TIFFSetField(file, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
  TIFFSetField(file, TIFFTAG_ROWSPERSTRIP, 1);
  TIFFSetField(file, TIFFTAG_IMAGEDESCRIPTION, description);
  p = image;
  for (i = height - 1; i >= 0; i--) {
    if (TIFFWriteScanline(file, p, i, 0) < 0) {
      free(image);
      TIFFClose(file);
      return 1;
    }
    p += width * sizeof(GLubyte) * 3;
  }
  TIFFClose(file);
  return 0;
}


void buildlut()
{
  int i;
  // builds the lookup table
  for(i=0; i<512; i++)
    {
      if (i<256) ltred[i]=(float)i/256.0; else ltred[i]=1.0;
      if (i<256) ltblue[i]=1.0-(float)i/256.; else ltblue[i]=0.0;
      if (i>255) ltgreen[i]=((float)i-256.)/256; else ltgreen[i]=0.0;
    }
  // builds the lookup table
  /*	for(i=0; i<512; i++)
	ltgreen[i]=ltblue[i]=ltred[i]=(float)i/512.0;
  */
}

void init(void) 
{
  
  if (whiteBG)
    glClearColor (1.0, 1.0, 1.0, 1.0); // white bg
  else
    glClearColor (0.0, 0.0, 0.0, 0.0); // black bg

   glShadeModel(GL_FLAT);
   glDisable(GL_LIGHTING);
   glEnable(GL_DEPTH_TEST);
   buildlut();
}

void readtempfile(float **pptemp, char *filename, double epoch)
{
  FILE *tmapfh=NULL;
  int facetidex=0, i=0;
  double fileepoch;
  float fdummy;
  float *ptemp;
  float temp;
  char linebuf[10000];
  char arrangeT=0;
  double lambdaSun, betaSun;
  int isSunDef=0;
  
  ptemp=*pptemp;
  if (tempmax==-1) arrangeT=1;
  
  if (arrangeT==1)
    {
      tempmax=0; 
      tempmin=10000;
    }

  tmapfh=fopen(filename,"rt");
  if (tmapfh==NULL)
    {
      printf("could not open temperature file\n");
      return;
    }

  if (ptemp) free(ptemp);
  if ((ptemp=(float*)malloc(pmesh->fn*sizeof(float)))==NULL)
    {
      printf("could not allocate temperature buffer\n");
      return;
    }

  while (!feof(tmapfh))
    {
      if (fgets(linebuf,10000,tmapfh)==NULL) break;
      sscanf(linebuf,"%lf %f %f %d %f %f %f %lf %lf %f", &fileepoch, &fdummy, &fdummy, &facetidex,
	     &fdummy, &fdummy, &fdummy, &lambdaSun, &betaSun, &temp);
      if ((fileepoch==epoch) && (facetidex>=0) && (facetidex<pmesh->fn))
	{
	  ptemp[facetidex]=temp;
	  if ((temp>tempmax) && (arrangeT==1)) tempmax=temp;
	  if ((temp<tempmin) && (arrangeT==1)) tempmin=temp;

	  if (isSunDef) continue;
	  printf("SunVec> %lf %lf", lambdaSun, betaSun);
	  lambdaSun*=(3.1416/180.);
	  betaSun/=(180*3.1416);
	  vRotatedSunN.x=cos(lambdaSun)*cos(betaSun);
	  vRotatedSunN.y=sin(lambdaSun)*cos(betaSun);
	  vRotatedSunN.z=sin(betaSun);
	  printf(" -- %lf %lf %lf\n", vRotatedSunN.x, vRotatedSunN.y, vRotatedSunN.z);
	  isSunDef=1;
	}
    }
  
  // return pointers
  *pptemp=ptemp;

  fclose(tmapfh);
  printf("red temperature map: MAX=%f; MIN=%f\n",tempmax, tempmin);
  
  for (i=0; i<pmesh->fn; i++)
    ptemp[i]-=tempmin;

  for (i=0; i<pmesh->fn; i++)
    ptemp[i]/=(tempmax-tempmin);
}

void readTempBinFile(float **pptemp, char *filename, double epoch)
{
  FILE *tmapfh=NULL;
  int facetidex=0, i=0;
  double fileepoch;
  float fdummy;
  float *ptemp;
  float temp;
  char linebuf[10000];
  char arrangeT=0;
  
  ptemp=*pptemp;
  if (tempmax==-1) arrangeT=1;
  
  if (arrangeT==1)
    {
      tempmax=0; 
      tempmin=10000;
    }

  tmapfh=fopen(filename,"rb");
  if (tmapfh==NULL)
    {
      printf("could not open temperature file\n");
      return;
    }

  if (ptemp) free(ptemp);
  if ((ptemp=(float*)malloc(pmesh->fn*sizeof(float)))==NULL)
    {
      printf("could not allocate temperature buffer\n");
      return;
    }

  while (!feof(tmapfh))
    {
      fread(&fileepoch,sizeof(double),1,tmapfh);
      printf("epoch %lf \n",fileepoch);
      //      sleep(1);
      if ((fabs(fileepoch-epoch)<1.e-6)) {
	fread(ptemp,sizeof(*ptemp),pmesh->fn,tmapfh);
	break;
      } else
	fseek(tmapfh, sizeof(*ptemp)*pmesh->fn, SEEK_CUR );
    }
  
  // return pointers
  *pptemp=ptemp;

  fclose(tmapfh);

  for (i=0; i<pmesh->fn; i++)
    {
      if ((ptemp[i]>tempmax) && (arrangeT==1)) tempmax=ptemp[i];
      if ((ptemp[i]<tempmin) && (arrangeT==1)) tempmin=ptemp[i];
    }
  
  printf("red temperature map: MAX=%f; MIN=%f\n",tempmax, tempmin);
  
  for (i=0; i<pmesh->fn; i++)
    {
      ptemp[i]-=tempmin;
      if (ptemp[i]<0) ptemp[i]=0;
    }
  
  for (i=0; i<pmesh->fn; i++)
    ptemp[i]/=(tempmax-tempmin);
}


void readTempBinFileAndMakeTiffs(float **pptemp, char *filename, double epoch)
{
  FILE *tmapfh=NULL;
  int facetidex=0, i=0;
  double fileepoch;
  float fdummy;
  float *ptemp;
  float temp;
  char linebuf[10000];
  char arrangeT=0;
  long counter=0;
  
  ptemp=*pptemp;
  if (tempmax==-1) arrangeT=1;
  
  if (arrangeT==1)
    {
      tempmax=0; 
      tempmin=10000;
    }

  tmapfh=fopen(filename,"rb");
  if (tmapfh==NULL)
    {
      printf("could not open temperature file\n");
      return;
    }

  if (ptemp) free(ptemp);
  if ((ptemp=(float*)malloc(pmesh->fn*sizeof(float)))==NULL)
    {
      printf("could not allocate temperature buffer\n");
      return;
    }

  // return pointers
  *pptemp=ptemp;

  while (!feof(tmapfh))
    {
      fread(&fileepoch,sizeof(double),1,tmapfh);
      fread(ptemp,sizeof(*ptemp),pmesh->fn,tmapfh);

      if  (arrangeT==1)
	for (i=0; i<pmesh->fn; i++)
	  {
	    if ((ptemp[i]>tempmax)) tempmax=ptemp[i];
	    if ((ptemp[i]<tempmin)) tempmin=ptemp[i];
	  }
      
      printf("red temperature map: MAX=%f; MIN=%f EPOCH=%lf\n",tempmax, tempmin,fileepoch);
  
      for (i=0; i<pmesh->fn; i++)
	{
	  ptemp[i]-=tempmin;
	  if (ptemp[i]<0) ptemp[i]=0;
	}
      
      for (i=0; i<pmesh->fn; i++)
	ptemp[i]/=(tempmax-tempmin);

      
      display();
      sprintf(linebuf,"meshTemp_%6.6ld.tiff",counter);
      writetiff(linebuf, "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);
      //      sleep(1);
      counter++;
    }

  
  fclose(tmapfh);

}



void oldsetlights(VECTOR pos)
{
   GLfloat mat_specular[] = { 1.0, 1.0, 1.0, 1.0 };
   GLfloat mat_shininess[] = { 50.0 };
   GLfloat light_position[] = { 10.0, 10.0, 10.0, 0.0 };
   glClearColor (0.0, 0.0, 0.0, 0.0);
   glShadeModel (GL_SMOOTH);

   glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
   glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
   glLightfv(GL_LIGHT0, GL_POSITION, light_position);

   glEnable(GL_LIGHTING);
   glEnable(GL_LIGHT0);
   glEnable(GL_DEPTH_TEST);
}

void setlights(VECTOR pos)
{
  /*
  GLfloat specular[] = {.8,.8,.8,1.0};
  GLfloat shinines[] = {100.0};
  //  GLfloat position[] = {(float)pos.x, (float)pos.y, (float)pos.z, 1.0}; //{1.0,1.0,1.0,1.0};
  GLfloat position[] = { 2.0, 2.0, 2.0, 1.0 };
  GLfloat light_ambient[] =  {0.3, 0.3, 0.3, 1.0};
  GLfloat light_diffuse[] =  {1.0, 1.0, 1.0, 1.0};
  GLfloat light_specular[] =  {1.0, 1.0, 1.0, 1.0};

  glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
  glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
  
  //Propriétés du matériau Face de devant prise en compte pour un éclairage de type SHININESS et SPECULAR.
  printf("light pos is at: x:%f y=%f z=%f w=%f\n", position[0],position[1],position[2],position[3]);
  glMaterialfv(GL_FRONT,GL_SPECULAR,specular);
  glMaterialfv(GL_FRONT,GL_SHININESS,shinines);
  glLightfv(GL_LIGHT0,GL_POSITION,position); //Lumière n°0
  glEnable(GL_LIGHTING); //Autorisation
  glEnable(GL_LIGHT0);
  //	glDepthFunc(GL_LESS);
  glEnable(GL_COLOR_MATERIAL);//Autorise la couleur
  glEnable(GL_DEPTH_TEST);
  
  glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
  glShadeModel(GL_SMOOTH);
  */
}


void display(void)
{
  unsigned long color, i;
  unsigned long ltind;
  float step=pmesh->maxsize/512*1.8;

  printf("now displaying fn=%d; vn=%d\n",pmesh->fn,pmesh->vn);
  //  glEnable(GL_DEPTH_TEST);
  //  glClearDepth(1.0);
  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    if (ptemp)// if there is a temperature map display the color scale bar
    {
      printf("Drawing temperature bar\n");
      //     glPopMatrix();
      //      glLoadIdentity();
      //      glOrtho(-.6, .6, -.6, .6, -.6, .6);
      //      gluLookAt(0,0,1, 0,0,0, 0,1,0); // look down from above z-axis
      glBegin(GL_QUADS);
      for (i=0; i<512; i++)
	{
	  ltind = (int)(i);//*(vis[i]/vismax);
	  glColor3f(ltred[ltind], ltgreen[ltind], ltblue[ltind]);
	  //	  glColor3f(1.0f, 1.0f, 1.0f); // if not uses the boring white color 
	  glVertex3d(pmesh->maxsize*1.5,-pmesh->maxsize*1.5+i*step,-pmesh->maxsize*1.5);
	  glVertex3d(pmesh->maxsize*1.5,-pmesh->maxsize*1.5+i*step,-pmesh->maxsize*1.65);
	  glVertex3d(pmesh->maxsize*1.5,-pmesh->maxsize*1.5+(i+1)*step,-pmesh->maxsize*1.65);
	  glVertex3d(pmesh->maxsize*1.5,-pmesh->maxsize*1.5+(i+1)*step,-pmesh->maxsize*1.5);
	  //      glVertex3d(.5,-.5,0);
      //     glPushMatrix();
	}
      glEnd();   
   }

  if ((mode & CHECKVIS)==CHECKVIS)
  {
      if ((mode & TRIANGLES)==TRIANGLES)
      {
	glBegin(GL_TRIANGLES);
	for (i=0; i<pmesh->fn; i++)	// loop for facets
	  {
	    
	    if (ptemp)// if there is a temperature map to display, it uses the LUT colors
	      {
		ltind = (int)(511.0*ptemp[i]);//*(vis[i]/vismax);
		glColor3f(ltred[ltind], ltgreen[ltind], ltblue[ltind]);
	      }
	    else 
	      glColor3f(0.7f, 0.7f, 0.7f); // if not uses the boring white color 
	    
	    glVertex3d(pmesh->v[pmesh->f[i].v1-1].x,
		       pmesh->v[pmesh->f[i].v1-1].y,
		       pmesh->v[pmesh->f[i].v1-1].z);
	    glVertex3d(pmesh->v[pmesh->f[i].v2-1].x,
		       pmesh->v[pmesh->f[i].v2-1].y,
		       pmesh->v[pmesh->f[i].v2-1].z);
	    glVertex3d(pmesh->v[pmesh->f[i].v3-1].x,
		       pmesh->v[pmesh->f[i].v3-1].y,
		       pmesh->v[pmesh->f[i].v3-1].z);
	  }
 	glEnd();
      }
      if ((mode & WIREFRAME)==WIREFRAME)
	{
	  printf("Drawing WIREFRAME\n");
	for (i=0; i<pmesh->fn; i++)	// loop for facets
	  {
	    glBegin(GL_LINE_LOOP);	// lines == wired mesh
	    glColor3f(0.0f, 0.0f, 1.0f);
	    glVertex3d(pmesh->v[pmesh->f[i].v1-1].x*1.01,
		       pmesh->v[pmesh->f[i].v1-1].y*1.01,
		     pmesh->v[pmesh->f[i].v1-1].z*1.01);
	    glVertex3d(pmesh->v[pmesh->f[i].v2-1].x*1.01,
		       pmesh->v[pmesh->f[i].v2-1].y*1.01,
		       pmesh->v[pmesh->f[i].v2-1].z*1.01);
	    glVertex3d(pmesh->v[pmesh->f[i].v3-1].x*1.01,
		       pmesh->v[pmesh->f[i].v3-1].y*1.01,
		       pmesh->v[pmesh->f[i].v3-1].z*1.01);
	    glEnd();
	  }
	}
     // draw axis
       if ((mode & AXES)==AXES)
	 {
	   glBegin(GL_LINES);
	   glColor3f(1.0f, 0.0f, 0.0f);
	   glVertex3f(0,0,0);
	   glVertex3f(pmesh->maxsize*1.5,0,0);
	   glEnd();
	   
	   glBegin(GL_LINES);
	   glColor3f(0.0f, 1.0f, 0.0f);
	   glVertex3f(0,0,0);
	   glVertex3f(0,pmesh->maxsize*1.5,0);
	   glEnd();
	   
	   glBegin(GL_LINES);
	   glColor3f(0.0f, 0.0f, 1.0f);
	   glVertex3f(0,0,0);
	   glVertex3f(0,0,pmesh->maxsize*1.5);
	   glEnd();

	   // the sun
	   glBegin(GL_LINES);
	   glColor3f(1.0f, 1.0f, 0.0f);
	   glVertex3f(0,0,0);
	   glVertex3f(vRotatedSunN.x*pmesh->maxsize*1.7,
		      vRotatedSunN.y*pmesh->maxsize*1.7,
		      vRotatedSunN.z*pmesh->maxsize*1.7);
	   glEnd();

	 }
    } else // MODE IS not CHECKVIS
    {
      glBegin(GL_TRIANGLES);
      for (i=0; i<pmesh->fn; i++)	// loop for facets
	{
	  //      glBegin(GL_LINE_LOOP);	// lines == wired mesh
	  color=i+1; 
	  //	  glColor3f(1.0f, 1.0f, 1.0f);
	  glColor3ub(color&0xff, (color>>8)&0xff, 0); //(color>>16)&0xff);
	  glVertex3d(pmesh->v[pmesh->f[i].v1-1].x,
		     pmesh->v[pmesh->f[i].v1-1].y,
		     pmesh->v[pmesh->f[i].v1-1].z);
	  glVertex3d(pmesh->v[pmesh->f[i].v2-1].x,
		     pmesh->v[pmesh->f[i].v2-1].y,
		     pmesh->v[pmesh->f[i].v2-1].z);
	  glVertex3d(pmesh->v[pmesh->f[i].v3-1].x,
		     pmesh->v[pmesh->f[i].v3-1].y,
		     pmesh->v[pmesh->f[i].v3-1].z);
	  //glEnd();
	}
      glEnd();	
   }

 
  glFlush();
  glFinish();
  glutSwapBuffers();
}

void lookatmesh(VECTOR DeltaA, VECTOR nA, double pixsize,  double *pima2d, long xsize, long ysize)
{	
  unsigned char	*pix;
  long   j, i, buflen=xsize*ysize;
  	 
  double Delta=sqrt(DeltaA.x*DeltaA.x + DeltaA.y*DeltaA.y + DeltaA.z*DeltaA.z);
  VECTOR DeltaN={DeltaA.x/Delta, DeltaA.y/Delta, DeltaA.z/Delta};
  double fovby2=0.5* xsize * Delta *0.72 * pixsize;
  double s=(DeltaN.x*nA.x+DeltaN.y*nA.y+DeltaN.z*nA.z);
  VECTOR np={ // vector parallel to north vector and perpendicular to DeltaA
    nA.x-DeltaN.x*s,  nA.y-DeltaN.y*s, nA.z-DeltaN.z*s};


  glupixelarea=4.*fovby2*fovby2/xsize/ysize;
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
  //	 glRotated(90, 1,0,0);
  printf("look at mesh: Delta: %lf %lf %lf\t Nv:  %lf %lf %lf\n",
	 DeltaN.x, DeltaN.y, DeltaN.z, np.x, np.y, np.z);
  printf("look at mesh: Delta: %lf \t FOV:  %lf (km) pixsize: %lf (km)\n",
	 Delta, fovby2*2, Delta*0.72*pixsize);
  gluLookAt(DeltaN.x, DeltaN.y, DeltaN.z, 0.0,0.0,0.0, nA.x, nA.y, nA.z);
  display();
  
  pix=(unsigned char*)malloc(buflen*sizeof(unsigned char)*3);
  if (pix==NULL) {printf("error allocating pix GL buffer\n"); exit(-2);}
  memset(pix,0,buflen*sizeof(char)*3);
  glReadPixels(0,0,xsize,ysize,GL_RGB,GL_UNSIGNED_BYTE,pix);
  printf("xsize=%d ysize=%d buflen=%ld; pix_add=%p; pima2d_add=%p\n",xsize,ysize,buflen,pix,pima2d);
 
  for (i=0; i<buflen; i++)
       pima2d[i]=1.0*pix[i*3]+256.0*pix[i*3+1]+65536.0*pix[i*3+2];
 
  free(pix);
}


void RenderMeshCheckVis(void)
{
  VECTOR vnorth={0.0, 0.39777710, 0.91748209};
  VECTOR vObserverAsteroid;
  VECTOR vSunAsteroid;

  double r;

  if (!pobs)
    {
      printf('No observations loaded: RenderMeshCheckVis RETURNING\n');
      return;
    }


  if (CurrentObs>=Nobs) CurrentObs=0; // reset to the initial observation

  PrintSubSolarSubObserver(&pobs[CurrentObs]);

  printf("Rendering Mesh for Viewing at Observation %d of %d, epoch=%lf \n",
	 CurrentObs+1,Nobs,pobs[CurrentObs].JD);
  printf("vo: %lf %lf %lf\n", 
	 pobs[CurrentObs].vo.x, pobs[CurrentObs].vo.y, pobs[CurrentObs].vo.z);
  // copy observer vector from observation
  memcpy(&vObserverAsteroid,&pobs[CurrentObs].vo, sizeof(VECTOR));
  // rotate Observer vector
  Eclip2Ast(&vObserverAsteroid, pmesh->lambda, pmesh->beta, pmesh->P, pmesh->phi0, pmesh->t0, pobs[CurrentObs].JD);

  // copy Sun vector from observation
  memcpy(&vRotatedSunN,&pobs[CurrentObs].vs, sizeof(VECTOR));
  // rotate Sun Vector
  Eclip2Ast(&vRotatedSunN, pmesh->lambda, pmesh->beta, pmesh->P, pmesh->phi0, pmesh->t0, pobs[CurrentObs].JD);
  r=sqrt(vRotatedSunN.x*vRotatedSunN.x + vRotatedSunN.y*vRotatedSunN.y + vRotatedSunN.z*vRotatedSunN.z);
  vRotatedSunN.x/=r;
  vRotatedSunN.y/=r;
  vRotatedSunN.z/=r;

  // rotate North vector
  Eclip2Ast(&vnorth, pmesh->lambda, pmesh->beta, pmesh->P, pmesh->phi0, pmesh->t0, pobs[CurrentObs].JD);
  lookatmesh(vObserverAsteroid, vnorth, pixsize, pima2d, Npix, Npix);

  PrintSubSolarSubObserver(&pobs[CurrentObs]);
  // move index to the next observation
  CurrentObs++;
}

void RenderMeshx()
{
  double fovby2=pmesh->maxsize*2;
  VECTOR x={1,0,0};

  if (fieldofview!=-1) fovby2=fieldofview/2;
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
  setlights(x);
  gluLookAt(1.0,0,0, 0,0,0, 0,0,1);
  display();
}

void RenderMeshxm()
{
  double fovby2=pmesh->maxsize*2;
  VECTOR x={-1,0,0};

  if (fieldofview!=-1) fovby2=fieldofview/2;
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
  setlights(x);
  gluLookAt(-1.0,0,0, 0,0,0, 0,0,1);
  display();
}

void RenderMeshy()
{
  double fovby2=pmesh->maxsize*2;
  VECTOR y={0,1,0};

  if (fieldofview!=-1) fovby2=fieldofview/2;
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
  setlights(y);
  gluLookAt(0,1,0, 0,0,0, 0,0,1);
  display();
}

void RenderMeshz()
{
  double fovby2=pmesh->maxsize*2;
  VECTOR z={0,0,1};

  if (fieldofview!=-1) fovby2=fieldofview/4;
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
  setlights(z);
  gluLookAt(0,0,1, 0,0,0, 0,1,0);
  display();
}

void RenderMeshXXX()
{
  double fovby2=pmesh->maxsize*2;
  VECTOR x={0,0,0};

  if (fieldofview!=-1) fovby2=fieldofview/2;
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
  setlights(x);
  gluLookAt(1.0,0.0,0.0, 0,0,0, 0,0,1);
  display();
  writetiff("mesh.tif", "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);
  exit(-1);
}

int AllocFluxFacetCube(double ***lppMap, int Nfacet, int Ndata)
{
  int iFacet;
  double **lpMap=NULL;

  if ((lpMap=(double**)malloc(sizeof(double)*Nfacet))==NULL)
    {
      printf("AllocFluxFacetCube FAILED\n");
      return -1;
    }

  *lppMap=lpMap; // pass back pointer

  for(iFacet=0; iFacet<Nfacet; iFacet++)
    lpMap[iFacet]=NULL;
  
  for(iFacet=0; iFacet<Nfacet; iFacet++)
    if ( (lpMap[iFacet]=(double*)malloc(sizeof(double)*Ndata))==NULL)
      {
       printf("AllocFluxFacetCube FAILED\n");
       return -2;
      }

  return 0; // all OK. alloc succeded
}

int FreeFluxFacetCube(double **lpMap, int Nfacet)
{
  int iFacet;

  for(iFacet=0; iFacet<Nfacet; iFacet++)
    if (lpMap[iFacet]) free(lpMap[iFacet]);

  free(lpMap);
  lpMap=NULL;
}

void RenderMesh(void)
{
  double **lpflux=NULL;
  VECTOR vnorth={0.0, 0.39777710, 0.91748209}, vNorthAsteroid;
  long iFacet, i, iData;
 
  do 
    {
      if (pobs->lambda) free(pobs->lambda);
      if (pobs->flux) free(pobs->flux);
      if (pobs->eflux) free(pobs->eflux);
      if (fread(&pobs->JD, sizeof(double), 1, fh)<1) break; // read JD
      if (fread(&pobs->vo, sizeof(VECTOR), 1, fh)<1) break; // read vector to the observer
      if (fread(&pobs->vs, sizeof(VECTOR), 1, fh)<1) break; // read vector to the sun
      if (fread(&pobs->flags, sizeof(unsigned int), 1, fh)<1) break; // read obs flags
      if (fread(&pobs->Ndata, sizeof(int), 1, fh)<1) break; // read ndata
      pobs->lambda=(double*)malloc(sizeof(double)*pobs->Ndata); // allocate the wavelengths (Ndata) for the observation
      fread(pobs->lambda, sizeof(double), pobs->Ndata, fh); // read the wavelengths
      pobs->flux=(double*)malloc(sizeof(double)*pobs->Ndata); // allocate the wavelengths (Ndata) for the observation
      fread(pobs->flux, sizeof(double), pobs->Ndata, fh); // read the wavelengths
      pobs->eflux=(double*)malloc(sizeof(double)*pobs->Ndata); // allocate the wavelengths (Ndata) for the observation
      fread(pobs->eflux, sizeof(double), pobs->Ndata, fh); // read the wavelengths
      
      printf("JD=%lf; Ndata=%d\n", pobs->JD, pobs->Ndata);
      printf("Rendering new observation ------------------------------\n");
      printf("#of facets %d\n", pmesh->fn);
      
     // alloc the flux buffer flux[Nfacets][Ndata]
      if (AllocFluxFacetCube(&lpflux, pmesh->fn, pobs->Ndata))
	{
	  printf("Render mesh alloc error THIS IS BIG PROBLEM\n");
	  exit(-2);
	}
       
      // read the flux for each facet and each lambda from the file
      for (iFacet=0; iFacet<pmesh->fn; iFacet++)
	if (fread(lpflux[iFacet],sizeof(double),pobs->Ndata,fh)!=pobs->Ndata) 
	  {
	    printf("fread failed\n"); 
	    break;
	  }  // exit the code if read fails

      // rotate the vector to the observer
      // NOOOOOO: the vector is already rotated as it is stored in the binary file
      //      Eclip2Ast(&pobs->vo, pmesh->lambda, pmesh->beta, pmesh->P, pmesh->phi0, pmesh->t0, pobs->JD);

      // copy North Vector
      memcpy(&vNorthAsteroid, &vnorth, sizeof(VECTOR));
      // rotate North vector
      Eclip2Ast(&vNorthAsteroid,   pmesh->lambda, pmesh->beta, pmesh->P, pmesh->phi0, pmesh->t0, pobs->JD);
 
      for (iData=0; iData<pobs->Ndata; iData++) // build the image cube
	{
	  lookatmesh(pobs->vo, vNorthAsteroid, pixsize, pima2d, Npix, Npix);
	  
	  for (i=0; i<(Npix*Npix); i++) // for each pixel
	      if (pima2d[i] > 0)
		pima2d[i]=(lpflux[(long)pima2d[i]-1][iData])*glupixelarea;
      
	  fwrite(&pobs->JD,sizeof(pobs->JD),1,ofh); 
	  fwrite(&pobs->lambda[iData],sizeof(pobs->lambda[iData]), 1, ofh);
	  fwrite(&pobs->flux[iData],sizeof(pobs->flux[iData]), 1, ofh);
	  fwrite(&pobs->eflux[iData],sizeof(pobs->eflux[iData]), 1, ofh);
	  fwrite(pima2d, sizeof(double), Npix*Npix, ofh); // write the image plane
	  NValidObs++;
	}
      
       //clean up
      if (lpflux) 
	{
	  FreeFluxFacetCube(lpflux, pmesh->fn);
	  lpflux=NULL;
	}

      //      sleep(2);
      printf("....Rendering Completed \n\n");
    } while (!feof(fh));
 
   // here we replace the content of the header
  fseek(ofh, 0, SEEK_SET); // move file pointer back to begin
  fwrite(&Npix, sizeof(Npix), 1, ofh);// number of pixel along X
  fwrite(&Npix, sizeof(Npix), 1, ofh);// numver of pixel along Y
  fwrite(&NValidObs, sizeof(NValidObs), 1, ofh);// number of planes e.g. images to be replaced later
  printf("NpixX=%ld, NpixY=%ld, Nvalid Planes = %ld\n",Npix, Npix, NValidObs);
  
  fclose(fh);
  fclose(ofh);  
  exit(0);
}

void keyboard(unsigned char key, int x, int y)
{
  VECTOR v={1.0,1.0,1.0};
  VECTOR vu={0,0,1};
  double fovby2=pmesh->maxsize*2;
  
  switch (key) {
  case '1':
    glLoadIdentity();
    glOrtho(-fovby2/1, fovby2/1, -fovby2/1, fovby2/1, -fovby2/1, fovby2/1);
    display();
    break;
  case '2':
    glLoadIdentity();
    glOrtho(-fovby2/1.5, fovby2/1.5, -fovby2/1.5, fovby2/1.5, -fovby2/1.5, fovby2/1.5);
    display();
    break;
  case '3':
    glLoadIdentity();
    glOrtho(-fovby2/1.8, fovby2/1.8, -fovby2/1.8, fovby2/1.8, -fovby2/1.8, fovby2/1.8);
    display();
    break;
  case '4':
    glLoadIdentity();
    glOrtho(-fovby2/2, fovby2/2, -fovby2/2, fovby2/2, -fovby2/2, fovby2/2);
    display();
    break;
  case '5':
    glLoadIdentity();
    glOrtho(-fovby2/2.3, fovby2/2.3, -fovby2/2.3, fovby2/2.3, -fovby2/2.3, fovby2/2.3);
    display();
    break;
  case '=':
    glRotatef(10.0,1.0,0.0,0.0);
    glutPostRedisplay();
    break;
  case '-':
    glRotatef(-10.0,1.0,0.0,0.0);
    glutPostRedisplay();
    break;
  case ']':
    glRotatef(10.0,0.0,1.0,0.0);
    glutPostRedisplay();
    break;
  case '[':
    glRotatef(-10.0,0.0,1.0,0.0);
    glutPostRedisplay();
    break;
  case 39: // '
    glRotatef(10.0,0.0,0.0,1.0);
    glutPostRedisplay();
    break;
  case ';':
    glRotatef(-10.0,0.0,0.0,1.0);
    glutPostRedisplay();
    break;
  case 's':
    glutPostRedisplay();
    break;
  case 'S':
    glutPostRedisplay();
    break;
  case 'e':
    RenderMeshCheckVis();
    glutPostRedisplay();
    break;
  case 'E':
    RenderMeshCheckVis();
    glutPostRedisplay();
    writetiff("mesh_atepoch.tif", "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);
    break;
    //  case 'E':
    //    lookatmesh(v, vu, pixsize, pima2d, Npix, Npix);
    //    glutPostRedisplay();
    //    break;
  case 'x':
    RenderMeshx();
    glutPostRedisplay();
    break;
  case 'X':
    RenderMeshxm();
    glutPostRedisplay();
    break;
  case 'y':
    RenderMeshy();
    glutPostRedisplay();
    break;
  case 'z':
    RenderMeshz();
    glutPostRedisplay();
    break;
  case 'T':
    readtempfile(&ptemp, tempfilename, tempepoch);
    glutPostRedisplay();
    break;
  case 'B':
    readTempBinFile(&ptemp, tempfilename, tempepoch);
    glutPostRedisplay();
    break;
  case 'b':
    readTempBinFileAndMakeTiffs(&ptemp, tempfilename, tempepoch);
    break;
  case 't':
    writetiff("mesh.tif", "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);
    printf("file mesh.tif should have been written on the disk\n");
    break;
  case 'A':
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
    gluLookAt(1,0,0, 0,0,0, 0,0,1);
    display();
    writetiff("mesh.xp.tif", "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
    gluLookAt(-1,0,0, 0,0,0, 0,0,1);
    display();
    writetiff("mesh.xm.tif", "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
    gluLookAt(0,1,0, 0,0,0, 0,0,1);
    display();
    writetiff("mesh.yp.tif", "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
    gluLookAt(0,-1,0, 0,0,0, 0,0,1);
    display();
    writetiff("mesh.ym.tif", "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
    gluLookAt(0,0,1, 0,0,0, 0,1,0);
    display();
    writetiff("mesh.zp.tif", "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-fovby2, fovby2, -fovby2, fovby2, -fovby2, fovby2);
    gluLookAt(0,0,-1, 0,0,0, 0,1,0);
    display();
    writetiff("mesh.zm.tif", "OpenGL-rendered asteroid mesh", 0, 0, Npix, Npix, COMPRESSION_LZW);
    break;
  case 27:
    exit(0);
    break;
  default:
    exit(0);
    break;
  }
}

int main(int argc, char** argv)
{
  MESH mesh;
  char fname[512]="mesh.obj";
  char fluxfilename[512]="flux_map.bin";
  char imagefluxfilename[512]="flux_image.bin";
  int res;
  double lscalemesh=1.0;
  char *spinfilename=NULL;
  FILE *spinfile=NULL;
  OBS   obs; // we need to define here at least one observation variable

  pobs=&obs; // the global variable pobs points at that observation. 
             // watch out: lambda and fluxes need to be allocated 

  tempmax=-1.0;
  mode=0;
//	SCANS the arguments
  while (argc>1)
    {
      if (argv[1][0]=='-')
	{
	  switch (argv[1][1])
	    {
	    case 'b':
	      whiteBG=1;
	      break;
	    case 'h': printf("This is glviewmesh\n Look at the source code for switches (main) \n"); exit(0);
	      break;
	    case 'T': 
	      argc--; argv++; strcpy(tempfilename, argv[1]);
	      argc--; argv++; tempepoch=atof(argv[1]);
	      printf("using temperature map %s at epoch %lf\n", tempfilename, tempepoch);
	      mode+=TEMPFILE;
	      break;
	    case 'n': // change number of pixels
	      argc--; argv++; Npix=atoi(argv[1]);
	      printf("Warning: number of pixel changed to:%d\n",Npix);
	      break;
	    case 'p': // change the pixelsize
	      argc--; argv++; pixsize=atof(argv[1]);
	      printf("Warning: pixelsize changed to %lf\n", pixsize);
	      break;
	    case 's': // mesh linear scale factor
	      argc--; argv++; lscalemesh=atof(argv[1]);
	      printf("Warning: mesh scale factor changed to %lf\n", lscalemesh);
	      break;
	    case 'm':
	      argc--; argv++; 
	      strcpy(fname, argv[1]);
	      printf("Warning: mesh filename changed to %s\n", fname);
	      break;
	    case 't': printf("Triangles deactivated \n"); 
	      mode=(mode & ~TRIANGLES);
	      break;
	    case 'w': printf("Wireframe activated \n"); 
	      mode+=WIREFRAME;
	      break;
	    case 'x': printf("Axes deactivated \n"); 
	      mode=(mode & ~AXES);
	      break;
	    case 'r': 	     
	      argc--; argv++; tempmin=atof(argv[1]);
	      argc--; argv++; tempmax=atof(argv[1]);
	      printf("Temperature range [%f:%f]\n",tempmin,tempmax); 
	      break;
	    case 'v':
	      //mode=0; 
	      visfh=NULL;
	      mode+=(CHECKVIS | TRIANGLES | AXES);
	      if (argc<=2) break;
	      argc--; argv++; 
	      strcpy(visfilename, argv[1]);
	      if ((visfilename[0]=='-') || (visfilename[0]=='+')) // switch found // rewind and break
		{
		  printf("Next switch found: rewind and break\n");
		  argc++; argv--;
		  break;
		}
	      visfh=fopen(visfilename, "rt");
	      printf("tried to open the visfile\n");
	      if (visfh==NULL)
		{
		  printf("Error while opening visfilename file\n");
		  exit(-2);
		}
	      printf("visfile %s opened ok\n", visfilename);
	      break;
	    case 'f': // force field of view when MODE_CHECKVIS
	      argc--; argv++; fieldofview=atof(argv[1]);
	      printf("Field of view forced to %lf (mesh units)\n",fieldofview); 	      
	      break;
	    case 'S': 
	      argc--; argv++; 
	      spinfilename=(char*)malloc(strlen(argv[1]));
	      strcpy(spinfilename, argv[1]);
	      printf("using spinvector file %s (DAMIT format) %lf\n", spinfilename);
	      break;
	    case 'o':
	      argc--; argv++; 
	      if (ReadObs(argv[1], &pobs, &Nobs))
		{
		  printf("Read Obs failed \n");
		  exit(-1);
		}
	      break;
	    default:
	      printf("unknown option -%s\n",&argv[1][1]);
	      return 0;
	      break;
	    } //switch
	}
      argc--; argv++;
    }

  //allocate the image
  pima2d=(double*)malloc(Npix*Npix*sizeof(double));
  if (pima2d==NULL)
    {
      printf("ima2d alloc FAILED\n");
      exit(-4);
    }
  printf("Image allocated OK pima2d_add=%p\n" ,pima2d);

  // READ SHAPE ---------------------------------
  //  scanf("%s",fname);
  memset(&mesh,0,sizeof(MESH));
  if ( (res=GenMeshWF(fname, &mesh))<=0)
    {
      printf("GenMeshWF FAILED\n");
      exit(-1);
    }
  printf("mesh %s loaded OK fn=%d; vn=%d\n",fname,mesh.fn,mesh.vn);

  // INITIALIZE THE POLE PARAMETERS of the MESH from the spinfile
  if (spinfilename)
    {
      spinfile=fopen(spinfilename, "rt");
      if (!spinfile)
	{
	  printf("Spinfile not found, WTF!\n");
	  if (spinfilename) free(spinfilename);
	  exit(-1);
	}
      if (fscanf(spinfile,"%lf %lf %lf %lf %lf", &mesh.lambda, &mesh.beta, &mesh.P, &mesh.t0,  &mesh.phi0)!=5)
	return -1;
      mesh.lambda/=RADEG;
      mesh.beta/=RADEG;
      mesh.P/=24.0;
      mesh.phi0/=RADEG;
      printf("lambda=%lf beta=%lf %lf %lf %lf\n", mesh.lambda, mesh.beta, mesh.P, mesh.t0,mesh.phi0);
      pmesh=&mesh;
      if (spinfile) fclose(spinfile);
    }else
    {
      // INITIALIZE THE POLE PARAMETERS of the MESH --- fg3 Scheirisch and Pravec
      mesh.lambda=0/RADEG;
      mesh.beta=-87/RADEG;
      mesh.P=3.6/24.0;
      mesh.phi0=0/RADEG;
      mesh.t0=0; //JD
      pmesh=&mesh;
      printf("%lf %lf %lf %lf %lf\n", pmesh->lambda, pmesh->beta, pmesh->P, pmesh->phi0, pmesh->t0);
    }
  // scale the mesh
  if (lscalemesh !=1.0)
    {
      scalemesh(pmesh,lscalemesh);
      printf("mesh scaled by %lf\n",lscalemesh);   
    }

   // initalize GL
  glutInit(&argc, argv);
  glutInitDisplayMode (GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);
  glutInitWindowSize (Npix, Npix); 
  glutInitWindowPosition (100, 100);
  glutCreateWindow (fname);
  init();

  if ((mode & CHECKVIS)==CHECKVIS)
    {
      printf("Program in check vis mode: press <e>\n");
      RenderMeshx();
    } else

    // to do movies enable this section {} and comment out the next
    // and do eg.
    // l=`awk '$4==0{print $1}' tmap_temps.dat`
    // v=0; for i in $l; do glviewmesh -m sp32.obj -T tmap_temps.dat $i -r 120 380; mv mesh.tif mesh_$i.tif; vv=`echo $v | awk '{printf("%04d",$1)}'`; mv mesh.tif mesh_$vv.tif; ((v++)); done
    // ffmpeg -framerate 20 -f image2 -i mesh_%04d.tif movie.mpg
    /*
    {
      mode+=(CHECKVIS | TRIANGLES | AXES);
      printf("test test \n"); 
      readtempfile(&ptemp, tempfilename, tempepoch);
      glutIdleFunc(RenderMeshXXX);
    }
    */
  
    {
       // get name and open flux map file --------------------------
      scanf("%s", fname);
      fh=fopen(fname, "rb");
      
      // read the TPM hader stored in the file
      fread(&mesh.fn,sizeof(mesh.fn),1,fh);//int
      fread(&mesh.isconcave,sizeof(mesh.isconcave),1,fh);//int
      fread(&mesh.lambda,sizeof(mesh.lambda),1,fh);//double
      fread(&mesh.beta,sizeof(mesh.beta),1,fh);//double
      fread(&mesh.P,sizeof(mesh.P),1,fh);//double
      fread(&mesh.phi0,sizeof(mesh.phi0),1,fh);//double
      fread(&mesh.t0,sizeof(mesh.t0),1,fh);//double
      fread(&mesh.maxsize,sizeof(mesh.maxsize),1,fh); //double

      printf("red header \n");
      printf("fn=%d lambda0=%lf beta0=%lf P=%lf t0=%lf phi0=%lf\n",
	     mesh.fn, mesh.lambda, mesh.beta, mesh.P, mesh.t0, mesh.phi0);
      
      // get name and open output file --------------------------
      scanf("%s", fname);
      ofh=fopen(fname, "wb");
      fwrite(&Npix, sizeof(Npix), 1, ofh);// number of pixel along X
      fwrite(&Npix, sizeof(Npix), 1, ofh);// numver of pixel along Y
      fwrite(&Npix, sizeof(Npix), 1, ofh);// number of planes e.g. images to be replaced later
      fwrite(&pixsize, sizeof(pixsize),1,ofh);// saves mas/pixel
     
      glutIdleFunc(RenderMesh);
    }

  glutDisplayFunc(display); 
  glutKeyboardFunc(keyboard);

  glutMainLoop();
  
  if(pima2d) free(pima2d);
  if (ofh) fclose(ofh);
  if (fh) fclose(fh);
  if (spinfilename) free(spinfilename);
  
  return 0;
}

  /*
  GLuint fbo;
  GLuint depthbuffer;
  GLuint img;
  GLenum glstatus;

  printf("OK.... \n");
  glGenFramebuffersEXT(1, &fbo); // generate the Frame Buffer Object
  printf("OK.... \n");
  glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fbo); // Bind it

  glGenRenderbuffersEXT(1, &depthbuffer); // generate the depth buffer for rendering 3D->2D
  glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, depthbuffer); // bind it

  // Generate the storage space associated with the render buffer
  glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT, Npix, Npix); 
  // attach it to the FBO we created earlier.
  glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, depthbuffer);

  //Add a Texture To Render To
  glGenTextures(1, &img);
  glBindTexture(GL_TEXTURE_2D, img); // bind it
  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, Npix, Npix, 0, GL_RGB, GL_UNSIGNED_BYTE, NULL);

  //attach the texture it to the FBO so we can render to it:
  glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, img, 0);

  // check if the stutus will equal GL_FRAMEBUFFER_COMPLETE_EXT 
  glstatus = glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);

  if (glstatus!=GL_FRAMEBUFFER_COMPLETE_EXT)
    {
      printf("glstatus!=GL_FRAMEBUFFER_COMPLETE_EXT !!\n");
      exit(-1);
    }

  //To render to a FBO you bind it and to stop rendering to it you call the above again with ‘0’ as the final parameter:

  glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fbo);
  glPushAttrib(GL_VIEWPORT_BIT);
  glViewport(0,0,Npix,Npix);

// RENDERING HERE HERE
// code here

   //stop rendering
   glPopAttrib();
   glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
   //clean up
   glDeleteFramebuffersEXT(1, &fbo);
   glDeleteRenderbuffersEXT(1, &depthbuffer);
*/
