/*
This program calculates the Stokes Parameter of the liear polarization
from a number of flux measurments:



To compile it under windows simply issue the Visual Studio Comamnd Prompt
go to the directory where the program is stored and issue the command
> cl f2stokes.c

To run it
> f2stokes -f e:\AdDocs\DataReduction\KoKa\P.txt -v 1


Marco Delbo June 23, 2007
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#define  PI  			3.1415926535897932384626433832795
#define  RAD 			0.017453292519943295769236907684886 // PI/180
#define  LN_10 			2.3025850929940456840179914546844
#define  LN_10by2p25 	0.9210340372
#define USAGE "Usage: f2stokes <filename>\n"

typedef struct tagPOL {
	int			beam;
	double 		flux;
	double 		eflux;
	double 		angle;
	double      SNR;
	struct tagPOL*	next;
} polstruct;

// global variables
int		verbose=0;
int		errormod=0;	//0=formal, 1=from the data
int		percent=0;	//1=display is %

int loadfile(char*, polstruct**, int *Nout);
int wmean(double *xm, double *sigxm, double *x, double *sig, long N);
int pol2stokes(double *q, double *eq, double angoff, polstruct *pol, int N, int Nangle);
void quicksort(double a[], int left, int right);


void main(int argc, char *argv[])
{
double	JD, hours, ave=0.;
int		j, filemode=0;
char	filename[256];
polstruct *pol;
double  q, eq, u, eu;
double  p, ep, theta, etheta, sign;
int		N, Nangle=4;
double  *angle;	// angles

    if (argc < 3)
	{
		printf("%s\n", USAGE);
		return;
	}

	for (j=0; j<argc/2; j++)	// scan additional parameters
	{
		if (strncmp(argv[j*2+1], "-f", 2)==0)
		{
			strcpy(filename, argv[j*2+2]);
			filemode=1;
		}
		if (strncmp(argv[j*2+1], "-v", 2)==0)
			verbose=atoi(argv[j*2+2]);
		if (strncmp(argv[j*2+1], "-a", 2)==0)
			Nangle=atoi(argv[j*2+2]);
		if (strncmp(argv[j*2+1], "-e", 2)==0)
			errormod=atoi(argv[j*2+2]);
		if (strncmp(argv[j*2+1], "-p", 2)==0)
			percent=atoi(argv[j*2+2]);
	}

// load file
  if (filemode)
   if ((loadfile(filename, &pol, &N)) < 0)
   {
	   return;
   }
//
  if (N>10000)
  {
	  printf("I think there is a file error: \n %d are too many lines\n", N);
	  return;
  }

// ============= count angles ===========================================
  angle=malloc(sizeof(double)*N);
  if (!angle)
  {
	  printf("error Idontknow, cannot allocate mem for angle vector\n");
	  printf("%d lines\n", N);
	  return;
  }
  for(j=0; j<N; j++)
	  angle[j]=pol[j].angle;
// sort angles in case they are not sorted
  quicksort(angle,0,N-1);
// now count independent angles
  Nangle=1;
  for(j=0; j<(N-1); j++)
    if ( (angle[j] < angle[j+1]-0.01) || (angle[j] > angle[j+1]+0.01)) Nangle++;
// now release the memory
  free(angle);
  Nangle/=2;

// process pol struct and derive Stokes' parameters ===========================
  if (verbose)
    printf("\n\n\n===STOKES PARAMETER Q ===\n");
  pol2stokes(&q, &eq,  0.0, pol,  N, Nangle);
  if (verbose)
    printf("\n\n\n===STOKES PARAMETER U ===\n");
  pol2stokes(&u, &eu, 22.5, pol,  N, Nangle);

  p=sqrt(u*u+q*q);
  ep=sqrt(pow(u*eu/p,2) + pow(q*eq/p,2));

  if (q >= 0)
  	sign=1.0;
  else
  	sign=-1.0;

  theta=0.5*atan2(u,q); //0.5*sign*acos(q/p);
  etheta = 0.5*1.0/(1. + u/q*u/q) * sqrt( 1/q/q*eu*eu + u*u/q/q/q/q*eq*eq  ); 
    //sqrt( pow(0.5/sqrt(1-q*q/p/p),2)*(1/p/p*ep*ep+q*q/p/p/p/p*ep*ep) );

  if (verbose)
    printf("\n\n\n===Results ===\n");
  printf("Q:%5.4lf+/-%5.4lf_", q, eq);
  printf("U:%5.4lf+/-%5.4lf ", u, eu);
  if (percent)
    printf("P: %5.2lf +/- %5.2lf ", p*100, ep*100);
  else
    printf("P: %5.4lf +/- %5.4lf ", p, ep);
  printf("PA: %6.2lf +/- %6.2lf\n", theta/RAD, etheta/RAD);
}


int pol2stokes(double *q, double *eq, double angoff, polstruct *pol, int N, int Nangle)
{
	double *fo, *efo, *fe, *efe;
	int		i,j, M, *spi, spN=0;
	double *angle, fd, efdq, iam, jam, *sp ;

	q[0]=0.0;
	eq[0]=0.0;

    angle	=malloc(Nangle*sizeof(double));
    fo    	=malloc(Nangle*sizeof(double));
    efo 	=malloc(Nangle*sizeof(double));
    fe  	=malloc(Nangle*sizeof(double));
    efe 	=malloc(Nangle*sizeof(double));

	if ((angle == NULL) || (fo ==NULL) || (efo==NULL) || (fe ==NULL) || (efe==NULL))
	   {
		   printf("ALLOC ERRROR\n");
		   return -301;
	   }

	memset(fo, 0, Nangle*sizeof(double));
	memset(efo, 0, Nangle*sizeof(double));
	memset(fe, 0, Nangle*sizeof(double));
	memset(efe, 0, Nangle*sizeof(double));

	M=0; for (i=0; i<Nangle; i++) M+=(Nangle-1-i);
    sp 	=malloc(M*sizeof(double));
    spi =malloc(M*sizeof(int));


	for (i=0; i<Nangle; i++)
	{
		angle[i]=i*45.0+angoff;
	}

    if (verbose) {
	  printf("%d angles; %d polarim data\n", Nangle, N);
	  printf("Angles at: ");
	  for (i=0; i<Nangle; i++)
	    printf("%6.1f ", angle[i]);
	  printf("\n");
	}

// ------ Stokes' parameter ---------------------------------------------
// flux average
	for (i=0; i<Nangle; i++)
	{
		for (j=0; j<N; j++)
		{
			if ((pol[j].angle >= angle[i]-0.01) && (pol[j].angle <= angle[i]+0.01))
			{
				if (pol[j].beam == 0)	// ordinary
				{
					fo[i]+=pol[j].flux/pol[j].eflux/pol[j].eflux;
					efo[i]+=1.0/pol[j].eflux/pol[j].eflux;
//					printf("%d %6.1lf %e %e\n", pol[j].beam, angle[i], fo[i], efo[i]);
				}
				if (pol[j].beam == 1)	// extraordinary
				{
					fe[i]+=pol[j].flux/pol[j].eflux/pol[j].eflux;
					efe[i]+=1.0/pol[j].eflux/pol[j].eflux;
//					printf("%d %6.1lf %e %e\n", pol[j].beam, angle[i], fe[i], efe[i]);
				}
			}
		}	// Npol

		if (efo[i] != 0.0)
		{
			fo[i]/=efo[i];
			efo[i]=1.0/sqrt(efo[i]);
		}
		if (efe[i] != 0.0)
		{
			fe[i]/=efe[i];
			efe[i]=1.0/sqrt(efe[i]);
		}
	}// Nangle

// dump fluxes
	if (verbose)
	{
		printf("final fluxes and errors\n");
		for (i=0; i<Nangle; i++)
		{
		  printf("ang: %6.1f ord: %10.2lf %8.2lf ext: %10.2lf %8.2lf \n",
		    angle[i], fo[i], efo[i], fe[i], efe[i]);
	    }
	}

	// calc the parameter


	M=0; spN=0;

    if (verbose)
		printf("Stokes parameter calculation:\n");
	for (i=0; i<Nangle; i++)
	{
		iam=fmod(angle[i],90);
		if ((fo[i] > 0) && (fe[i] > 0))
		for (j=i+1; j<Nangle; j++)
		{
			jam=fmod(angle[j],90);
			if ((fo[j] >0) && (fe[j] >0))
			{
				spi[spN]=0;

				fd=0.5*((fo[i]-fe[i]) / (fo[i]+fe[i]) - (fo[j]-fe[j]) / (fo[j]+fe[j]));
				efdq= pow((fe[i]/(fo[i]+fe[i])/(fo[i]+fe[i]))*efo[i], 2) +
				      pow((fo[i]/(fo[i]+fe[i])/(fo[i]+fe[i]))*efe[i], 2) +
				      pow((fe[j]/(fo[j]+fe[j])/(fo[j]+fe[j]))*efo[j], 2) +
				      pow((fo[j]/(fo[j]+fe[j])/(fo[j]+fe[j]))*efe[j], 2);

				if (iam > jam)
				  fd*=-1;
				if ((iam <= jam-0.05) || (iam >= jam+0.05))
				{ q[0]+=fd/efdq; eq[0]+=1.0/efdq; M++; spi[spN]=1;}

				sp[spN]=fd;
				spN++;

				if (verbose)
				  printf("ang: %5.1f - %5.1f : q%1d%1d= %7.5lf +/- %7.5lf :: %5.1f %5.1f -- %d\n",
					angle[i], angle[j], i+1,j+1, fd, sqrt(efdq), iam, jam, spi[spN-1]);
			}
		}
	}
	q[0]/=eq[0];
	eq[0]=1.0/sqrt(eq[0]);

	fd=0.;
	for (i=0; i<spN; i++)
	{
	  if (spi[i] == 1)
	    fd+=((sp[i]-q[0])*(sp[i]-q[0]));
	  else
	    fd+=(sp[i]*sp[i]);
	}
	fd/=spN;
	fd=sqrt(fd);

	if (verbose)
		printf("final q: %lf +/- %lf (%lf) \n", q[0], eq[0], fd);

	switch (errormod){
		case 0:
		break;
		case 1:
		  eq[0]=fd;	// use error calculated as the STDDEV of the q
		break;
	}

	free(angle);
	free(fo );
	free(efo);
	free(fe );
	free(efe);
	free(sp );
	free(spi);

	return 1;
}

int loadfile(char* filename, polstruct **ppol, int *Nout)
{
	FILE	*fh;
	char	buf[256];
	int		beam, N=0, i=0;
	double  flux, eflux, mag, emag, angle, errn=0, Zp, SNR;
	polstruct *pol;

//    printf("converting file %s\n", filename);

	if ((fh=fopen(filename, "rt"))==NULL)
	{
		printf("error -501: Cannot open file\n");
		return -501;
	}

// scan and counts records
	N=0;
    while (!feof(fh))
    {
    	if (NULL==fgets(buf,256,fh))
			break;

		if (buf[0]=='#')
		  continue;
		if ((sscanf(buf, "%d %lf %lf %lf %lf", &beam, &flux, &mag, &emag, &angle))==5)	// Philippe format
//		if ((sscanf(buf, "%d %lf %lf %lf", &beam, &flux, &mag, &emag, &angle))==4)
  			N++;
	}

	fseek(fh, 0, SEEK_SET);
	pol = malloc(sizeof(polstruct)*N);
	*Nout=N;
	*ppol=pol;

    while (!feof(fh))
    {
    	if (NULL==fgets(buf,256,fh))
    	{
//			printf("error -502: Input past end of file\n");
			errn=2;
			break;
		}

 //   	printf(">%s\n", buf);
		if (buf[0]=='#')
		  continue;
		if ((sscanf(buf, "%d %lf %lf %lf %lf", &beam, &flux, &mag, &emag, &angle))<5)
//		if ((sscanf(buf, "%d %lf %lf %lf", &beam, &flux, &eflux, &angle))<4)
		{
//			printf("error -503: Input error while scanning line\n");
			errn= 3;
		} else
		{
			Zp=mag-log10(flux);
			// error on mag: d/dx(log_a u) = 1/(u ln_a) du/dx
			// mag=log_10(f)+Zp -> emag = 1/(f ln_10) eflux -> eflux = emag f ln_10
			eflux = emag * flux * LN_10by2p25;
			SNR=flux/eflux;
			pol[i].beam=beam;
			pol[i].flux=flux;
			pol[i].eflux=eflux;
			pol[i].angle=angle;
			pol[i].SNR=SNR;
			i++;
			if (verbose)
//			  printf("%d %10.2lf %8.2lf %5.1lf %6.2lf\n",
//			  		beam, flux, eflux, SNR, angle);
			  printf("%d %10.2lf %8.2lf %5.1lf %6.3lf %6.3lf %6.2lf %6.3lf\n",
			  		beam, flux, eflux, SNR, mag, emag, angle, Zp);
  			errn=0;
		}
	}

	fclose(fh);

	return errn;
}


/* calculate the Weighted mean and the Weighted standard deviation
*/
int wmean(double *xm, double *sigxm, double *x, double *sig, long N)
{
	long   i;

	xm[0]=0;
	sigxm[0]=0;

	for (i=0; i<N; i++)
	{
		xm[0]    += (x[i]/sig[i]/sig[i]);
		sigxm[0] += (1.0/sig[i]/sig[i]);
	}
	xm[0]/=sigxm[0];
	sigxm[0]=1.0/sqrt(sigxm[0]);

	return 0;
}

void swap(double a[], int i, int j) {
    double tmp = a[i];
    a[i] = a[j];
    a[j] = tmp;
}

int Random(int i, int j) {
    return i + rand() % (j-i+1);
}

void quicksort(double a[], int left, int right) {
    int last = left, i;

    if (left >= right) return;

    swap(a,left,Random(left,right));
    for (i = left + 1; i <= right; i++)
        if (a[i] < a[left])
            swap(a,++last,i);
    swap(a,left,last);
    quicksort(a,left,last-1);
    quicksort(a,last+1,right);
}
