#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "tm.h"
#include "macros.h"

#define NPT 6
#define MA 2
#define SPREAD 0.001

#define NPT_MONTE_CARLO 33
#define MIN_STRLEN	5
#define	NMAX		10000
#define NINT		100

#define USAGE "This is FitTM by Marco Delbo"

#define PROG     "FitTM"
#define AUTHOR   "Marco Delbo"
#define VERSION  "3.2"
#define REL_DATE "June 30, 2009"

#ifdef _SHORT_STRINGS
#define HELP USAGE
#else
#define HELP USAGE"\n\n\
INPUTS:\n\
Model, H, G, emissivity, eta, pV, r, Delta, pha\n\
lambda_1, flux_1, flux_unc_1\n\
lambda_2.........\n\
.......\n\
lambda_N, flux_N, flux_unc_N\n\
\n\
where:\n\
Model is a number equal to:\n\
0 for the NEATM\n\
1 for the NEATM with default (input) value for eta\n\
2 for the STM\n\
3 for the FRM\n\
H and G are the abs. magnitude and\n\
        the slope param of the phase function\n\
        Bowell et al., 1998\n\
pV  is the guess geometric visible albedo\n\
eta is the initial value of the beaming parameter\n\
r     (AU) is the heliocentric distance of the asteroid\n\
delta (AU) is the distance to the observer\n\
pha (degrees) is the phase angle\n\
lambda (micron) is the wavelenght\n\
flux  (Jy) is the monocromatic flux density\n\
flux_unc (Jy) is the uncertainty of f\n\n\
  [N is the number of measurments]\n\
\n\
these values can be written in an ASCII file (fittm.ini)\n\
example\n\
0 15.99  0.15 0.9 1.38 0.05 4.078 3.520 12.7\n\
15.769 0.00121 0.00003\n\
22.327 0.00219 0.00004\n\
\n\
The file can be piped in:\n\
fittm < fittm.ini\n\
\n\
OUPUTS:\n\
 Diameter, diameter uncertainty, pV, pV uncertainty, eta, eta uncertainty, chi^2 of the fit.\n\
\n\
ADDITIONAL Parameters\n\
to be input as line arguments:\n\
e.g. fittm -v -n 1000 < fittm.ini\n\
-b value: Changes de default value of the beta_e parameter of the STM.\n\
   Default is 0.01 mag/deg.\n\
-e include the uncertainty on H and G values in the error analysis.\n\
   The input values have to be changed to:\n\
   H, eH, G, eG, epsilon, eta, pV, r, Delta, pha\n\
   [followed by lambda, fluxes, fluxes_uncertanties] as usual.\n\
-f filename: Read input from file. Same as < filename.\n\
-v VERBOSE mode: the code outputs a lot of info during the fit.\n\
-n <number> Number of iteration in Monte Carlo error estimation.\n\
   = 0 for NO estimation of the uncertainities\n\
-V <V to Bands ratio> add the reflected light component\n"
#endif

extern double gasdev(long *idum);

double *w, *f, *ef;	// wavelength, flux and flux_uncertainty vectors
double H=19.8, eH=0.0; 	// H value and its uncertainty
double G=0.15;
double Delta=0.10232;	// distance from the earth
double r=1.075;		// distance from the sun
double pha=26.0;	// phase angle

double pV=0.11;		// Geometric Albedo
int    Nw=NPT;
double R=2.62/2.;	// Radius of the asteroid in km (1.fit param)
double eta=1.0;		// eta value
double epsilon=0.9;	// emissivity
int    N=50;		// number of iterations
double eR=0, eD=0.0, 	// uncertainties
       epV=0.0, eeta=0.0,
       mD=0.0, mpV=0.0, // mean values
       meta=0.0;
int    TRun=0, TModel=2;
double chisq;
char   name[256];
int    verbose=0;
double reflVtoBandRatio=1.4; // ratio of the reflectivity betwenn V and a thermal wavelenght


double mean(double *x, long N)
{
	long	i;
	double	xm=0.0;

	for (i=0; i<N; i++)
		xm+=x[i];

	xm/=(double)N;

	return	xm;
}

double stddev(double *x, double xm, long N)
{
	long	i;
	double	variance=0.0;

	for (i=0; i<N; i++)
		variance+=(x[i]-xm)*(x[i]-xm);

	variance/=(double)(N-1);

	return sqrt(variance);
}

double Chisq(double *y, double *ey, double *My, int n)
{
	int		i;
	double  chisq=0.;

	for (i=0; i<n; i++)
	  chisq+=((y[i]-My[i])*(y[i]-My[i])/ey[i]);
	return chisq;
}

void Fit(TM *ptm, GEOM *pgeom, double *pw, double *pf, double *pef, long Nw, double *pchisq)
{
	double	R=ptm->D/2;
	double	eR=0;

  	if (verbose)
	  printf("ptm->model = %d\tptm->flags = %d\n",ptm->model, ptm->flags);

	switch (ptm->model)
	{
		case 4 ://STM with user input eta (kind of NEATM-fix)
			STMFit(pw, pf, pef, Nw, &R, &eR, &ptm->pV, &ptm->epV,
				ptm->H, &ptm->eta, &ptm->eeta, ptm->epsilon,
				pgeom->Delta, pgeom->r, pgeom->pha,
				ptm->G, pchisq, NINT, ptm->flags);
			break;

		case 3 ://FRM
			ptm->eta=1.0;
			FRMFit(pw, pf, pef, Nw, &R, &eR, &ptm->pV, &ptm->epV,
				ptm->H, &ptm->eta, &ptm->eeta, ptm->epsilon,
				pgeom->Delta, pgeom->r, pgeom->pha,
				ptm->G, pchisq, NINT, ptm->flags);
			break;

		case 2:	//STM
			ptm->eta=0.756;
			STMFit(pw, pf, pef, Nw, &R, &eR, &ptm->pV, &ptm->epV,
				ptm->H, &ptm->eta, &ptm->eeta, ptm->epsilon,
				pgeom->Delta, pgeom->r, pgeom->pha,
				ptm->G, pchisq, NINT, ptm->flags);
			break;

		case 1:	//NEATM with fixed eta value
			// this implies that eta won't be fitted.
			NEATMFitFix(pw, pf, pef, Nw, &R, &eR, &ptm->pV, &ptm->epV,
				ptm->H, &ptm->eta, &ptm->eeta, ptm->epsilon,
				pgeom->Delta, pgeom->r, pgeom->pha,
				ptm->G, pchisq, NINT, ptm->flags);
			break;
		case 0:	//NEATM
			NEATMFit(pw, pf, pef, Nw, &R, &eR, &ptm->pV, &ptm->epV,
				ptm->H, &ptm->eta, &ptm->eeta, ptm->epsilon,
				pgeom->Delta, pgeom->r, pgeom->pha,
				ptm->G, pchisq, NINT, ptm->flags);
			break;
		default:
			printf("\nINVALID MODEL error 105");
			break;
	}

	ptm->D=2*R;
//	ptm->eD=2*eR;
}

int MCFit(TM *ptm, GEOM *pgeom, double *pw, double *pf, double *pef, long Nw, long Nhits)
{
	double	H, G;
	double *pVa=NULL, *pDa=NULL, *peta=NULL;
	double *pff;				// flux + random deviates
	long	idum;
	long	i, j;
	TM	mctm;
	double	chisq=0.0;
	double	pV, D, eta;

	if (verbose)
	{
		printf("\nnow performing Monte Carlo\n");
		printf("Number of iterations: %d\n", Nhits);
	}

	//---------------------ALLOC-----------------------------------
	// alloc buffer where to store pV
	if((pVa=(double*)malloc(Nhits*sizeof(double)))==NULL)
	{
		printf("\n memory alloc error 201");
		return -201;
	}
	// alloc buffer where to store D
	if((pDa=(double*)malloc(Nhits*sizeof(double)))==NULL)
	{
		printf("\n memory alloc error 202");
		return -202;
	}
	// alloc buffer where to store eta
	if((peta=(double*)malloc(Nhits*sizeof(double)))==NULL)
	{
		printf("\n memory alloc error 203");
		return -203;
	}
	// alloc buffer where to store flux + its random deviates
	if((pff=(double*)malloc(Nw*sizeof(double)))==NULL)
	{
		printf("\n memory alloc error 205");
		return -205;
	}

	// fill the mctm structure
	memcpy(&mctm, ptm, sizeof(TM));

	// Monte Carlo method to estimate deviates on pV D and eta
	// uses the Numerical Recepies gasdev routine which is based on
	// the ran1() uniform number generator
	// initialize idum
	idum=(long)-time( NULL );
	// loop on i
	for (i=0; i<Nhits; i++)	// loop to perform the Monte Carlo
	{
		do{
			mctm.H=ptm->H+ptm->eH*gasdev(&idum);	// alter H
			mctm.G=ptm->G+ptm->eG*gasdev(&idum);	// alter G

			for (j=0; j<Nw; j++)	// alter flux values but does not allow them to go below zero
			{
				do {
					pff[j]=pf[j]+pef[j]*gasdev(&idum);
				} while (pff[j] <= 0);
			}

			mctm.eta=ptm->eta;	// reset to initial values for pV and
			mctm.pV=ptm->pV;	// eta before calling the functions that fits the thermal model
			Fit(&mctm, pgeom, pw, pff, pef, Nw, &chisq); // perform the thermal model fit using corrupted fluxes

		} while((isnan(mctm.pV)) || (mctm.pV >= 1) || (mctm.pV <= 0) );

		peta[i]=mctm.eta;	// stores results into the arrays
		pDa[i]=mctm.D;
		pVa[i]=mctm.pV;

		if (verbose)
			printf("#%d of %d D=%8.3lf pV=%5.3lf eta=%5.3lf csq=%7.3lf\n", i, Nhits, pDa[i], pVa[i], peta[i], chisq);
	}

	D=mean(pDa, Nhits);			// calc mean diameter
	ptm->eD=stddev(pDa, D, Nhits);		// calc standard deviation of the diameter

	pV=mean(pVa, Nhits);			// calc mean albedo
	ptm->epV=stddev(pVa, pV, Nhits);	// calc standard deviation of the albedo

	eta=mean(peta, Nhits);			// calc mean eta value
	ptm->eeta=stddev(peta, eta, Nhits);	// calc standard deviation of the eta value


	// ------------ when finished dealloc the buffers ----------------------
	if (pVa) free(pVa);
	if (pDa) free(pDa);
	if (peta) free(peta);
	if (pff) free(pff);

	if (verbose)
	{
		printf("<D>=%f eD=%f\n", D, ptm->eD);
		printf("<pV>=%f epV=%f\n", pV, ptm->epV);
		printf("<eta>=%f eeta=%f\n", eta, ptm->eeta);
	}
}


// input parameters:
// H, G, guess pV, r, Delta, alpha
//

// USAGE e.g.
// fittm -2 14.80 0.15 0.2 1.2494 1.1813 49.0 0.756 11.7 12.5 10.7 0.034 0.023 0.018 0.001 0.002 0.001
// fittm -2 14.80 0.15 0.2 1.2494 1.1813 49.0 0.756 11.7 0.034 0.001 12.5 0.023 0.002 10.7 0.018 0.001
void dumpname()
{
	printf("%s by %s version %s made on %s\n", PROG,AUTHOR,VERSION,REL_DATE);
}


main(int argc, char* argv[])
{
	TM	tm;
	double *pw, *pf, *pef;	// wavelength, flux and flux_uncertainty vectors
	GEOM	geom;
	int	i, status=1;
	char*	filename=NULL;
	FILE*	fh;
	long	NMC= NPT_MONTE_CARLO;
	int	nolabels=0;
	int	hgerr=0;


//  FILE	*stream0 = freopen( "freopen.out", "w", stdout ); //IT WORKS!!
//	FILE	*stream  = freopen( "freopen.out", "w", stderr ); //IT WORKS!!

	memset(&tm,0,sizeof(TM));
	tm.model=-1;
	tm.betaE=0.01;	// default is 0.01 mag/deg

	while (argc>1)
	{
		if (argv[1][0]=='-')
		{
    		  switch (argv[1][1])
    		  {
			case 'b': 	// change the value of the betaE
				argc--; argv++; tm.betaE=atof(argv[1]);
				break;
			case 'e':	//requires errors on H and G
				hgerr=1;
				break;
			case 'f': 	// get the filename
				argc--; argv++; filename=argv[1];
				break;
			case 'h':
				printf("%s\n", HELP);
				return -1;
				break;
			case 'n': 	// get the number of montecarlo points
				argc--; argv++; NMC=atol(argv[1]);
				break;
			case 'm': 	// get the model to fit
				argc--; argv++; tm.model=atol(argv[1]);
				break;
			case 'v':
				  verbose=1;
				  tm.flags+=TM_MODE_VERBOSE;
				break;
			case 'p':
				  tm.flags+=TM_MODE_FIXALBEDO;
				break;
		        case 'V':
		                 argc--; argv++; 
				 reflVtoBandRatio=atof(argv[1]);
				 tm.flags+=TM_MODE_ADDREFLCOMP;
			       break;
			default:
				printf("unknown option -%s\n",argv[1][1]);
				return 0;
				break;
		  }//switch
		}//if
	    argc--; argv++;
	 }//while

	if (verbose)
	  dumpname();

	if ((verbose) && (filename))
	  printf("using input file %s\n", filename);

	if (filename)
	  stdin=fopen(filename, "rt");

	if (tm.model<0)
	  scanf("%d", &tm.model);

	if (verbose)
	{
		switch (tm.model)
		{
		case 0 :
			printf("using the NEATM\n");
			break;
		case 1 :
			printf("using the NEATM with user defined eta\n");
			break;
		case 2 :
			printf("using the STM\n");
			break;
		case 3 :
			printf("using the FRM\n");
			break;
		case 4 :
			printf("using the STM with user defined eta\n");
			break;
		}
	}


    // reads H, G, epsilon, eta, guess_pV,
	if (hgerr)
	  scanf("%lf %lf %lf %lf %lf %lf %lf", &tm.H, &tm.eH, &tm.G, &tm.eG, &tm.epsilon, &tm.eta, &tm.pV);
	else
	  scanf("%lf %lf %lf %lf %lf", &tm.H, &tm.G, &tm.epsilon, &tm.eta, &tm.pV);

//applies the fixed pV mode if the guess pV is negative
	if (tm.pV<0)
	{
		if (verbose) printf("v> using fixed albedo mode\n");
		tm.pV=fabs(tm.pV); //change sign
		tm.flags += TM_MODE_FIXALBEDO;
	}

	if (verbose) {
	  if (hgerr)
	    printf("H=%lf eH=%lf G=%lf eG=%lf eps=%lf eta=%lf pV(guess)=%lf\n", tm.H, tm.eH, tm.G, tm.eG, tm.epsilon, tm.eta, tm.pV);
	  else
	    printf("H=%lf G=%lf eps=%lf eta=%lf pV(guess)=%lf\n", tm.H, tm.G, tm.epsilon, tm.eta, tm.pV);
	}


    // reads r, Delta, pha
	scanf("%lf %lf %lf", &geom.r, &geom.Delta, &geom.pha);
	if (verbose)
	  printf("r= %lf delta= %lf pha= %lf\n", geom.r, geom.Delta, geom.pha);

// alloc flux flux unc and wavelength vectors
	pf=malloc(NMAX*sizeof(double));
	pef=malloc(NMAX*sizeof(double));
	pw=malloc(NMAX*sizeof(double));

	if ((pf==NULL) || (pef==NULL) || (pw==NULL))
	{
		printf("malloc Error\n");
		if (pf) free(pf);
		if (pef) free(pef);
		if (pw) free(pw);
		return -2;
	}

// get data
    	Nw=0;
	while (scanf("%lf %lf %lf", &pw[Nw], &pf[Nw], &pef[Nw])==3)
	{
	    Nw++;
	}

	if (verbose)
	{
	  printf("#of data=%d\n", Nw);
	  printf("lambda\t\tflux\t\terror\n", Nw);
 	  for (i=0; i<Nw; i++)
	 	printf("%lf\t%lf\t%lf\n", pw[i], pf[i], pef[i]);
	}

	if (NMC)
	  MCFit(&tm, &geom, pw, pf, pef, Nw, NMC);

	Fit(&tm, &geom, pw, pf, pef, Nw, &chisq);
	printf("l>    D(km)   sigmaD     pV    sigma_pV	 eta   sigma_eta   chi^2\n");
	printf("o> %8.3lf %7.3lf\t %5.3lf %4.3lf\t %5.3lf %4.3lf\t %5.1lf\n",
		   tm.D, tm.eD, tm.pV, tm.epV, tm.eta, tm.eeta, chisq);

}


