// this is the source file containing the simple thermal models
// STM, FRM, NEATM
// and the fucntions to fit the STM, FRM, NEATM to fluxes
// fitTM is the driver program
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "macros.h"
#include "constants.h"
#include "tm.h"

#define		ETA_0 0.2
#define		ETA_1 8.2
#define		ETA_S 0.1

// STM flux in Jy /////////////////////////////////////////////////////////////////////////
int   STMflux_Jy(double	*w,	// wavelength. vector in um
		double 	*f,	// flux vector
		int	nw,	// number of elements in w vector
		double 	Tmax,	// Subsolar point temperature
		double 	epsilon,	// emissivity
		double 	R,		// Radius of the asteroid in km
		double 	betaE,	// thermal IR phase coefficient [1]
		double 	Delta,	// distance from the earth
		double 	alfa,	// phase angle
		int	N)		// number of iterations
{

	int	i,k;
	double	theta;				// angle
	double	T;				// actual T on the circle
	double  d = Delta * AUm;		// Delta in m
	double	fac;
	double	dtheta = DPIBY2/(double)N;

	if (N > 1000) N=1000;

	memset(f, 0, nw*sizeof(double));
	for(i=1; i<N; i++)				// for any elem. circ aureola (R=1)
	{								// find the angle of the elemenatry...
		theta = dtheta*((double)i+0.5);
		T = Tmax * pow(cos(theta), .25);	// find its temperature

		for(k=0; k<nw; k++)
		  f[k] += PLANCK_M_Jy(w[k]*1.e-6, T) *	// add the planck flux *
		    cos(theta) *				// angle from the line of sight *
		    sin(theta)*D2PI*dtheta;		// extension of the surface
	}
	// check  : by Putting a comment on the PLANCK line and removing the..
	// ...      mutiplication by fac produce the projected area of the
	// ...		unit sphere = pi
	//          Commenting also the cos(theta) line you get the surface
	// ...		of a half of the unit sphere

	fac = R*R*1.e+6*		    // emitting surface of the Object (1/2)
	  epsilon /					// emissivity of the surface
	  (DPI*d*d)*				// distance from the earth
	  pow(MAGBASE, -betaE*fabs(alfa));// thermal IR correction

	for(k=0; k<nw; k++)
	  f[k] *= fac;

	return 0;
}

// STM flux in W/m-2/um ///////////////////////////////////////////////////////////
int   STMflux_W (double *w,	// wavelength. vector in um
				  double *f,	// flux vector
				  int	 nw,	// number of elements in w vector
			      double Tmax,	// Subsolar point temperature
			      double epsilon,	// emissivity
				  double R,		// Radius of the asteroid in km
			      double betaE,	// thermal IR phase coefficient [1]
			      double Delta,	// distance from the earth
			      double alfa,	// phase angle
			      int	 N)		// number of iterations
{

	int		i,k;
	double	theta;					// angle
	double	T;						// actual T on the circle
	double  d = Delta * AUm;		// Delta in m
	double	fac;
	double	dtheta = DPIBY2/(double)N;

	if (N > 1000) N=1000;

	memset(f, 0, nw*sizeof(double));
	for(i=1; i<N; i++)				// for any elem. circ aureola (R=1)
	{								// find the angle of the elemenatry...
		theta = dtheta*((double)i+0.5);
		T = Tmax * pow(cos(theta), .25);	// find its temperature

		for(k=0; k<nw; k++)
		  f[k] += PLANCK_M_W(w[k]*1.e-6, T) *	// add the planck flux *
		    cos(theta) *				// angle from the line of sight *
		    sin(theta)*D2PI*dtheta;		// extension of the surface
	}
	// check:   by Putting a comment on the PLANCK line and removing the..
	// ...      mutiplication by fac produce the projected area of the
	// ...		unit sphere = pi
	//          Commenting also the cos(theta) line you get the surface
	// ...		of a half of the unit sphere

	fac = R*R*1.e+6*		    // emitting surface of the Object (1/2)
	  epsilon /					// emissivity of the surface
	  (DPI*d*d)*				// distance from the earth
	  pow(MAGBASE, -betaE*fabs(alfa));// thermal IR correction

	for(k=0; k<nw; k++)
	  f[k] *= fac;

	return 0;
}

// FRM flux in Jy ///////////////////////////////////////////////////////////
int FRMflux_Jy(double *w,	// wavelength vector in um
				double *f,	// flux vector
				int	   nw,  // number of elements in the w vector
				double Tmax,// Subsolar point temperature
				double epsilon,// emissivity
				double R,	// Radius of the asteroid in km
				double Delta,// distance from the earth
				int	   N)	// number of iterations
{
	int		i,k;
// this time theta is the latidue respect the spin vector
// and the sun is in the asteroid equatorial plane
	double	theta;			// angle
	double	dtheta=DPIBY2/(double)N;
	double	T;				// actual T on the circle
	double  d = Delta * AUm;	// Delta in m
	double	da, fac;

	if (N > 1000) N=1000;

	memset(f, 0, nw*sizeof(double));
	for(i=0; i<N; i++)		// for any elem. circ aureola (R=1)
	{
		theta = dtheta*((double)i+0.5);
		da = cos(theta)*dtheta*	// half a circle viewd by obs
			 cos(theta);			// projection cosine
		T = Tmax * pow(cos(theta), .25);	// find its temperature

		for(k=0; k<nw; k++)
		  f[k] += PLANCK_M_Jy(w[k]*1.e-6, T) *	// add the planck flux *
		        da;
	}

	fac = 4.*R*R*1.e+6*
	  epsilon /					// emissivity of the surface
	  (DPI*d*d);				// distance from the earth

	for(k=0; k<nw; k++)
	  f[k] *= fac;

	return 0;
}

// FRM flux in W/m-2/um ///////////////////////////////////////////////////////////
int  FRMflux_W(double *w,	// wavelength vector in um
				double *f,	// flux vector
				int	   nw,  // number of elements in the w vector
				double Tmax,// Subsolar point temperature
				double epsilon,// emissivity
				double R,	// Radius of the asteroid in km
				double Delta,// distance from the earth
				int	   N)	// number of iterations
{
	int		i,k;
// this time theta is the latidue respect the spin vector
// and the sun is in the asteroid equatorial plane
	double	theta;			// angle
	double	dtheta=DPIBY2/(double)N;
	double	T;				// actual T on the circle
	double  d = Delta * AUm;	// Delta in m
	double	da, fac;

	if (N > 1000) N=1000;

	memset(f, 0, nw*sizeof(double));
	for(i=0; i<N; i++)		// for any elem. circ aureola (R=1)
	{
		theta = dtheta*((double)i+0.5);
		da = cos(theta)*dtheta*	// half a circle viewd by obs
			 cos(theta);			// projection cosine
		T = Tmax * pow(cos(theta), .25);	// find its temperature

		for(k=0; k<nw; k++)
		  f[k] += PLANCK_M_W(w[k]*1.e-6, T) *	// add the planck flux *
		        da;
	}

	fac = 4.*R*R*1.e+6*
	  epsilon /					// emissivity of the surface
	  (DPI*d*d);				// distance from the earth

	for(k=0; k<nw; k++)
	  f[k] *= fac;

	return 0;
}

// NEATM flux in Jy ///////////////////////////////////////////////////////////
int NEATMflux_Jy(double *w,	// wavelength. vector in um
		 double *f,	// flux vector
		 int	nw,	// number of elements in w vector
		 double Tmax,	// Subsolar point temperature
		 double epsilon,// emissivity
		 double R,	// Radius of the asteroid in km
		 double Delta,	// distance from the earth
		 double alfa,	// phase angle
		 int	N)	// number of iterations
{

	int	i,j,k;
	double	theta, phi,
		dtheta = DPIBY2 / (double) N,
		dphi = D2PI  / (double) N,
		da,
		fac;
	double	T;						// actual T on the circle
	double  d = Delta * AUm;		// Delta in m
	double	pha=alfa*DPI/180.;		// alpha in rad
	double	ps;

	if (N > 1000) N=1000;

	memset(f, 0, nw*sizeof(double));
	for(i=1; i<N; i++)				// for any elem. circ aureola (R=1)
	{								// find the angle of the elemenatry...
	  theta = dtheta*((double)i+.5);// colatitude
	  da = dphi*dtheta*sin(theta);
	  for(j=0; j<N; j++)
	  {
	    phi=dphi*((double)j+.5);
		if ((ps=cos(theta)*cos(pha)+
			 sin(theta)*sin(pha)*cos(phi))<=0) continue;
		T = Tmax * pow(cos(theta), .25);	// find its temperature
		if (T>=4.)
		{
		  for(k=0; k<nw; k++)
		    f[k]+= PLANCK_M_Jy(w[k]*1.e-6, T)* // add the planck flux *
		          ps*
		          da;
		}                       // extension of the surface
	  }
	}

	fac = R*R*1.e+6*		// emitting surface of the Object (1/2)
	epsilon /					// emissivity of the surface
	(DPI*d*d);					// distance from the earth PI -> because STERAD

	for(k=0; k<nw; k++)
	  f[k] *= fac;

	return 0;
}

// NEATM flux in W/m-2/um ///////////////////////////////////////////////////////////
int NEATMflux_W (double *w,	// wavelength. vector in um
		 double *f,	// flux vector
		 int	 nw,	// number of elements in w vector
		 double Tmax,	// Subsolar point temperature
		 double epsilon,// emissivity
		 double R,	// Radius of the asteroid in km
		 double Delta,	// distance from the earth
		 double alfa,	// phase angle
		 int	 N)	// number of iterations
{

	int	i,j,k;
	double	theta, phi,
		dtheta = DPIBY2 / (double) N,
		dphi = D2PI  / (double) N,
		da,
		fac;
	double	T;						// actual T on the circle
	double  d = Delta * AUm;		// Delta in m
	double	pha=alfa*DPI/180.;		// alpha in rad
	double	ps;

	if (N > 1000) N=1000;

	memset(f, 0, nw*sizeof(double));
	for(i=1; i<N; i++)				// for any elem. circ aureola (R=1)
	{								// find the angle of the elemenatry...
	  theta = dtheta*((double)i+.5);// colatitude
	  da = dphi*dtheta*sin(theta);
	  for(j=0; j<N; j++)
	  {
	    phi=dphi*((double)j+.5);
		if ((ps=cos(theta)*cos(pha)+
			 sin(theta)*sin(pha)*cos(phi))<=0) continue;
		T = Tmax * pow(cos(theta), .25);	// find its temperature
		if (T>50.)
		for(k=0; k<nw; k++)
		  f[k]+= PLANCK_M_W(w[k]*1.e-6, T)* // add the planck flux *
		          ps*
		          da;					// extension of the surface
	  }
	}

	fac = R*R*1.e+6*		// emitting surface of the Object (1/2)
	epsilon /					// emissivity of the surface
	(DPI*d*d);					// distance from the earth

	for(k=0; k<nw; k++)
	  f[k] *= fac;

	return 0;
}



///////////////////////////////////////////////
// fit the Near Earth Asteroid Thermal Model //
///////////////////////////////////////////////
//
// Marco Delbo
//
// revisions:
// 07 JUL 2002: Now a fit with fixed eta is possibile
// if the value of eeta is set lower than zero (negative)
// eta is not treted as fitting parameter.
// if eeta is positive eta is a fitted parameter
//
// July 2005
// Got annoyed by the LM fit, I decided to get rid of it and make use of a brute force
// - more robust method
//
int NEATMFit(double *w,		// wavelength. vector in um
	     double *f,		// flux vector
	     double *ef,		// error on flux vector
	     int    nw,		// number of elements in w vector
	     double *R,		// Radius of the asteroid in km (1.fit param)
	     double *eR,		// Error on Diameter
	     double *Pv,		// Geometric Albedo
	     double *ePv,		// error on Pv
	     double H,			// H value
	     double *eta,		// eta value
	     double *eeta,		// error on eta value
	     double epsilon,		// emissivity
	     double Delta,		// distance from the earth
	     double r,			// distance from the sun
	     double alfa,		// phase angle
	     double G,			// G value
	     double *pChisq,		// Chi squared value of the fit
	     int    N,			// number of iterations
	     int    mode)		// mode e.g Verbose
{
	double	pv=*Pv;				// use the user-provided guess albedo.
	double  A; 
    	double	Tss;
	double	*fr=NULL;
	double  beginchisq, bestchisq, chisq=0.0;
	double	Dsq, Dsqbest, DsqDen;
	int	i,j,k;
	double	num, den;
	int	NN=(int)((ETA_1-ETA_0)/ETA_S) + 1;
	double	*petaf=NULL, *pcs=NULL;
	double	a, b, x[3], y[3];
	
	if (mode & TM_MODE_VERBOSE) printf("Entering NEATMfit\n");

	if (N > 10000) {printf("More than 10000 wavelenghts??? \n qutting\n"); return -1;}
	if (nw<=0) {printf("No data?? quitting\n"); return -1; }
	if (pv<=0) pv=0.05;//reset pV to low albedo object 

	if ( (fr=(double*)malloc(sizeof(double)*nw)) == NULL) // alloc reference flux vector
	{
		printf("malloc error 1541 \n");
		return -1;
	}
	if ( (petaf=(double*)malloc(sizeof(double)*NN)) == NULL) // alloc reference flux vector
	{
		printf("malloc error 1542 \n");
		if (fr) free(fr);
		return -1;
	}
	if ( (pcs=(double*)malloc(sizeof(double)*NN)) == NULL) // alloc reference flux vector
	{
		printf("malloc error 1543 \n");
		if (petaf) free(petaf);
		if (fr) free(fr);
		return -1;
	}

	// fill in the eta vector
	for (j=0; j<NN; j++)
		petaf[j]=ETA_0+ETA_S*j;

	if (mode & TM_MODE_VERBOSE)
 	   printf("neatm init>%lf\t%lf\t%lf\t%lf\n", H, pv, A, *R*2);

	*R=H2DIAM(H, pv)/2.0;		// calculate the diameter from the H value and pv
	A=Pv2A(pv, G);			// derive the Bolometric Bond's Albedo

	for (j=0; j<NN; j++)
	  {
	    //		printf("%d\n",j);
	    Tss = TSS(A, petaf[j], r, epsilon); // derive subsolar temperature
	    NEATMflux_Jy(w, fr, nw, Tss, epsilon, *R, Delta, alfa, N);	// calc NEATM flux
	    if ((mode & TM_MODE_ADDREFLCOMP)) 
	      AddReflComp_Jy(w, fr, nw, H, G, r, Delta, alfa, 1.4); // it adds the relfected comp.
	
	    // compute the appropriate Dsq-value that minimize the X^2
	    num=0; den=0;
	    for (i=0; i<nw; i++)	
	      {
		num+=((f[i]*fr[i])/ef[i]/ef[i]);
		den+=((fr[i]*fr[i])/ef[i]/ef[i]);
	      }
	    Dsq=num/den;
	      
	    // compute the chisquare
	    chisq=0.0; for (i=0; i<nw; i++) chisq+=((f[i]-Dsq*fr[i])*(f[i]-Dsq*fr[i])/(ef[i]*ef[i]));
	    // store it
	    pcs[j]=chisq;
	      
		// readjust albedo
	    if (!(mode & TM_MODE_FIXALBEDO))
	      {
		*Pv=H2Pv(H,*R*2*sqrt(Dsq)); 
		if(*Pv>1.0) *Pv=1.0; // limit the albedo to 1
		if(*Pv<0.0) *Pv=0.01; // limit the albedo to 0.01 in case it falls below 0.0
		A = Pv2A(*Pv, G);
	      }

	    if (mode & TM_MODE_VERBOSE)
	      printf("neatm> %d\t%lf\t%lf\t%lf\t%lf\t%lf\n", j, petaf[j], pcs[j],*Pv, A,*R*2*sqrt(Dsq));
	      
	    if ((j>2) && ((pcs[j-2]-pcs[j-1])>0) && ((pcs[j]-pcs[j-1])>0))
	      { i=j-1; break;}	
	  }
	  //	  }

	  if (mode & TM_MODE_VERBOSE)
	    printf("neatm best>%d<\t%lf\t%lf\n",i, petaf[i], pcs[i]);

	  memcpy(&x, &petaf[i-1], sizeof(double)*3);
	  memcpy(&y, &pcs[i-1], sizeof(double)*3);
	  x[1]-=x[0]; x[2]-=x[0];
	  y[1]-=y[0]; y[2]-=y[0];
	  b=(y[1]*x[2]*x[2] - x[1]*x[1]*y[2])/(x[1]*x[2]*x[2] - x[2]*x[1]*x[1]);
	  a=(y[1] - b*x[1])/x[1]/x[1];
	  *eta=-b/2/a + x[0];

	  *R=*R*sqrt(Dsq);	// correct R
	  // readjust albedo
	  if (!(mode & TM_MODE_FIXALBEDO))
	    {
	      *Pv=H2Pv(H,*R*2);
	      A = Pv2A(*Pv, G);
	    }
	  if (mode & TM_MODE_VERBOSE)
	    printf("eta=%lf Diam=%lf pV=%lf  and now doing final adjustment\n",*eta,*R*2,*Pv);


	for (j=0; j<3; j++)
	{
		Tss = TSS(A, *eta, r, epsilon); // derive subsolar temperature
		NEATMflux_Jy(w, fr, nw, Tss, epsilon, *R, Delta, alfa, N);	// calc NEATM flux
		if ((mode & TM_MODE_ADDREFLCOMP)) 
		  AddReflComp_Jy(w, fr, nw, H, G, r, Delta, alfa, 1.4); // it adds the relfected comp.
	
		// compute the appropriate Dsq-value that minimize the X^2
        	num=0; den=0;
		for (i=0; i<nw; i++)	
		{
		 num+=((f[i]*fr[i])/ef[i]/ef[i]);
		 den+=((fr[i]*fr[i])/ef[i]/ef[i]);
		}

		Dsq=num/den;
		*R=*R*sqrt(Dsq);	// correct R
		
		if (!(mode & TM_MODE_FIXALBEDO))
	        {
			*Pv=H2Pv(H,*R*2);	// correct the albedo
			A= Pv2A(*Pv, G);	// derive the Bolometric Bond's Albedo
		}
		
		// compute the chisquare
		chisq=0.0; for (i=0; i<nw; i++) chisq+=((f[i]-Dsq*fr[i])*(f[i]-Dsq*fr[i])/(ef[i]*ef[i]));
		
		if (mode & TM_MODE_VERBOSE)
		  printf("%d>\t%lf\t%lf\t%lf\t%lf\n", j, *Pv, A,*R*2, chisq);
	}	

	// compute the chisquare
	chisq=0.0; for (i=0; i<nw; i++) chisq+=((f[i]-Dsq*fr[i])*(f[i]-Dsq*fr[i])/(ef[i]*ef[i]));
	*pChisq=chisq;

	free(pcs);
	free(fr);
	free(petaf);
}
// August 2005
// NEATM Fitted with fix value of eta now uses a more robust
// brute force fit
//
int NEATMFitFix(double *w,	// wavelength. vector in um
	double *f,		// flux vector
	double *ef,		// error on flux vector
	int    nw,		// number of elements in w vector
	double *R,		// Radius of the asteroid in km (1.fit param)
	double *eR,		// Error on Diameter
	double *Pv,		// Geometric Albedo
	double *ePv,		// error on Pv
	double H,		// H value
	double *eta,		// eta value
	double *eeta,		// error on eta value
	double epsilon,		// emissivity
	double Delta,		// distance from the earth
	double r,		// distance from the sun
	double alfa,		// phase angle
	double G,		// G value
	double *pChisq,		// Chi squared value of the fit
	int	 N,		// number of iterations
	int	mode)		// mode e.g Verbose
{
	double	pv=*Pv;		// use the user-provided guess albedo.
	double  A = Pv2A(pv, G);
    	double	Tss = TSS(A, *eta, r, epsilon);
	double	*fr;
	double  chisq=0.0;
	double	Dsq, Dsqbest;
	int	i,j;
	double	num, den;
	int	icorrpv=1;

	if (*ePv<0) icorrpv=0;			// keep albedo fixed
	if (N > 1000) N=1000;
	if (nw<=0) return -1;
	if (pv<=0) pv=0.2;
	*R = H2DIAM(H, pv)/2.0;		// calculate the diameter from the H value and pv

	if ( (fr=(double*)malloc(sizeof(double)*nw)) == NULL) // alloc reference flux vector
	{
		printf("malloc error 1541 \n");
		return -1;
	}

	NEATMflux_Jy(w, fr, nw, Tss, epsilon, *R, Delta, alfa, N);

	// compute the appropriate Dsq-value that minimize the X^2
        num=0; den=0;
	for (i=0; i<nw; i++)	
	{
	  num+=((f[i]*fr[i])/ef[i]/ef[i]);
	  den+=((fr[i]*fr[i])/ef[i]/ef[i]);
	}
	Dsq=num/den;
	*R=*R*sqrt(Dsq);	// correct the Radius

	j=0;
	while (j<=4)
	{
		if (icorrpv) pv=H2Pv(H,*R*2);
		A = Pv2A(pv, G);
		Tss = TSS(A, *eta, r, epsilon);
		NEATMflux_Jy(w, fr, nw, Tss, epsilon, *R, Delta, alfa, N);
	        num=0; den=0;
		for (i=0; i<nw; i++)	
		{
		  num+=((f[i]*fr[i])/ef[i]/ef[i]);
	  	  den+=((fr[i]*fr[i])/ef[i]/ef[i]);
		}
		Dsq=num/den;
		*R=*R*sqrt(Dsq);	// correct the Radius
		j++;
	}


	chisq=0.0;
// compute chisquare
	for (i=0; i<nw; i++) chisq+=((f[i]-Dsq*fr[i])*(f[i]-Dsq*fr[i])/(ef[i]*ef[i]));
    	*R=*R*sqrt(Dsq);
 	if (icorrpv)	
     	  *Pv=H2Pv(H,*R*2);
	*pChisq=chisq;

	free(fr);
}


///////////////////////////////////////////////
// fit the Fast Rotating Thermal Model //
///////////////////////////////////////////////
// July 2006
// FRM Fitted with fix value of eta now uses a more robust
// brute force fit
//
int   FRMFit(double *w,	// wavelength. vector in um
	double *f,	// flux vector
	double *ef,	// error on flux vector
	int    nw,	// number of elements in w vector
	double *R,	// Radius of the asteroid in km (1.fit param)
	double *eR,	// Error in Diameter
	double *Pv,	// Geometric Albedo
	double *ePv,	// error on Pv
	double H,		// H value
	double *eta,	// eta value
	double *eeta,	// error on eta value
	double epsilon,// emissivity
	double Delta,	// distance from the earth
	double r,		// distance from the sun
	double alfa,	// phase angle
	double G,		// G value
	double *pChisq,	 // Chi squared value of the fit
	int    N,			// number of iterations
	int    mode)		// mode e.g Verbose
{
	double	pv=*Pv;					// use the user-provided guess albedo.
	double  A = Pv2A(pv, G);
    	double	Tss = TSSFRM(A, r, *eta*epsilon);
	double	*fr;
	double  beginchisq, bestchisq, chisq=0.0;
	double	Dsq;
	int	i,j;
	double	num, den;
	int	icorrpv=1;

	if (*ePv<0) icorrpv=0;			// keep albedo fixed
	if (N > 1000) N=1000;
	if (nw<=0) return -1;
	if (pv<=0) pv=0.2;
	*R = H2DIAM(H, pv)/2.0;		// calculate the diameter from the H value and pv

	if ( (fr=(double*)malloc(sizeof(double)*nw)) == NULL) // alloc reference flux vector
	{
		printf("malloc error 1541 \n");
		return -1;
	}

	FRMflux_Jy(w, fr, nw, Tss, epsilon, *R, Delta, N);

	// compute the appropriate Dsq-value that minimize the X^2
        num=0; den=0;
	for (i=0; i<nw; i++)	
	{
	  num+=((f[i]*fr[i])/ef[i]/ef[i]);
	  den+=((fr[i]*fr[i])/ef[i]/ef[i]);
	}
	Dsq=num/den;
	*R=*R*sqrt(Dsq);	// correct the Radius


	j=0;
	while (j<=6)
	{
		if (icorrpv) pv=H2Pv(H,*R*2);
		A = Pv2A(pv, G);
		Tss = TSSFRM(A, r, *eta*epsilon);
		FRMflux_Jy(w, fr, nw, Tss, epsilon, *R, Delta, N);
		// compute the appropriate Dsq-value that minimize the X^2
        	num=0; den=0;
		for (i=0; i<nw; i++)	
		{
	  	  num+=((f[i]*fr[i])/ef[i]/ef[i]);
	  	  den+=((fr[i]*fr[i])/ef[i]/ef[i]);
		}
		Dsq=num/den;
		*R=*R*sqrt(Dsq);	// correct the Radius
		j++;
	}


	chisq=0.0;
// compute chisquare
	for (i=0; i<nw; i++) chisq+=((f[i]-Dsq*fr[i])*(f[i]-Dsq*fr[i])/(ef[i]*ef[i]));
    	*R=*R*sqrt(Dsq);
    	if (icorrpv) *Pv=H2Pv(H,*R*2);
	*pChisq=chisq;

	free(fr);
}

///////////////////////////////////////////////
// fit the Standard Thermal Model //
///////////////////////////////////////////////
// July 2006
// STM Fitted with fix value of eta now uses a more robust
// brute force fit
//
int   STMFit(double *w,	// wavelength. vector in um
	double *f,	// flux vector
	double *ef,	// error on flux vector
	int    nw,	// number of elements in w vector
	double *R,	// Radius of the asteroid in km (1.fit param)
	double *eR,	// Error in Diameter
	double *Pv,	// Geometric Albedo
	double *ePv,	// error on Pv
	double H,	// H value
	double *eta,	// eta value
	double *eeta,	// error on eta value
	double epsilon, // emissivity
	double Delta,	// distance from the earth
	double r,	// distance from the sun
	double alfa,	// phase angle
	double G,	// G value
	double *pChisq, // Chi squared value of the fit
	int	 N,	// number of iterations
	int	mode)	// mode e.g Verbose
{
	double	pv=*Pv;					// use the user-provided guess albedo.
	double  A;
        double	Tss;
	double	*fr;
	double	num, den;
	double  beginchisq, bestchisq, chisq=0.0;
	double	Dsq;
	int		i,j;
	int	icorrpv=1;

	if (*ePv<0) icorrpv=0;			// keep albedo fixed
	if (N > 1000) N=1000;
	if (nw<=0) return -1;
	if (pv<=0) pv=0.2;

	if ( (fr=(double*)malloc(sizeof(double)*nw)) == NULL) // alloc reference flux vector
	{
		printf("malloc error 1541 \n");
		return -1;
	}

	*R = H2DIAM(H, pv)/2.0;		// calculate the diameter from the H value and pv
	A = Pv2A(pv, G);		// calculate A
	Tss = TSS(A, *eta, r, epsilon);	// calculate Tss
	STMflux_Jy(w, fr, nw, Tss, epsilon, *R, 0.01, Delta, alfa, N);

	// compute the appropriate Dsq-value that minimize the X^2
        num=0; den=0;
	for (i=0; i<nw; i++)	
	{
	  num+=((f[i]*fr[i])/ef[i]/ef[i]);
	  den+=((fr[i]*fr[i])/ef[i]/ef[i]);
	}
	Dsq=num/den;
	*R=*R*sqrt(Dsq);	// correct the Radius

	j=0;
	while (j<=6)
	{
		if (icorrpv) pv=H2Pv(H,*R*2);	// correct the albedo
		A = Pv2A(pv, G);	// and the Bolometric bond albedo
		Tss = TSS(A, *eta, r, epsilon);
		STMflux_Jy(w, fr, nw, Tss, epsilon, *R, 0.01, Delta, alfa, N);
        	num=0; den=0;
		for (i=0; i<nw; i++)	
		{
	  	  num+=((f[i]*fr[i])/ef[i]/ef[i]);
	  	  den+=((fr[i]*fr[i])/ef[i]/ef[i]);
		}
		Dsq=num/den;
		*R=*R*sqrt(Dsq);	// correct the Radius
		j++;
	}


	chisq=0.0;
// compute chisquare
	for (i=0; i<nw; i++) chisq+=((f[i]-Dsq*fr[i])*(f[i]-Dsq*fr[i])/(ef[i]*ef[i]));
    	*R=*R*sqrt(Dsq);
    	if (icorrpv) *Pv=H2Pv(H,*R*2);
	*pChisq=chisq;

	free(fr);
}

int ReflectedLight_Jy(double *w, double *rlc, int Nw, double v, double vtowratio)
{
  double fluxAst36=SOLARFLUX36MANT*pow(10.0,SOLARFLUX36EXP-(v-SOLARVMAG)/2.5)*vtowratio;
  double BBfjy36 = PLANCK_M_Jy(3.6e-6,SOLARBBTEMPERATURE);
  double BBfjyw;
  int i;

  for (i=0; i<Nw; i++)
    rlc[i]+=(fluxAst36*PLANCK_M_Jy(w[i]*1.e-6,SOLARBBTEMPERATURE)/BBfjy36);
}


int ReflectedLight_W(double *w, double *rlc, int Nw, double v, double vtowratio)
{
  //  double fluxAst36=SOLARFLUX36MANT*pow(10.0,SOLARFLUX36EXP-(v-SOLARVMAG)/MAGBASE)*vtowratio;
  //  double BBfjy36 = PLANCK_M_Jy(3.6e-6,SOLARBBTEMPERATURE);
  double fluxAst335=SOLARFLUX335MANT*pow(10.0,SOLARFLUX335EXP-(v-SOLARVMAG)/2.5)*vtowratio;
  double BBfjy335 = PLANCK_M_Jy(3.35e-6,SOLARBBTEMPERATURE);

  double BBfjyw;
  int i;

  // printf("%lf %lf \n",fluxAst36,BBfjy36); // DEBUG
  for (i=0; i<Nw; i++)
    //    rlc[i]+=(fluxAst36*PLANCK_M_W(w[i]*1.e-6,SOLARBBTEMPERATURE)/BBfjy36);
    rlc[i]+=(fluxAst335*PLANCK_M_W(w[i]*1.e-6,SOLARBBTEMPERATURE)/BBfjy335);
    
}


double h2v(double H, double G, double r, double Delta, double pha)
{
  double tanalpha = tan(0.5*pha/RADEG);
  double phi1 = exp( -HG_A1* (pow(tanalpha,HG_B1)));
  double phi2 = exp( -HG_A2* (pow(tanalpha,HG_B2)));
  double Halpha = H - MAGBASE*log10( (1.0-G)*phi1 + G*phi2 );
  Halpha += 5.0*log10(r*Delta);
  //  printf("###H=%lf G=%lf r=%lf Delta=%lf pha=%lf v=%lf\n",H,G,r,Delta,pha,Halpha);

  return Halpha;
}

int  AddReflComp_Jy(double *w, double *fr, int nw, double H, double G,
		 double r, double Delta, double alfa, double vtowratio) // it adds the relfected comp.
{
  double v=h2v(H, G, r, Delta, alfa);
   ReflectedLight_Jy(w, fr, nw, v, vtowratio);
}

int  AddReflComp_W(double *w, double *fr, int nw, double H, double G,
		 double r, double Delta, double alfa, double vtowratio) // it adds the relfected comp.
{
  double v=h2v(H, G, r, Delta, alfa);
   ReflectedLight_W(w, fr, nw, v, vtowratio);
}
