#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
//#include <sys\stat.h>       /* S_ constant definitions */
//#include <fcntl.h>          /* O_ constant definitions */
//#include <io.h>
#include <float.h>
#include "craters.h"

#ifndef CONSTANTS_H
  #include "../tms/constants.h"
#endif

#ifndef MACROS_H
  #include "../tms/macros.h"
#endif

#ifndef MESH_H
  #include "mesh.h"
#endif

#ifndef max
        #define max( a, b ) ( ((a) > (b)) ? (a) : (b) )
#endif

#ifndef min
        #define min( a, b ) ( ((a) < (b)) ? (a) : (b) )
#endif

double  *sumcos=NULL, *sumT4=NULL;

int freecrater(CRATER *pcrat)
{
	int	i;

	if (pcrat == NULL)					// if not a valid pointer
		return -1;

	if (pcrat->pni!=NULL) free(pcrat->pni);	
	if (pcrat->ptheta!=NULL) free(pcrat->ptheta);	
	if (pcrat->ptheta0!=NULL) free(pcrat->ptheta0);
	if (pcrat->pT!=NULL) free(pcrat->pT);	
	if (pcrat->pS!=NULL) free(pcrat->pS);	
	if (pcrat->pV!=NULL) free(pcrat->pV);	
	if (pcrat->n!=NULL)  free(pcrat->n);	

	if (pcrat->pTB != NULL)
	{
		for (i=0; i<pcrat->Nt; i++)
			free(pcrat->pTB[i]);
		free(pcrat->pTB);
		pcrat->pTB=NULL;
	}

	if (pcrat->pTB_dt != NULL)
	{
		free(pcrat->pTB_dt);
		pcrat->pTB_dt=NULL;
	}

	if (sumcos != NULL)
	{
		free(sumcos); sumcos=NULL;
	}

	if (sumT4 != NULL)
	{
		free(sumT4); sumT4=NULL;
	}

	return 1;	// success
}

int initcrater(CRATER *pcrat,			// point to crater struct
	       int Nrings,				// number of rings
	       double gamma,			// crater opening angle in radians
	       double rc, 				// radius of the sphical section
	       double A,				// albedo of the tiles
	       double epsilon,			// surface emissivity
	       double Gamma,			// thermal inertia
	       double density,			// Crater density (1.0=100%)
	       VECTOR vZenith,
	       VECTOR vSouth)
{

	double	coswk1, coswk;
	double	ni, mu0;
	double  tanthetabar=0.0;
	int		k, j, i;
	VECTOR	vtmp;

	if (pcrat == NULL)	// if not a valid pointer
		return -1;						// exit
			// try to alloc crater
//	if ((pcrat=(CRATER*)malloc(sizeof(CRATER)))==NULL)
//		return -1;						// failed to alloc crater
	
	pcrat->A=A;							// stores the albedo of the tiles
	pcrat->gamma=gamma;					// and the crater opening angle
	pcrat->rc=rc;						// and the crater radius
	pcrat->epsilon=epsilon;				// emissivity 
	pcrat->density=density;				// crater density
	pcrat->projArea=DPI*rc*rc*sin(gamma)*sin(gamma);
	pcrat->Gamma=Gamma;					// stores thermal inertia

	// CALCULATE LAGERROS MODEL PARAMETERS (Thermal physics of asteroids. IV paper)
	pcrat->S=(1-cos(pcrat->gamma))/2;

	// apply the correction to the Bond Albedo
	pcrat->A *=(1.0-sin(pcrat->gamma/2)*sin(pcrat->gamma/2))
	  /(1.0-A*sin(pcrat->gamma/2)*sin(pcrat->gamma/2));

	// calculate the number of tiles...
	// there are 4i tiles in the i-th ring
	pcrat->Nt = 2*Nrings*(Nrings+1);
	pcrat->Nr = Nrings;

	// allocate vectors for crater tiles
	if ((pcrat->pni=(double*)malloc(pcrat->Nt*sizeof(double)))==NULL)
	{free(pcrat);return -1;}			// failed to alloc 
	if ((pcrat->ptheta=(double*)malloc(pcrat->Nt*sizeof(double)))==NULL)
	{free(pcrat);return -1;}			// failed to alloc 
	if ((pcrat->ptheta0=(double*)malloc(pcrat->Nt*sizeof(double)))==NULL)
	{free(pcrat);return -1;}			// failed to alloc 
	if ((pcrat->pT=(double*)malloc(pcrat->Nt*sizeof(double)))==NULL)
	{free(pcrat);return -1;}			// failed to alloc 
	if ((pcrat->pS=(int*)malloc(pcrat->Nt*sizeof(int)))==NULL)
	{free(pcrat);return -1;}			// failed to alloc 
	if ((pcrat->pV=(int*)malloc(pcrat->Nt*sizeof(int)))==NULL)
	{free(pcrat);return -1;}			// failed to alloc 
	if ((pcrat->n=(VECTOR*)malloc(pcrat->Nt*sizeof(VECTOR)))==NULL)
	{free(pcrat);return -1;}			// failed to alloc 

	pcrat->tileArea=D2PI*rc*rc*(1-cos(gamma))/(float)pcrat->Nt;

	// calculates colatitude of each tile
	coswk1=cos(gamma);
	for (k=(Nrings-1); k>=0; k--)
	{
	  coswk=(2.0+(float)k*coswk1)/(float)(k+2);
	  ni=0.5*(acos(coswk)+acos(coswk1));

//		printf("\n%d %f..", k, ni/RAD);
	  mu0	= DPIBY2/(double)(k+1);
		// in the k-th ring there are 4k tiles with the same colatitide
		// set their colatitude
	  for (j=2*k*(k+1); j<2*(k+1)*(k+2); j++)
	    {
	      i = j - 2*k*(k+1);
	      //			printf("%d ",j);
	      pcrat->pni[j]=ni;	// this is tile latitude
	      pcrat->n[j].x=-cos(mu0*(double)i)*sin(ni);
	      pcrat->n[j].y=-sin(mu0*(double)i)*sin(ni);
	      pcrat->n[j].z=cos(ni);
	    }
	  coswk1=coswk;
	}

	// calculation of Hapke's thetabar
	pcrat->thetabar=0.0;
	for (k=0; k<pcrat->Nr; k++)			// for every ring
	{ // element latitude is constant within the ring
	  pcrat->thetabar += (double)(k+1) * tan(pcrat->pni[2*k*(k+1)]);	
	}
    
	pcrat->thetabar*= (pcrat->density*
			   8.0/DPI/pcrat->Nt);
	pcrat->thetabar=atan(pcrat->thetabar)/RAD;

	// create the crater reference frame
	normalize(vZenith, &pcrat->vZenith);
	normalize(vSouth,  &pcrat->vSouth);
	vtmp.x=VECx(pcrat->vZenith, pcrat->vSouth);	 // define the East vector in the asteroid reference frame 
	vtmp.y=VECy(pcrat->vZenith, pcrat->vSouth); 
	vtmp.z=VECz(pcrat->vZenith, pcrat->vSouth);
	normalize(vtmp, &pcrat->vEast);

	memset(pcrat->pT, 0, sizeof(double)*pcrat->Nt);

	pcrat->pTB=NULL;
	pcrat->pTB_dt=NULL;

	if (pcrat->Gamma > 0)
	{
	  pcrat->pTB = (double**)malloc(sizeof(double*)*pcrat->Nt);
	  if (!pcrat->pTB)
	    return -1;

	  for (i=0; i<pcrat->Nt; i++) 
	    {
	      pcrat->pTB[i]=(double*)malloc(sizeof(double)*NTSLABS);
	      if (!pcrat->pTB[i]) i = pcrat->Nt+10;
			memset(pcrat->pTB[i], 0, sizeof(double)*NTSLABS);
	    }

	  pcrat->pTB_dt=(double*)malloc(sizeof(double)*NTSLABS);
	  if (!pcrat->pTB_dt) i = pcrat->Nt + 10;
	  memset(pcrat->pTB_dt, 0, sizeof(double)*NTSLABS);

	  if (i > pcrat->Nt)
	    {
	      freecrater(pcrat);
	      return -1;
	    } 
	}
	
	// alloc global buffers
	if (!sumcos) sumcos=(double*)malloc(pcrat->Nt*sizeof(double));
	if (!sumT4) sumT4=(double*)malloc(pcrat->Nt*sizeof(double));

			
	if ((sumcos==NULL) || (sumT4==NULL))
	  {
	    freecrater(pcrat);
	    return -1;
	  }

	return 0; 
}

int setCratTempBuff(CRATER	*pcrat,
		    double	temp)
{
	int	i,j;

	if ((pcrat->pTB==NULL) || (pcrat->pTB_dt==NULL))
		return -1;			// set failed, you need to allocate temperature buffers first

	for (i=0; i<pcrat->Nt; i++)
	{
		pcrat->pT[i]=temp;
		for (j=0; j<NTSLABS; j++)
			pcrat->pTB[i][j]=temp;
		pcrat->pTB_dt[i]=temp;
	}

	return 1;
}

// since the structure crater is used several times, i.e. for each tile
// the normal to the tile is not a property of the crater 
//
int setshadow(CRATER	*pcrat,			// point to crater
			  VECTOR	vSun,			// sun vector
			  double	***LUT)
{
/*
	if (pcrat == NULL)					// if the crater is not valid
		return -1;						// the crater is not a valid pointer

	if (pcrat->pS == NULL)
		return -1;						// not a valid pointer to shadow buffer


	memset(pcrat->pS, 0, sizeof(int)*pcrat->Nt); // set all elements to 0 - i.e. no shadow

	if ( (costheta=SCALAR(vSun, pcrat->vZenith))<=0)	// cos of the solar zenit angle with respect to the crater normal
		return -2;
    cosSouth=SCALAR(vSun,pcrat->vSouth);
    cosEast =SCALAR(vSun,pcrat->vEast);
	
	theta=acos(costheta)/RAD;
	mu=atan2(cosEast, cosSouth)/RAD;
	if (mu <0) mu+=180;

*/
	return 1;
}

// since the structure crater is used several times, i.e. for each tile
// the normal to the tile is not a property of the crater 
//
int setvisibility(CRATER   *pcrat,			// point to crater
		  VECTOR   norm,			// normal of the surface where the crater lies
		  VECTOR   vsun,			// sun vector
		  VECTOR    vearth)			// Earth vector	
{
	VECTOR	vEarthNormal,
			vtmp,
			vEast,
			vSouth;
	double  y1, y2,						// y1 and y2 distances of Emery et al.
	  cZenitSun,
	  gammaprime, 
	  costheta, theta, 
	  cosEast, cosSouth,
	  muE, mu, sinmu, tau, z, zprime,
	  rhoprime, cosgamma;			// aux variables
	int  k,j;

	if (pcrat == NULL)					// if the crater is not valid
		return -1;						// the crater is not a valid pointer

	if (pcrat->pV == NULL)
		return -1;						// not a valid pointer to VISIBILITY buffer

	pcrat->Delta=normalize(vearth, &vEarthNormal);		// normalize Earth vector

	vtmp.x=VECx(norm, vsun);			// define the East vector in the asteroid reference frame 
	vtmp.y=VECy(norm, vsun); 
	vtmp.z=VECz(norm, vsun);
	normalize(vtmp, &vEast);

	cZenitSun=SCALAR(vsun, norm);		// component of the sun vector along the normal
	vtmp.x=vsun.x-norm.x*cZenitSun;     // define the South vector in the asteroid reference frame 
	vtmp.y=vsun.y-norm.y*cZenitSun;
	vtmp.z=vsun.z-norm.z*cZenitSun;
	normalize(vtmp, &vSouth);


	memset(pcrat->pV, 0, sizeof(int)*pcrat->Nt); // set all elements to 1.0 - i.e. no Earth shadow

	costheta=SCALAR(vEarthNormal, norm);	// cos of the earth zenit angle with respect to the crater normal
	cosEast=SCALAR(vEarthNormal, vEast);	// cos of the earth zenit angle with respect to the crater normal
	cosSouth=SCALAR(vEarthNormal, vSouth);	// cos of the earth zenit angle with respect to the crater normal
	
	pcrat->vEarthC.x=cosSouth;				// saves back the Earth vector in the crater reference frame
	pcrat->vEarthC.y=cosEast;				// ... it will be used to compute the flux towards the earth ...
	pcrat->vEarthC.z=costheta;

	if (costheta <=0)						// Earth is below crater horizon
	{
		for (j=0; j<pcrat->Nt; j++)			// put all crater tiles into shadow
			pcrat->pV[j]=1;
		return 2;							// all crater elements are in shadow
	}

	theta=acos(costheta);					// Earth zenit angle with respect to the crater normal
	muE=atan2(cosEast, cosSouth);
		
//    if (theta <= (DPIBY2 - pcrat->gamma))		// all elements are illuminated
//		return 0;			

	z=pcrat->rc * sin(pcrat->gamma);
	cosgamma=cos(pcrat->gamma);

	for (k=0; k<pcrat->Nr; k++)			// for every ring
	{
		tau	 = pcrat->rc*sin(pcrat->pni[2*k*(k+1)]);	// element latitude is constant within the ring
		for (j=0; j<4*(k+1); j++)			// j is the index of the crater element
		{
			mu			= DPIBY2/(double)(k+1)*(double)j + muE;
			sinmu		= sin(mu);
			zprime		= sqrt(z*z-tau*tau*sinmu*sinmu);
			rhoprime	= sqrt(pcrat->rc*pcrat->rc * cosgamma*cosgamma + zprime*zprime);
			gammaprime	= asin(zprime / rhoprime);

			y2			= rhoprime*sin(DPI-2*theta-gammaprime);
			y1			= tau*cos(mu);
			if (y1 >= y2) pcrat->pV[j+2*k*(k+1)]=1;	// element is in shadow	-> set it to 1
		} // for j (tile index)
	}	// for k (ring index)
	return 1;
}

int resetcratertemp(CRATER *pcrat, //pointer to the crater 
		    double T)      //temperature at which the crater has to be set
{
  int i,j;

  for(i=0; i<pcrat->Nt; i++)
    {
      pcrat->pT[i]=T;
    }

  if ((pcrat->pTB) && (pcrat->Gamma>0))// if the heat diffusion buffer is allocated and Gamma>0
    {
      for(i=0; i<pcrat->Nt; i++)
	  for(j=0; j<NTSLABS; j++)
	    pcrat->pTB[i][j]=T;

    }
}

int setcratertemp(CRATER	*pcrat,			// point to crater
		  VECTOR	vSun,			// sun vector (MUST BE NORMALIZED)
		  double        r,			// heliocentric distance
		  double	***LUT)			// direction cosine x shadows
{
	double	costheta, 
	  cosSouth,
	  cosEast,
	  Tss4,						// subsolar equilibrium temperature to the 4th
	  Eref0,
	  E0thermal;
	int		k,j,i,l,iter;
	int		theta, mu;
	double ccc;

	if (pcrat == NULL)					// if the crater is not valid
		return -1;						// the crater is not a valid pointer
									

	memset(pcrat->pT, 0, sizeof(double)*pcrat->Nt); // set all temperature to 0

	costheta=SCALAR(vSun,pcrat->vZenith);	// cos of the solar zenit angle with respect to the crater normal
        cosSouth=SCALAR(vSun,pcrat->vSouth);
        cosEast =SCALAR(vSun,pcrat->vEast);
	
	theta=(int)(acos(costheta)/RAD);
	if (theta >=90) return -2;
	mu   =(int)(atan2(cosEast, cosSouth)/RAD);
	if (mu <0) mu+=360;

	Tss4=SOLARC/r/r*(1.0-pcrat->A)/SIGMA/pcrat->epsilon;	// epsilon

	Eref0=Tss4*(1.0-cos(pcrat->gamma))/2/pcrat->Nt*
	  pcrat->A*pcrat->A/(1-pcrat->A*pcrat->gamma/DPI);

	//victor
	// Eref0=Tss4*(1.0-cos(pcrat->gamma))/2/pcrat->Nt*
	//  pcrat->A/(1-pcrat->A*pcrat->gamma/DPI); 
	
	E0thermal=0.5*(1.0-cos(pcrat->gamma))/pcrat->Nt;
	
    
	for (i=0; i<pcrat->Nt; i++)
	{
		  sumcos[i]=0.0;
		  for (l=0; l<i; l++)
//			  sumcos[i]+=pcrat->n[l].x*cosSouth + pcrat->n[l].y*cosEast + pcrat->n[l].z*costheta;
			  sumcos[i]+=LUT[theta][mu][l]; 
		  for (l=i+1; l<pcrat->Nt; l++)
//			  sumcos[i]+=pcrat->n[l].x*cosSouth + pcrat->n[l].y*cosEast + pcrat->n[l].z*costheta;
			  sumcos[i]+=LUT[theta][mu][l]; 
	}

	
	for (k=0; k<pcrat->Nr; k++)			// for every ring
	  {
	    for (j=0; j<4*(k+1); j++)			// j is the index of the crater element
	      {
		i=j+2*k*(k+1);
		pcrat->pT[i]=pow(Tss4*				// subsolar equilibrium temperature to the 4th 
				 LUT[theta][mu][i]	// shadows * cosine of the solar zenith angle
				 +Eref0*sumcos[i],	// Eref/(eps sigma)
				 0.25);
	      } // for j (tile index)
	  }	// for k (ring index)
	
	for (iter=0; iter<10; iter++)
	  {

	    for (i=0; i<pcrat->Nt; i++)
	      {
		sumT4[i]=0.0;
		for (l=0; l<i; l++)
		  sumT4[i]+=(pcrat->pT[l]*pcrat->pT[l]*pcrat->pT[l]*pcrat->pT[l]);
		for (l=i+1; l<pcrat->Nt; l++)
		  sumT4[i]+=(pcrat->pT[l]*pcrat->pT[l]*pcrat->pT[l]*pcrat->pT[l]);		
	      }

	    for (k=0; k<pcrat->Nr; k++)			// for every ring
	      {
		for (j=0; j<4*(k+1); j++)			// j is the index of the crater element
		  {
		    i=j+2*k*(k+1);
		    if (LUT[theta][mu][i]==0)
		      ccc=0;
		    else
		      ccc=(pcrat->n[i].x*cosSouth + pcrat->n[i].y*cosEast + pcrat->n[i].z*costheta);
		    pcrat->pT[i]=pow(Tss4*		// subsolar equilibrium temperature to the 4th
				     ccc
				     +
				     Eref0*sumcos[i]		// Eref/(eps sigma)
				     +
				     E0thermal*sumT4[i]
				     ,0.25);
		  } // for j (tile index)
	      }	// for k (ring index)
	  } // for iter

	return 1;
}

///////////////////////////////////////////////////////////////////////////////////////
///////// SET THE TEMPS inside the Carters Using the Lagerros method
////////// Lagerros  Thermal physics of asteroids. IV eq. (18)
////////// MDB - Prague 1/11/11 //////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////
int SetCraterTempLagerros(CRATER *pcrat,	// point to crater
			  VECTOR vSun,		// sun vector (MUST BE NORMALIZED)
			  double r,		// heliocentric distance
			  double ***LUT)	// direction cosine x shadows
{
  double	costheta, 
    cosSouth,
    cosEast,
    Tss4,						// subsolar equilibrium temperature to the 4th
    E0;
  int  k,j,i,l,iter;
  int  theta, mu;
  
  if (pcrat == NULL)					// if the crater is not valid
    return -1;					// the crater is not a valid pointer
  
  memset(pcrat->pT, 0, sizeof(double)*pcrat->Nt); // set all temperature to 0
  
  costheta=SCALAR(vSun,pcrat->vZenith);	// cos of the solar zenit angle with respect to the crater normal
  cosSouth=SCALAR(vSun,pcrat->vSouth);
  cosEast =SCALAR(vSun,pcrat->vEast);
	
  theta=(int)(acos(costheta)/RAD);
  if (theta >=90) return -2;
  mu   =(int)(atan2(cosEast, cosSouth)/RAD);
  if (mu <0) mu+=360;

  Tss4=SOLARC/r/r*(1.0-pcrat->A)/SIGMA/pcrat->epsilon;	// epsilon
  E0=((1-pcrat->S)*pcrat->A+pcrat->epsilon)/
     (1-pcrat->S*pcrat->A)*Tss4*pcrat->S*costheta;
    
  for (k=0; k<pcrat->Nr; k++)			// for every ring
    {
      for (j=0; j<4*(k+1); j++)			// j is the index of the crater element
	{
	  i=j+2*k*(k+1);
	  pcrat->pT[i]=Tss4*LUT[theta][mu][i]	// shadows * cosine of the solar zenith angle
	    +E0;
	  pcrat->pT[i]=pow(pcrat->pT[i],0.25);
	} // for j (tile index)
    }	// for k (ring index)
  return 1;
}



// determines crater tempartures using the heat diffusion
// 
int setcratertempHD(CRATER	*pcrat,			// point to crater
			        VECTOR	vSun,			// sun vector (MUST BE NORMALIZED)
				    double    r,			// heliocentric distance
				    double	***LUT,			// direction cosine x shadows
					double	dx,
					double	dt,	
					double	Tsid
			        )
{
	double	costheta, 
		    cosSouth,
			cosEast,
			inFlux,						// subsolar equilibrium temperature to the 4th
			Eref0,
			E0thermal;
	int		i,ix,l;
	int		theta, mu;
	double	cons1 = dt* D2PI/(Tsid) /dx/dx;
	double	cons2 = 2.*dt*sqrt(D2PI/Tsid)/pcrat->Gamma/dx;
	double	inputenergy;

	if (pcrat == NULL)					// if the crater is not valid
		return -1;						// the crater is not a valid pointer
									

	costheta=SCALAR(vSun, pcrat->vZenith);	// cos of the solar zenit angle with respect to the crater normal
    cosSouth=SCALAR(vSun,pcrat->vSouth);
    cosEast =SCALAR(vSun,pcrat->vEast);
	
	theta=(int)(acos(costheta)/RAD);
	mu   =(int)(atan2(cosEast, cosSouth)/RAD);
	if (mu <0) mu+=360;

	inFlux=SOLARC/r/r*(1.0-pcrat->A)/pcrat->epsilon;			// Tss^4*sigma*epsilon

	Eref0=SOLARC/pcrat->epsilon/r/r*(1.0-cos(pcrat->gamma))/2/pcrat->Nt*
		  pcrat->A*(1.0-pcrat->A)/
		  (1-pcrat->A*pcrat->gamma/DPI);						// Eref / Sum(cos(theta(i)))
	
	E0thermal=0.5*(1.0-cos(pcrat->gamma))/pcrat->Nt				
//		         * pcrat->epsilon*SIGMA;						// E0thermal * sigma * epsilon
		         * SIGMA;						// MDB20071216 epsilon=1 inside craters // E0thermal * sigma * epsilon
	    
	if (theta<90)
		for (i=0; i<pcrat->Nt; i++)
		{
		  sumcos[i]=0.0;
		  for (l=0; l<i; l++)
			  sumcos[i]*=LUT[theta][mu][l]; 
		  for (l=i; l<pcrat->Nt; l++)
			  sumcos[i]*=LUT[theta][mu][l]; 
		}

    
	for (i=0; i<pcrat->Nt; i++)
	{
		  sumT4[i]=0.0;
		  for (l=0; l<i; l++)
			  sumT4[i]+=(pcrat->pT[l]*pcrat->pT[l]*pcrat->pT[l]*pcrat->pT[l]);
		  for (l=i; l<pcrat->Nt; l++)
			  sumT4[i]+=(pcrat->pT[l]*pcrat->pT[l]*pcrat->pT[l]*pcrat->pT[l]);

	}

	for (i=0; i<pcrat->Nt; i++)
	{
		if (theta < 90)
			inputenergy = inFlux	* LUT[theta][mu][i] 
				    + Eref0		* sumcos[i]
					+ E0thermal * sumT4[i];
		else
			inputenergy =  E0thermal * sumT4[i];

		pcrat->pTB_dt[0] = pcrat->pTB[i][0] + 
						2.* cons1 * ( pcrat->pTB[i][1]-pcrat->pTB[i][0] ) 
//						- cons2 * ( pcrat->epsilon * SIGMA 
						- cons2 * ( SIGMA						// MDB20071216 epsilon=1 inside craters 
						* pcrat->pTB[i][0]*pcrat->pTB[i][0]*pcrat->pTB[i][0]*pcrat->pTB[i][0]
						- inputenergy);

		for (ix=1; ix<(NTSLABS-1); ix++)
			pcrat->pTB_dt[ix] = pcrat->pTB[i][ix] + cons1* (pcrat->pTB[i][ix+1] - 2*pcrat->pTB[i][ix] + pcrat->pTB[i][ix-1]);

		pcrat->pTB_dt[ix] = pcrat->pTB_dt[ix-1] = pcrat->pTB[i][ix-2];

		for(ix=0; ix<NTSLABS; ix++)
	      pcrat->pTB[i][ix] = pcrat->pTB_dt[ix];

		pcrat->pT[i]=pcrat->pTB[i][0];
	} // for i (tile index)
}

int getcraterflux(CRATER*   pcrat,			// point to crater structure
		  VECTOR    vEarth,			// Earth vector
		  double*   plambda,		// pointer to wavelengths in um
		  double*   pflux,			// pointer to flux vector in mJy
		  short	    nlambda,		// number of wavelenghts
		  double    ***LUT,			// look up table
		  int       *pfiltind,
		  ColorCorrection *pCCorr)
{
  double	B;
  int		i, l;
  double        cosSouth, 
                cosEast,
                costheta;
  int		theta, mu;
  double        ccfactor;

  if (nlambda > 1000) return -1;

  for (l=0; l<nlambda; l++)
    pflux[l]=0.0;
  

  costheta=SCALAR(vEarth, pcrat->vZenith);
  cosSouth=SCALAR(vEarth,pcrat->vSouth);
  cosEast =SCALAR(vEarth,pcrat->vEast);
  
  theta=(int)(acos(costheta)/RAD);
  if (theta >=90) return -2;
  mu   =(int)(atan2(cosEast, cosSouth)/RAD);
  if (mu <0) mu+=360;
  
  for (l=0;l<nlambda; l++)
    {
      for (i=0;i<pcrat->Nt;i++)
	{ 
	  if (LUT[theta][mu][i]<=0) continue;
	  
	  B=pow(plambda[l],-5)*NIST_C1/(exp(NIST_C2/(plambda[l]*pcrat->pT[i]))-1)/PI; //in steradians (diveded by pi)

	  if ((pfiltind[l] >= 0) && (pCCorr))
	    {
	      
	      if((int)pcrat->pT[i] > 9999) printf("Temperature too high!\n");
	      ccfactor = pCCorr->CCTable[pfiltind[l]][(int)pcrat->pT[i]];
	      //VictorTBD: we should set ccfactor = 1 instead of calling the pointer anyway
	      
	      pflux[l]+= (B*	                           // planck function
			  //	       pcrat->epsilon*	  // MDB20071216 epsilon=1 inside craters (surface emissivity)
			  LUT[theta][mu][i]*		// direction cosine including shadow term
			  ccfactor);             // color correction factor
	    }
	  else
	    {
	      pflux[l]+= (B*	                           // planck function
			  //	       pcrat->epsilon*	  // MDB20071216 epsilon=1 inside craters (surface emissivity)
			  LUT[theta][mu][i]);		// direction cosine including shadow term
	    }
	}
      
      pflux[l] *= pcrat->tileArea;						// tile AREA
      pflux[l] *= pcrat->epsilon;
    }
  return 1; 
}

///////////////////////////////////////////////////////////////////////////////////////
///////// Get the Flux from the carter using the Lagerros Equation
////////// Lagerros  Thermal physics of asteroids. IV eq. (19)
////////// MDB - Prague 1/11/11 //////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////
///////// COLOR CORRECTION TERM added on Jan 30, 2013
/////////////////////////////////////////////////////////////////////////////////////
int GetCraterFluxLagerros(CRATER*   pcrat,			// point to crater structure
			  VECTOR    vEarth,			// Earth vector
			  double*   plambda,		// pointer to wavelengths in um
			  double*   pflux,			// pointer to flux vector in mJy
			  short	    nlambda,		// number of wavelenghts
			  double    ***LUT,			// look up table
			  int       *pfiltind,         // filter index (for the color correction)
			  ColorCorrection *pCCorr      // color correction structure (pointer)
			  )
{
  double	B, E0;
  int		i, l;
  double  cosSouth, 
    cosEast,
    costheta;
  int		theta, mu;
  double        ccfactor;

  //			DeltaQAU	= NORMAQ(vearth)*AU*AU;

  if (nlambda > 1000) return -1;

  memset(pflux, 0, sizeof(double)*nlambda);
  //  for (l=0; l<nlambda; l++)  pflux[l]=0.0;
  

  costheta=SCALAR(vEarth, pcrat->vZenith);
  cosSouth=SCALAR(vEarth,pcrat->vSouth);
  cosEast =SCALAR(vEarth,pcrat->vEast);
  
  theta=(int)(acos(costheta)/RAD);
  if (theta >=90) return -2;
  mu   =(int)(atan2(cosEast, cosSouth)/RAD);
  if (mu <0) mu+=360;
  
  E0=(1-pcrat->epsilon)*pcrat->S*(1-pcrat->S)*costheta;

  for (l=0;l<nlambda; l++)
    {
      for (i=0;i<pcrat->Nt;i++)
	{ 
	  B=pow(plambda[l],-5)*NIST_C1/
	    (exp(NIST_C2/(plambda[l]*pcrat->pT[i]))-1)/PI; //in steradians (divided by pi)

	  if ((pfiltind[l] >= 0) && (pCCorr))
	    {
	      
	      if((int)pcrat->pT[i] > 999) printf("Temperature too high!\n");
	      ccfactor = pCCorr->CCTable[pfiltind[l]][(int)pcrat->pT[i]];
	      
	      pflux[l]+=(B*			       // planck function
			 //			 pcrat->epsilon*	               //MDB20071216 epsilon=1 inside craters     Victor20130319
			 LUT[theta][mu][i]             	// direction cosine including shadow term
			 + B*E0)*                     // part 2 of Lagerros paper IV Eq. (19)
		ccfactor;		             // color correction term		
	    }
	  else
	    {
	      pflux[l]+= B*			       // planck function
		//pcrat->epsilon*	               // MDB20071216 epsilon=1 inside craters Victor20130319: it shouldn't be here if it is below
		LUT[theta][mu][i];					// direction cosine including shadow term
	      // part 2 of Lagerros paper IV Eq. (19)
	      pflux[l]+= B*E0;
	    }
	}
      pflux[l] *= pcrat->tileArea;						// tile AREA
      pflux[l] *= pcrat->epsilon;
    }
  return 1; 
}


// this function allocate and fill the Shadow/visibility look up table 
// for a crater of given opening angle and number of tiles
// in the unlucky case the function returns -1 (alloc failure)
// you are better to call cratShadowLUTdelete() which attempt to free allocated buffer
// 
// init MDB 05 may, 2004, La Silla (Chile)
int cratShadowLUTcreate(CRATER *pcrat, double ****pLUT)
{
	int		i,j;
	double	***LUT;

	if (*pLUT != NULL) cratShadowLUTdelete(*pLUT);

	// allocate the look-up-table
	if ((LUT=(double ***)malloc(91*sizeof(double**)))==NULL)
		return -1;

	for (i=0; i<=90; i++)
	{
	  if ((LUT[i]=(double **)malloc(360*sizeof(double*)))==NULL)
	    return -1;
	  for (j=0; j<360; j++)
	    if ((LUT[i][j]=(double *)malloc(pcrat->Nt*sizeof(double)))==NULL)
	      return -1;
	}

	// pass back the vectors to the calling routine
	*pLUT=LUT;

	for (i=0; i<=90; i++)
	  for (j=0; j<360; j++)
	    cratShadowCalc(pcrat, (double)i*RAD, (double)j*RAD, LUT[i][j]);
}

// this function frees the LUT created by using cratShadowLUTcreate()
// init MDB 05 may, 2004, La Silla (Chile)
// MDB Oct 10, 2005: fixed a major bug, which caused memory freeing problems and crashed the BAMlib
// if repeated calls to IDLputCraters() were issued
int cratShadowLUTdelete(double ***LUT)
{
	int i,j;

	if (LUT == NULL) return -1;		// already deallocated

	for (i=0; i<=90; i++)
	{
	  if (LUT[i])
	    {
	      for (j=0; j<360; j++)
		{
		  if (LUT[i][j])
		    free(LUT[i][j]);
		}
	      free(LUT[i]);
	    }
	}

	free(LUT);

	return 1;
}

// this function clalculate the scalar product of a given vector v with
// each crater tile 
// the scalar product is set to zero if a tile is not illuminated by vector v
// or it is in shadow
int cratShadowCalc(CRATER *pcrat, 
		   double theta,		// in radians
		   double azim,		// in radians
		   double *result)
{
	double		cosSouth, 
	  cosEast, 
	  costheta,
	  mu, sinmu,
	  rhoprime,
	  gammaprime,
	  z, zprime,
	  cosgamma,
	  y1,y2,
	  tau;
	int	i,j,k;

	
	cosSouth=cos(azim)*sin(theta);
	cosEast	=sin(azim)*sin(theta);
	costheta=cos(theta);

	z=pcrat->rc * sin(pcrat->gamma);
	cosgamma=cos(pcrat->gamma);

	for (k=0; k<pcrat->Nr; k++)			// for every ring
	{
	  tau	 = pcrat->rc*sin(pcrat->pni[2*k*(k+1)]);	// element latitude is constant within the ring
	  for (j=0; j<4*(k+1); j++)			// j is the index of the crater element
	    {
	      i=j+2*k*(k+1);	// make tile index
	      
	      // if the scalar product with the sun/earth vector is less then zero
	      if ((result[i]= pcrat->n[i].x*cosSouth + 
		   pcrat->n[i].y*cosEast +
		   pcrat->n[i].z*costheta)<=0) 
		{
		  result[i]=0;	// set it to zero
		  continue;		// and do not check for shadow
		}

	      mu         = DPIBY2/(double)(k+1)*(double)j - azim;	// it has to be -azim (LH576 Santiago ch -> frankfurt, Business class may 10, 2004)
	      sinmu	 = sin(mu);
	      zprime	 = sqrt(z*z-tau*tau*sinmu*sinmu);
	      rhoprime	 = sqrt(pcrat->rc*pcrat->rc * cosgamma*cosgamma + zprime*zprime);
	      gammaprime = asin(zprime / rhoprime);

	      y2			= rhoprime*sin(DPI-2*theta-gammaprime);
	      y1			= tau*cos(mu);
	      if (y1 >= y2) result[i]=0;	// element is in shadow	-> set it to 0
	    } // for j (tile index)
	}	// for k (ring index)

}

 
int dumpcrater(CRATER	*pcrat, 
	       char*	filename,
	       VECTOR	vSun,		// normalized sun vector
	       VECTOR   vEarth,
	       double	***LUT)			

{
	FILE	*fh;
	int		k,j;
	double	mu;  
	int	    thetaE, muE,
			muS, thetaS;
	double	costheta,
			cosSouth,
			cosEast;


	fh=fopen(filename, "wt");
	if (fh == NULL) return -1;

	costheta=SCALAR(vSun, pcrat->vZenith);		// cos of the solar zenit angle with respect to the crater normal
	cosSouth=SCALAR(vSun, pcrat->vSouth);
	cosEast =SCALAR(vSun, pcrat->vEast);
	
	thetaS=(int)(acos(costheta)/RAD);
	muS=(int)(atan2(cosEast, cosSouth)/RAD);
	if (muS <0) muS+=360;

	costheta=SCALAR(vEarth, pcrat->vZenith);		// cos of the solar zenit angle with respect to the crater normal
	cosSouth=SCALAR(vEarth, pcrat->vSouth);
	cosEast =SCALAR(vEarth, pcrat->vEast);
	
	thetaE=(int)(acos(costheta)/RAD);
	muE=(int)(atan2(cosEast, cosSouth)/RAD);
	if (muE <0) muE+=360;

	fprintf(fh, "%d %d %f %f %f\n", pcrat->Nr, pcrat->Nt, pcrat->rc, pcrat->gamma, costheta);
	
	for (k=0; k<pcrat->Nr; k++)			// for every ring
	{
//		fprintf(fh,"\n");
		for (j=0; j<4*(k+1); j++)			// j is the index of the crater element
		{
			mu = DPIBY2/(k+1)*j;
			fprintf(fh, "%02d %02d %03d %7.1f %7.1f %7.3f %7.3f %7.1f\n", k, j, j+2*k*(k+1), 
				mu/RAD, pcrat->pni[j+2*k*(k+1)]/RAD, 
				LUT[thetaS][muS][j+2*k*(k+1)],
				LUT[thetaE][muE][j+2*k*(k+1)], 
				pcrat->pT[j+2*k*(k+1)]);
		}
	}
	fclose(fh);
}

int dupmLUT(CRATER*	pcrat,
			char*	filename,
			double ***LUT
			)
{
	FILE	*fh;
	int		i,j,k;

	fh=fopen(filename, "wt");
	if (fh == NULL) return -1;

	for (i=0; i<=90; i++)
      for (j=0; j<360; j++)
	  {
        for (k=0; k<pcrat->Nt; k++)
			fprintf(fh, "%f ", LUT[i][j][k]);
	    fprintf(fh, "\n"); 
	  }

	fclose(fh);	
}

int CraterCalcTempreatureStatistics(CRATER *pcrat,
				    double *pTaverage,
				    double *pTstddev,
				    double *pTmax,
				    double *pTmin)      
{
  int iTileIndex;
  pTaverage[0]=pTstddev[0]=pTmax[0]=pTmin[0]=0.0;
  pTmin[0]=1.e6;

  for(iTileIndex=0; iTileIndex<pcrat->Nt; iTileIndex++)
    {
      pTaverage[0]+=pcrat->pT[iTileIndex];
      pTmax[0]=max(pcrat->pT[iTileIndex], pTmax[0]);
      pTmin[0]=min(pcrat->pT[iTileIndex], pTmin[0]);
    }
  pTaverage[0]/=pcrat->Nt;
  
  for(iTileIndex=0; iTileIndex<pcrat->Nt; iTileIndex++)
      pTstddev[0]+=(pTaverage[0]-pcrat->pT[iTileIndex])*(pTaverage[0]-pcrat->pT[iTileIndex]);

  pTstddev[0]/=pcrat->Nt;
  pTstddev[0]=sqrt(pTstddev[0]);
}
