#define __CRATERS_H

#ifndef MESH_H
  #include "mesh.h"
#endif

//Marco, 2013-01-30
#ifndef COLORCORRECTION_H
  #include "colorcorrection.h"
#endif

#define  NTSLABS			32	// Number of slabs in the temperature buffer

typedef struct tagCRATER	// Crater structure
{
  double  A;			// albedo
  double  epsilon;		// emissivity
  double  gamma;		// opening angle
  int	  Nr;			// number of rings
  int	  Nt;			// number of tiles
  double  rc;			// radius of the crater
  double  tileArea;		// Area of the tiles (tiles have same area)
  double  *pni;			// pointer to tile colatitude	
  double  *ptheta;		// pointer to solar zenith angles with respect to each tile
  double  *ptheta0;		// pointer to Earth zenith angles with respect to each tile
  double  *pT;			// pointer to tile temperature
  int	  *pS;			// pointer to Shadowing term
  int	  *pV;			// pointer to Viewing term (equivalent to shadowing for the Earth)
  VECTOR  *n;			// normal to the element in the crater reference frame
  VECTOR  vEarthC;		// normalized vector to the Earth in the crater reference frame
  double  Delta;			// Geocentric distance in AU
  double  density;		// Crater density (1.0=100%)
  double  projArea;		// Crater zenital projected area = PI*rc^2*[cos(gamma)]^2
  double  thetabar;		// The Hapke's (1984) mean surface slope
  // orthonormal crater reference frame. Vectors are defined in the shape reference frame
  VECTOR  vZenith,		// zenith vector
    vSouth,			// south vector
    vEast;			// east vector	
  // heat diffusion stuff
  double  Gamma;			// Thermal inertia
  double  **pTB;			// temperature buffer
  double  *pTB_dt;		// temperature buffer at new time 
  // Lagerros Thermal physics of asteroids. IV paper
  double  S;        // depth to radius ratio = (1-cos(\gamma))

} CRATER;

// GLOBAL VARIABLEs for craters
extern double  *sumcos, *sumT4;

// functions prototypes
int freecrater(CRATER *pcrat);

int initcrater(CRATER *pcrat,			// point to crater struct
	       int Nrings,			// number of rings
	       double gamma,			// crater opening angle in radians
	       double rc, 			// radius of the sphical section
	       double A,			// albedo of the tiles
	       double epsilon,			// surface emissivity
	       double Gamma,			// crater thermal inertia
	       double density,			// Crater density (1.0=100%)
	       VECTOR vzenith,
	       VECTOR vsouth);

int setshadow(CRATER	*pcrat,			// point to crater
	      VECTOR	vSun,			// sun vector
	      double	***LUT);

int setvisibility(CRATER	*pcrat,			// point to crater
		  VECTOR	norm,			// normal of the surface where the crater lies
		  VECTOR	vsun,			// sun vector
		  VECTOR    vearth			// Earth vector
		  );	

int resetcratertemp(CRATER *pcrat, //pointer to the crater 
		    double T);     //temperature at which the crater has to be set

int setcratertemp(CRATER	*pcrat,			// point to crater
		  VECTOR	vSun,			// sun vector (MUST BE NORMALIZED)
		  double    r,				// heliocentric distance
		  double	***LUT			// direction cosine x shadows
		  );

int SetCraterTempLagerros(CRATER *pcrat,		// point to crater
		  VECTOR	vSun,			// sun vector (MUST BE NORMALIZED)
		  double    r,				// heliocentric distance
		  double	***LUT			// direction cosine x shadows
		  );

int multiplycratertemp(CRATER	*pcrat,			// point to crater
		       double	 scale);		// scaling factor

int setcratertempHD(CRATER	*pcrat,			// point to crater
		    VECTOR	vSun,			// sun vector (MUST BE NORMALIZED)
		    double    r,			// heliocentric distance
		    double	***LUT,			// direction cosine x shadows
		    double	dx,
		    double	dt,	
		    double	Tsid
		    );

int setCratTempBuff(CRATER	*pcrat,
		    double	temp);

int getcraterflux(CRATER*	pcrat,			// point to crater structure
		  VECTOR    vEarth,			// Earth vector
		  double*	plambda,		// pointer to wavelengths in um
		  double*	pflux,			// pointer to flux vector in mJy
		  short		nlambda,		// number of wavelenghts
		  double	***LUT,			// look up table
		  int           *pfiltind,              // filter index for the color correction
		  ColorCorrection *pCCorr		// ColorCorrection Structure  
		  );

int GetCraterFluxLagerros(CRATER* pcrat,	        // point to crater structure
		  VECTOR    vEarth,			// Earth vector
		  double*	plambda,		// pointer to wavelengths in um
		  double*	pflux,			// pointer to flux vector in mJy
		  short		nlambda,		// number of wavelenghts
		  double	***LUT,			// look up table
		  int           *pfiltind,              // filter index for the color correction
		  ColorCorrection *pCCorr		// ColorCorrection Structure  
		  );

int dumpcrater(CRATER	*pcrat, 
	       char*	filename,
	       VECTOR	vSun,		// normalized sun vector
	       VECTOR   vEarth,
	       double	***LUT);			


int dupmLUT(CRATER*	pcrat,
	    char*	filename,
	    double ***LUT
	    );

int cratShadowLUTcreate(CRATER *pcrat, 
			double ****LUT);

int cratShadowLUTdelete(double ***LUT);

int cratShadowCalc(CRATER *pcrat, 
		   double theta,		// in radians
		   double azim,		// in radians
		   double *result);

int CraterCalcTempreatureStatistics(CRATER *pcrat,	
				    double *pTaverage,	
				    double *pTstddev,	
				    double *pTmax,	
				    double *pTmin);      
