#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifndef LC_H
  #include "lc.h"
#endif

#ifndef TPM_H
  #include "tpm.h"
#endif

#ifndef MESH_H
  #include "mesh.h"
#endif

#include "../tms/constants.h"
#include "../tms/tm.h"
#include "../tms/macros.h"


#define SWAP(a,b) {swap=(a);(a)=(b);(b)=swap;}

void gaussj(double **a, int n, double **b, int m)
{
	int *indxc,*indxr,*ipiv;
	int i,icol,irow,j,k,l,ll;
	double big,dum,pivinv,swap;

	indxc=(int*)malloc(sizeof(int)*n);
	indxr=(int*)malloc(sizeof(int)*n);
	ipiv=(int*)malloc(sizeof(int)*n);

	for (j=0;j<n;j++) ipiv[j]=0;
	for (i=0;i<n;i++)
	  {
	    big=0.0;
	    for (j=0;j<n;j++)
	      if (ipiv[j] != 1)
		for (k=0;k<n;k++) {
		  if (ipiv[k] == 0) {
		    if (fabs(a[j][k]) >= big) {
		      big=fabs(a[j][k]);
		      irow=j;
		      icol=k;
		    }
		  }
		}
	    ++(ipiv[icol]);
	    if (irow != icol) {
	      for (l=0;l<n;l++) SWAP(a[irow][l],a[icol][l])
				  for (l=0;l<m;l++) SWAP(b[irow][l],b[icol][l])
						      }
	    indxr[i]=irow;
	    indxc[i]=icol;
	    if (a[icol][icol] == 0.0) {
	      printf("gaussj: Singular Matrix\n"); //exit(-10);
	    }
	    pivinv=1.0/a[icol][icol];
	    a[icol][icol]=1.0;
	    for (l=0;l<n;l++) a[icol][l] *= pivinv;
	    for (l=0;l<m;l++) b[icol][l] *= pivinv;
	    for (ll=0;ll<n;ll++)
	      if (ll != icol) {
		dum=a[ll][icol];
		a[ll][icol]=0.0;
		for (l=0;l<n;l++) a[ll][l] -= a[icol][l]*dum;
		for (l=0;l<m;l++) b[ll][l] -= b[icol][l]*dum;
	      }
	  }
	for (l=(n-1);l>=0;l--) {
	  if (indxr[l] != indxc[l])
	    for (k=0;k<n;k++)
	      SWAP(a[k][indxr[l]],a[k][indxc[l]]);
	}
	free(ipiv);
	free(indxr);
	free(indxc);
}


void covsrt(double **covar, int ma, int ia[], int mfit)
{
  int i,j,k;
  double swap;

  for (i=mfit;i<ma;i++)
    for (j=0;j<i+1;j++) covar[i][j]=covar[j][i]=0.0;
  k=mfit-1;
  for (j=ma-1;j>=0;j--) {
    if (ia[j]) {
      for (i=0;i<ma;i++) SWAP(covar[i][k],covar[i][j]);
      for (i=0;i<ma;i++) SWAP(covar[k][i],covar[j][i]);
      k--;
    }
  }
}
#undef SWAP


void mrqcof(TPM* ptpm, EPH *peph, int Neph, OBS *pobs, int Nobs, double *a, int *ia,
	int ma, double **alpha, double *beta, double *chisq)
{
  int i,j,k,l,m,mfit=0, iObs, iData;
  double ymod,wt,sig2i,dy;
  double yda0[50][10];
  double yda1[50][10];

  for (j=0;j<ma;j++)
    if (ia[j]) mfit++;

  for (j=0;j<ma;j++) 
    {
      for (k=0;k<ma;k++) alpha[j][k]=0.0; 
      beta[j]=0.0;
    }

  *chisq=0.0;
  
  // stores back the parameters in the TPM structure from the a vector
  ptpm->D2               = a[0];    // Diameter Squared 
  ptpm->BeamingParameter = a[1];    // Beaming parameter  
  ptpm->A                = 1-a[2];  // Bond Albedo
  ptpm->Gamma            = a[3];    // Thermal inertia
  ptpm->Rp               = a[4];    // Ratio of pIR/pV  
			    
  printf("mrqcof> input parameters: "); 
  for(i=0;i<ma;i++)
    if(ia[i])
      printf("a[%d]=%lf ", i,a[i]);
  printf("\n");
  
  // we call the model here and fill the ModelFlux, ReflectedFlux and the dyda matrix
  // dyda is a vector in NR but it must be a matrix here
  // call the model which fills obs and the dyda thing
  TPM_noTI_noCrat_Flux(ptpm, peph, Neph, pobs, Nobs); // run the TPM with calc of the derivatives
  
  for (iObs=0; iObs<Nobs; iObs++) // for each observation
    {
      for (iData=0; iData<pobs[iObs].Ndata; iData++) // loop on data (more data per obs)
	{
	  sig2i=1.0/(pobs[iObs].eflux[iData]*pobs[iObs].eflux[iData]); // inverse of the square of the error
	  // calculate Observed - Model flux
	  dy=pobs[iObs].flux[iData]-pobs[iObs].ModelFlux[iData]-pobs[iObs].ReflectedFlux[iData]; // O-C
	  
	  *chisq += dy*dy*sig2i;  // add to the chi squared
	  
	  printf("lm> %9.7lf %9.4lf %+9.7lf %9.7lf %9.7lf %2d %2d %9.7lf\n",
		 pobs[iObs].JD, pobs[iObs].lambda[iData],dy,
		 pobs[iObs].flux[iData],
		 pobs[iObs].ModelFlux[iData],//+pobs[iObs].ReflectedFlux[iData],
		 iData, iObs, pobs[iObs].ReflectedFlux[iData]);
	  
	  //calculate coefficients:
	  for (j=0,l=0;l<ma;l++) {
	    if (ia[l]) {
	      //	      printf("mrqcof> calculating coeffs. for a[%d]\n",l);
	      wt=pobs[iObs].dyda[iData][l]*sig2i;
	      //printf("dyda[%d][%d][%d] = %lf\n",iObs,iData,l,pobs[iObs].dyda[iData][l]);
	      for (k=0,m=0;m<=l;m++)
		if (ia[m]) alpha[j][k++] += wt*pobs[iObs].dyda[iData][m];
	      beta[j++] += dy*wt;		
	    }
	  } 
	  
	}// loop over data
    }// loop over obs
  
  // it is a symmetric matrix 
  for (j=1;j<mfit;j++)
    for (k=0;k<j;k++) alpha[k][j]=alpha[j][k]; //we had k<j-1
  
  printf("mrqcof> ");
  for (l=0;l<ma;l++)
    printf("beta[%d] = %e ",l,beta[l]);
  printf("\n");
  
  printf("mrqcof> chi2 = %e\n",*chisq);
}


void mrqmin(TPM* ptpm, EPH *peph, int Neph, OBS *pobs, int Nobs, double a[], int ia[],
	int ma, double **covar, double **alpha, double *chisq, double *alamda)
{
  void covsrt(double **covar, int ma, int ia[], int mfit);
  void gaussj(double **a, int n, double **b, int m);
  void mrqcof(TPM* ptpm, EPH *peph, int Neph, OBS *pobs, int Nobs, double *a, int *ia,
	      int ma, double **alpha, double *beta, double *chisq);
  
  int i,j,k,l,cond=1;
  static int mfit;
  static double ochisq,*atry,*beta,*da,*acheck,**oneda;
  
  if (*alamda < 0.0) 
    {
      atry  =(double*)malloc(sizeof(double)*ma);
      beta  =(double*)malloc(sizeof(double)*ma);
      da    =(double*)malloc(sizeof(double)*ma);
      acheck=(double*)malloc(sizeof(double)*ma);

      for (mfit=0,j=0;j<ma;j++)
	if (ia[j]) mfit++;
      
      // allocate the oneda matrix (the deltas of the coeff.) 
      oneda=(double**)malloc(sizeof(double)*mfit);
      for (i=0; i<mfit; i++)
	oneda[i]=(double*)malloc(sizeof(double));
      
      *alamda=0.001;
      mrqcof(ptpm, peph, Neph, pobs, Nobs, a,ia,ma,alpha,beta,chisq); 
      ochisq=(*chisq);
      printf("mrqmin> initial chi2 = %e\n",ochisq);

      for (j=0;j<ma;j++)
	{ 
	  atry[j]  =a[j];
	  acheck[j]=a[j];
	}      
    }

  printf("1mrqmin>  ");
  for (j=0;j<mfit;j++) {
    for (k=0;k<mfit;k++) covar[j][k]=alpha[j][k];
    covar[j][j]=alpha[j][j]*(1.0+(*alamda));
    oneda[j][0]=beta[j];
    printf("beta[%d]: %e ",j,beta[j]);
  }
  
  printf("\n2mrqmin> before gaussj ");
  for(j=0;j<mfit;j++)
    printf("oneda[%d]: %e  ", j,oneda[j][0]);
  printf("\n");
  
  gaussj(covar,mfit,oneda,1);
  
  printf("3mrqmin>  after gaussj ");
  for(j=0;j<mfit;j++)
    printf("oneda[%d]: %e ", j,oneda[j][0]);
  printf("\n");
  
  for (j=0;j<mfit;j++)
    da[j] = oneda[j][0];
  

  if (*alamda == 0.0) 
    {

      printf("\n\nmrqmin> lambda = 0; calculating uncertainties\n");
      mrqcof(ptpm, peph, Neph, pobs, Nobs, a,ia,ma,alpha,beta,chisq);      
      ochisq=(*chisq);
      
      for (j=0;j<mfit;j++) {
	for (k=0;k<mfit;k++) covar[j][k]=alpha[j][k];
	oneda[j][0]=beta[j];
      }

      gaussj(covar,mfit,oneda,1);
  
      covsrt(covar,ma,ia,mfit);
      covsrt(alpha,ma,ia,mfit);
      
      // deallocate the oneda matrix 
      for (i=0; i<mfit; i++)
	free(oneda[i]);
      free(oneda);
      free(da);
      free(beta);
      free(atry);
      return;
    }

  for (j=0,l=0;l<ma;l++)
    if (ia[l]) atry[l]=a[l]+da[j++];
  
  mrqcof(ptpm, peph, Neph, pobs, Nobs, atry,ia,ma,alpha,beta,chisq);
  
  if (*chisq < ochisq)
    {
      printf("mrqmin> chi2 = %e (decreased) ochisq = %e\n",*chisq,ochisq);
      *alamda *= 0.1;
      ochisq=(*chisq);

      /*  // We do not need this because we calculate 
	  // the covariance matrix at the end
	for (j=0;j<mfit;j++) {	
	for (k=0;k<mfit;k++) alpha[j][k]=covar[j][k];
	beta[j]=da[j];
	}
      */

      for (l=0;l<ma;l++) a[l]=atry[l]; 
    } 
  else
    {
      printf("mrqmin> chi2 = %e (did not decrease) ochisq = %e\n",*chisq,ochisq);
      *alamda *= 10.0;
      *chisq=ochisq;

    }

}

int FindFlag(char *flags, char *lett)
{
  //Check whether there is the parameter flag in the LM
  // -F entry in order to let the parameter free or fix it
  
  char *buff=NULL;
  int resu;

  buff=strstr(flags,lett);
  if(buff){
    //printf("there is at least one %s in %s\n",lett,flags);
    resu = 1; //we fit it
  }
  else{
    //printf("there is no %s in %s\n",lett,flags);
    resu = 0; //we fix it
  }
  return resu;
}

int TPM_LMFit(TPM* ptpm, EPH *peph, int Neph, OBS *pobs, int Nobs, char* pLMfl)
{

  int i,*ia,iter,itst,j,k,mfit=NPARAMS;
  double alamda,chisq,ochisq,**covar,**alpha;
  double a[NPARAMS];
  char *D2_flag="D",*BP_flag="B",*BA_flag="A",
    *TI_flag="G",*RP_flag="R";
  
  ia=(int*)malloc(sizeof(int)*NPARAMS); //ivector(1,MA);
  memset(ia, 0, sizeof(int)*NPARAMS);
  
  // alloc the covar matrix // it was: covar=matrix(1,MA,1,MA);
  covar=(double**)malloc(sizeof(double*)*NPARAMS);
  for (i=0; i<NPARAMS; i++)
    covar[i]=(double*)malloc(sizeof(double)*NPARAMS);
  
  // alloc the alpha matrix // it was: alpha=matrix(1,MA,1,MA);
  alpha =(double**)malloc(sizeof(double*)*NPARAMS);
  for (i=0; i<NPARAMS; i++)
    alpha[i]=(double*)malloc(sizeof(double)*NPARAMS);

  // guess values of the parameters
  a[0]=ptpm->D2;               // Scale Squared
  a[1]=ptpm->BeamingParameter; // Beaming parameter
  a[2]=1-ptpm->A;              // Reflectivity: 1-A
  a[3]=ptpm->Gamma;            // Thermal Inertia
  a[4]=ptpm->Rp;               // Albedo ratio pIR/pV 
  
  // search for the corresponding flag in the 
  // command line. If it is found, we fit. If
  // it is not, the parameter is fixed at the 
  // guess value
  ia[0] = FindFlag(pLMfl,D2_flag);
  ia[1] = FindFlag(pLMfl,BP_flag);
  ia[2] = FindFlag(pLMfl,BA_flag);
  ia[3] = FindFlag(pLMfl,TI_flag);
  ia[4] = FindFlag(pLMfl,RP_flag);
  
  printf("TPM_LMFit. Fitting the following parameters:\n");
  for(i=0; i<NPARAMS; i++)
    if(ia[i]) printf("a[%d] ",i);
  printf("\n");
  
  for (iter=1;iter<=2;iter++) {
    alamda = -1;
    mrqmin(ptpm, peph, Neph, pobs, Nobs,a,ia,NPARAMS,covar,alpha,&chisq,&alamda);
    k=0;
    itst=0;
    for (;;) {
      printf("\n%s %2d %17s %10.4f %10s %9.2e\n","Iteration #",k,
	     "chi-squared:",chisq,"alamda:",alamda);
      printf("%8s %8s %8s %8s %8s\n",
	     "D2","eta","1-A","Gamma","Rp");
      printf("%9.4f  ",sqrt(a[0]));
      for (i=1;i<NPARAMS;i++) printf("%9.4f  ",a[i]);
      printf("\n");
      k++;
      ochisq=chisq;
      mrqmin(ptpm, peph, Neph, pobs, Nobs,a,ia,NPARAMS,covar,alpha,&chisq,&alamda);
      if (chisq > ochisq)
	itst=0;
      else if (fabs(ochisq-chisq) < 0.001)
	itst++;
      if (itst < 3) continue;
      
      alamda=0.0;
      mrqmin(ptpm, peph, Neph, pobs, Nobs,a,ia,NPARAMS,covar,alpha,&chisq,&alamda);
      printf("\n%s %2d %17s %10.4f %10s %9.2e\n","Final#",k,
	     "chi-squared:",chisq,"alamda:",alamda);
      printf("%9s %9s %9s %9s %9s\n",
	     "D2","eta","A","Gamma","Rp");
      printf("%9.4f  ",sqrt(a[0]));
      for (i=1;i<NPARAMS;i++) printf("%9.4f  ",a[i]);
      printf("\n");
      printf("\nUncertainties:\n");
      for (i=0;i<NPARAMS;i++)  printf("%9.4f ",sqrt(covar[i][i]));
      printf("\n\n");
      break;
    }
  }
  
  printf("lmr> %9.4f %9.4f  ",sqrt(a[0]), sqrt(covar[0][0])/2.0);
  for (i=1;i<NPARAMS;i++) 
    if(i==2) 
      printf("%9.4f %9.4f  ",(1-a[i]),sqrt(covar[i][i]));
    else
      printf("%9.4f %9.4f  ",a[i],sqrt(covar[i][i]));
  printf("%9.4f\n",chisq);
  
  for(i=0; i<NPARAMS; i++)
    {      
      free(alpha[i]);
      free(covar[i]);
    }
  
  free(alpha);
  free(covar);
  free(ia);
  return 0;
}


// DIRTY DERIVATIVES ---------------------------------------
// d a[0]
/*
  ptpm->D2=a[1];//-0.1; // Diameter Squared 
  ptpm->BeamingParameter=a[0]-0.05; // Beaming parameter
  printf("In mrQCOF a[0]=%lf a[1]=%lf D2=%lf eta=%lf\n", a[0], a[1], ptpm->D2, ptpm->BeamingParameter);
  
  TPM_noTI_noCrat_Flux(ptpm, peph, Neph, pobs, Nobs); // run the TPM with calc of the derivatives
  for (iObs=0; iObs<Nobs; iObs++) // for each observation
    for (iData=0; iData<pobs[iObs].Ndata; iData++) // loop on data (more data per obs)
      yda0[iObs][iData]=pobs[iObs].ModelFlux[iData];
  
  
  // d a[1]
  ptpm->D2=a[0]; // Diameter Squared 
  ptpm->BeamingParameter=a[1]-0.01; // Beaming parameter
  printf("In mrQCOF a[0]=%lf a[1]=%lf D2=%lf eta=%lf\n", a[0], a[1], ptpm->D2, ptpm->BeamingParameter);
  TPM_noTI_noCrat_Flux(ptpm, peph, Neph, pobs, Nobs); // run the TPM with calc of the derivatives
  for (iObs=0; iObs<Nobs; iObs++) // for each observation
    for (iData=0; iData<pobs[iObs].Ndata; iData++) // loop on data (more data per obs)
      yda1[iObs][iData]=pobs[iObs].ModelFlux[iData];
  
  */

  /*
  for (iObs=0; iObs<Nobs; iObs++) // for each observation
    for (iData=0; iData<pobs[iObs].Ndata; iData++) // loop on data (more data per obs)
      {
	pobs[iObs].dyda[iData][0]=(pobs[iObs].ModelFlux[iData]-yda0[iObs][iData])/0.05;
	//pobs[iObs].dyda[iData][1]=(pobs[iObs].ModelFlux[iData]-yda1[iObs][iData])/0.05;
	}
  */


/*
  //check for unphysical negative values: if there are,
  //make a small change with the sign of the proposed 
  //unphysical da. If alambda is not zero we do not perform
  //this resetting, because we will not need to further 
  //calculate fluxes
  for (j=0,l=0;l<ma;l++)
    if ((ia[l]) && *alamda != 0.0)
      {
	acheck[l]=a[l]+da[j++];
	if (acheck[l]<0.05)
	  {
	    j--;
	    da[j]   = da[j]/fabs(da[j])*0.00001;
	    printf("Reset da[%d] to almost zero\n",j++);
	  }	
      }
  
  
  //check for unphysically large values of eta or Rp
  //we skip parameter a[0] because it will usually be
  //the diameter, which we allow to grow as large as 
  //necessary
  for (j=1,l=1;l<ma;l++)
    if ((ia[l]) && *alamda != 0.0)
      {
	acheck[l]=a[l]+da[j++];
	if (acheck[l]>50)
	  {
	    j--;
	    da[j]   = 50.0 - a[j];
	    printf("Reset da[%d] to almost 50\n",j++);
	  }	
      }
  */
