#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "obs.h"

/* ReadObs is now compatible with the format of
J Durech's convex inversion
and
B. Carry's KOALA codes

Observations input file:
Number of observations
JD, number of datapoints for this observation
SunX, SunY, SunZ (in the frame of the asteroid)
EartX, EarthY, EarthZ (in the frame of the asteroid)
lambda1, flux1, sigma_flux1
.... 
....
lamndaN, fluxN, sigma_fluxN

...
...
JD, number of datapoints for this observation
SunX, SunY, SunZ (in the frame of the asteroid)
EartX, EarthY, EarthZ (in the frame of the asteroid)
lambda1, flux1, sigma_flux1
.... 
....
lamndaN, fluxN, sigma_fluxN


Example of input file:

2
2445604.5091097867    3
-1.2562543990637          -1.6116273891164         0.5606997267613
-0.2558626315738          -1.5490425784241         0.5606971355123
   12.0   4.096   0.363
   25.0   5.294   1.000   2
   60.0   2.514   0.647


2445604.5806755028    3
-1.2556816410465          -1.6123080063047         0.5607617666951
-0.2553870846930          -1.5484993502869         0.5607591909714
   12.0   4.547   0.403      
   25.0   6.197   1.172
   60.0   2.617   0.676
*/

int ReadObs(char *fname, OBS **lpobs, long *pnobs)
{
  FILE* 	file_in;
  char 	linebuf[256];
  long 	nobs=0, iobs, idata;
  OBS	*obs;
  int	nfields;

  if ((file_in = fopen( fname, "rt"))== NULL ) 
    return -1;
  
  while(!feof(file_in))
    {
      if (NULL==fgets(linebuf,256,file_in)) break;
      if (linebuf[0]=='#') continue; // this is a comment
      if (sscanf(linebuf,"%ld",&nobs)<1) continue; // read the number of observations
      printf("ReadObs: Number of Observations to load: %d\n", nobs);
      break;
    }

  obs=malloc(nobs*sizeof(OBS));
  if (obs==NULL)
    {
      printf("E ReadObs 01 Memory not allocated\n");
      return -2; //failed
    }
  
  // pass back the pointers to the allocated OBS
  *lpobs=obs;
  *pnobs=nobs;
	
  for (iobs=0; iobs<nobs; iobs++) 
    {// for each observations
      // get the epoch and the number of datapoints
      //      fscanf(file_in,"%lf %d",&obs[iobs].JD,&obs[iobs].Ndata); 
      
      // get the Observation DATA: JD, Number of Obs, FLAGS 
      while(!feof(file_in))
	{
	  if (NULL==fgets(linebuf,256,file_in)) break;
	  if (linebuf[0]=='#') continue; // this is a comment
	  if ((sscanf(linebuf,"%le %d %d",&obs[iobs].JD,
		      &obs[iobs].Ndata,&obs[iobs].flags)<2)) continue; 
	  printf("ReadObs: Observation <%d> JD=%le Ndata=%d FLAGS=%d\n",
		 iobs, obs[iobs].JD,obs[iobs].Ndata,obs[iobs].flags); 
	  break;
	}


      // do some sanity checks
      if ((obs[iobs].Ndata<=0)||(obs[iobs].Ndata>10000))
	{
	  printf("E ReadObs 02 Too many or too few data points\n");
	  return -3; //failed
	}
   
      // get the Sun vector
      while(!feof(file_in))
	{
	  if (NULL==fgets(linebuf,256,file_in)) break;
	  if (linebuf[0]=='#') continue; // this is a comment
	  if (sscanf(linebuf,"%lf %lf %lf",&obs[iobs].vs.x,
		      &obs[iobs].vs.y, &obs[iobs].vs.z)<3) continue;       
	  printf("ReadObs: SUN=(%lf,%lf,%lf)\n",obs[iobs].vs.x,obs[iobs].vs.y,obs[iobs].vs.z);       
	  break;
	}

      // get the Observer vector
      while(!feof(file_in))
	{
	  if (NULL==fgets(linebuf,256,file_in)) break;
	  if (linebuf[0]=='#') continue; // this is a comment
	  if (sscanf(linebuf,"%lf %lf %lf",&obs[iobs].vo.x,
		     &obs[iobs].vo.y, &obs[iobs].vo.z)<3) continue;
	  printf("ReadObs: Observer=(%lf, %lf, %lf)\n",obs[iobs].vo.x,obs[iobs].vo.y,obs[iobs].vo.z);
	  break;
	}
      
      // alloc the memory for the data points
      obs[iobs].lambda=(double*)malloc(sizeof(double)*obs[iobs].Ndata);
      obs[iobs].flux=(double*)malloc(sizeof(double)*obs[iobs].Ndata);
      obs[iobs].eflux=(double*)malloc(sizeof(double)*obs[iobs].Ndata);
      obs[iobs].ModelFlux=(double*)malloc(sizeof(double)*obs[iobs].Ndata);
      obs[iobs].CraterFlux=(double*)malloc(sizeof(double)*obs[iobs].Ndata);
      obs[iobs].FilterIndex=(int*)malloc(sizeof(int)*obs[iobs].Ndata);
      obs[iobs].ReflectedFlux=(double*)malloc(sizeof(double)*obs[iobs].Ndata);

      // alloc the memory for the derivative of the flux with respect to the params
      obs[iobs].dyda=(double**)malloc(sizeof(double)*obs[iobs].Ndata);
      for (idata=0; idata<obs[iobs].Ndata; idata++)
	{
	  obs[iobs].dyda[idata]=(double*)malloc(sizeof(double)*NPARAMS);
	  memset(obs[iobs].dyda[idata],0,sizeof(double)*NPARAMS);
	}

      //and set it to zero as default
      memset(obs[iobs].lambda,0,sizeof(double)*obs[iobs].Ndata);
      memset(obs[iobs].flux,0,sizeof(double)*obs[iobs].Ndata);
      memset(obs[iobs].eflux,0,sizeof(double)*obs[iobs].Ndata);
      memset(obs[iobs].ModelFlux,0,sizeof(double)*obs[iobs].Ndata);
      memset(obs[iobs].CraterFlux,0,sizeof(double)*obs[iobs].Ndata);
      memset(obs[iobs].FilterIndex,0,sizeof(int)*obs[iobs].Ndata);
      memset(obs[iobs].ReflectedFlux,0,sizeof(double)*obs[iobs].Ndata);

      // do some sanity checks if the mem was allocated ok
      if ((!obs[iobs].lambda)||(!obs[iobs].flux)||(!obs[iobs].eflux)
	  ||(!obs[iobs].ModelFlux)||(!obs[iobs].CraterFlux)||(!obs[iobs].FilterIndex))
	{
	  printf("E ReadObs 03 Memory not allocated\n");
	  return -3; //failed
	}

      // reads the datapoints 
      for(idata=0; idata<obs[iobs].Ndata; idata++)
	{
	  while(!feof(file_in))
	    {
	      obs[iobs].FilterIndex[idata]=-1;
	      if (NULL==fgets(linebuf,256,file_in)) break;
	      if (linebuf[0]=='#') continue; // this is a comment
	      if (sscanf(linebuf,"%lf %lf %lf %d",
			 &obs[iobs].lambda[idata],
			 &obs[iobs].flux[idata],
			 &obs[iobs].eflux[idata],
			 &obs[iobs].FilterIndex[idata])<3) continue;
	      printf("ReadObs: lambda=%lf flux=%lf sigmaflux=%lf filtIndex=%d\n",
		     obs[iobs].lambda[idata],
		     obs[iobs].flux[idata],
		     obs[iobs].eflux[idata],
		     obs[iobs].FilterIndex[idata]);
	      break;
	    }
	}
      // reset the link to the ephemeris
      obs[iobs].peph=NULL;
   }


  fclose(file_in);
  return 0;
}

int Obs2JD(OBS *obs, long nobs, double **lpjd)
{
  long 	i;
  double	*jd;
	
  jd=malloc(nobs*sizeof(double));
  if (jd==NULL)
    {
      printf("E ObsJD 01 Memory not allocated\n");
      return 0; //failed
    }
  // pass back the addresses 

  *lpjd=jd;
  for (i=0; i<nobs; i++)
    jd[i]=obs[i].JD;
  
  return 1;
}

int FreeObs(OBS *pobs, long nobs)
{
  int iobs, idata; 
  
  if(!pobs) return -1; // likely no obs
  if(nobs<=0) return -2; // likely no obs
  
  for(iobs=0; iobs<nobs; iobs++)
    {
      if (pobs[iobs].lambda) free(pobs[iobs].lambda);
      if (pobs[iobs].flux) free(pobs[iobs].flux);
      if (pobs[iobs].eflux) free(pobs[iobs].eflux);
      if (pobs[iobs].ModelFlux) free(pobs[iobs].ModelFlux);
      if (pobs[iobs].CraterFlux) free(pobs[iobs].CraterFlux);

      if (pobs[iobs].dyda) 
	{
	  for (idata=0; idata<pobs[iobs].Ndata; idata++)
	    if (pobs[iobs].dyda[idata]) free(pobs[iobs].dyda[idata]);
	  free(pobs[iobs].dyda);
	}
    }

  free(pobs);
}
