#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifndef LC_H
  #include "lc.h"
#endif

#ifndef TPM_H
  #include "tpm.h"
#endif

/*
#ifndef EPHEMS_H
  #include "ephems.h"
#endif
*/

#ifndef MESH_H
  #include "mesh.h"
#endif

#include "../tms/constants.h"
#include "../tms/tm.h"
#include "../tms/macros.h"

// typedefinition for the LM fit of the TPM
int TPM_LMFit(TPM* ptpm, EPH *eph, int Neph, OBS *pobs, int Nobs, char* pLMfl);

//#define VERSION "TPM 5.0.2 July 03, 2012\n"
// corrected bug in the calculation of the SS and SO angles with roughness and no Gamma

//#define VERSION "TPM 5.0.3 July 04, 2012\n"
// corrected bug in the allocation of the color correction table and added the freecolorcorrection

//#define VERSION "TPM 5.0.4 July 10, 2012\n"
// corrected minor bug in the following line in runtpm.c
//	      printf("using spinvector file %s (DAMIT format)\n", pSpinFileName);

// #define VERSION "TPM 5.1.0 November 06, 2012\n"
// corrected bug in the Lagrerros approximation of crater flux with thermal inertia
// when the crater is visible and illuminated use the Lagerros approximation
// when the crater is visible and NOT illuminated use flux from flat facet
//
// also other minor bugs have been corrected in the big case switch at the end of this file.

// #define VERSION "TPM 5.2.0 March 20, 2013\n"
// Victor and MDB correced bug that mutiplied twice the flux by the emissivity in the Lagerros
// crater case.

//#define VERSION "TPM 5.3.0 April 19, 2013\n"
// in tpm.c PrintFacetFlux and DumpFacetFlux have now a FLAG that takes into account
// if the code is running using the heat diffusion equation in craters or the 
// the  Lagerros approximation

// #define VERSION "TPM 5.3.1 September 20, 2013\n"
// minor modifications

//#define VERSION "TPM 5.4.1 September 28, 2013\n"
// redesigned output
// correct calculation of the size in 2 fashions
// Area Equivalent Diameter
// Volume Equivalent Diameter
// restored loop sequence in the obs output and ministeps for no crat and TI.

//#define VERSION "TPM 5.5.0 November 18, 2013\n"
// -j switch which allows the user to calculate a chi2 and resiuduals without the scale

#define VERSION "TPM 5.6.0 January 19, 2014\n"
// -e switch which allows the user to scale the error on the fluxes.

#define VERSION "TPM 5.7.0 January 20, 2014\n"
// -X switch which allows the user to scale the fluxes.

#define VERSION "TPM 5.8.0 Novembre 05, 2015\n"
// -I switch which allows the user to change the number of rings inside craters (discretization)

#define VERSION "TPM 5.9.0 December 3, 2015\n"
// Corrected bug that took dtMax from T/NptsRotation no matter how fine the ephemeris step was
// Bryce arrived very late at Rstaurant and Kalehd wanted to murder him. 

#define USAGE "\n\
  This is the ThermoPhysical Model (delbo@oca.eu) [last chage: June 05, 2012]\n\
  USAGE:\n\
  ----------------------------------\n\
  echo meshfile ephfile obsfile lambda0 beta0 P phi0 JD0 Npts epsilon\
  Gamma A gamma rho_c scaleModel modelmode | runtpm\n\
  ----------------------------------\n\
  of course over one line, where:\n\
    shapefile  & name of the file of the mesh & must have wavefront format\n\
    ephfile    & name of the file of the ephemeris & ASCII \n\
    epsilon    & infrared emissivity\n\
    Gamma      & Thermal inertia               & J m-2 s-0.5 K-1\n\
    A          & Bolometric Bond Albedo\n\
    gamma      & Semiaperture angle of craters & degree\n\
    rho_c      & crater surface density \n\
\n\
  -h this help\n\
  -T <filename> load facets tempeartures from binary file\n\
  -l <luminosity> Scale the sun luminosity: 1=Sun 10=ten times the sun 0.5=half the sun\n\
  -n <number of slabs>: number of elements in the depth buffer [default is 32]\n\
  -x <dx>: length step: default is 0.25 x the heat lenght scale=sqrt(kappa/c/rho/omega)\n\
  -t <tr>: change the temperature threshold in the check fo stabiity: default is 1 deg\n\
  -i <Ti>: initial temperature: force all facets and the depth buffer to the Ti at start\n\
  -a <JD0> <JD1>: initialize temperatures taking the average insolation between JD0 and JD1\n\
  -r <rho>: material density [kg m-3]; default= 2000 kg m-3 \n\
  -c <c>: material heat capacity [J kg-1 K-1]; default=700 J kg-1 K-1\n\
  -A <alpha>: thermal expansion coefficient [K-1]; default=0 (thermal expansion calculation deactivated) \n\
              set alpha=1.E-5 (@400 K) for chondrite like material \n\
  -m <lambda> <mu>: matarial Lame coefficients [GPa]; \n\
                   default: lambda = 35.5 GPa (@400 K) \n\
			    mu = 28.5 GPa (@400 K)\n\
  -L <model> : TPM calculate visibile lightcurves\n\
     <model>=1 Geometric scattering\n\
     <model>=2 Lambert scattering\n\
     <model>=3 Lommel Seeliger Scattering\n\
  -N uses normal files instead of mesh (obj)\n\
  -S <spinfile> with lambda (deg), beta (deg), Period (h) JD0 (days), phi0 (deg) \n\
  -o <observationfile>\n\
  -j the optimized scale will not be used for chi2 and residuals \n\
  -e <errorScale> to scale errors by a constant factor \n\
  -X <fluxScale> to scale fluxes by a constant factor  \n\						       
\n\
FORMAT of the tmap_temps.dat FILE:\n\
JD, ptpm->dt(sec), ptpm->dx(in length_scale), number of the facet, lambda, beta of the facets, NTslabs, TEMPs...\n"


/* this function calculates the Residuals to the data
given an optional dScale
MDB Madrid April 25th, 2012 

bugfix: MDB MOMO May 30, 2012
  added control on positivity of flux, to avoid using visibilities
  which are given with minus sign in the observation file.
*/
int CalcResiduals(OBS* pObs, int iNobs,  double dScale, double* pdResiduals)
{
  double dScaleSquared=dScale*dScale;
  int    iObs, iData, iTotal=0;
  double dOminusC;
  
  *pdResiduals=0;

  for (iObs=0; iObs<iNobs; iObs++)
    for (iData=0; iData<pObs[iObs].Ndata; iData++)
      if (pObs[iObs].flux[iData]>0)
	{
	  dOminusC=
	    pObs[iObs].flux[iData] // observed flux
	    -pObs[iObs].ModelFlux[iData] // MODEL thermal flux
	    *dScaleSquared // mesh scaling factor (BEST) squared
	    -pObs[iObs].ReflectedFlux[iData];
	  
	  *pdResiduals+=dOminusC*dOminusC;
	  iTotal++;
	}
  *pdResiduals/=iTotal;
  *pdResiduals=sqrt(*pdResiduals);
}

/* this function calculates the ChiSquared to the data
given an optional dScale
MDB and Victor Nice Feb 19, 2012 

bugfix: MDB MOMO May 30, 2012
  added control on positivity of flux, to avoid using visibilities
  which are given with minus sign in the observation file.
*/
int CalcChiSquared(OBS* pObs, int iNobs,  double dScale, double* pdChiSq)
{
  double dScaleSquared=dScale*dScale;
  int    iObs, iData;
  double dOminusC;
  
  *pdChiSq=0;

  for (iObs=0; iObs<iNobs; iObs++)
    for (iData=0; iData<pObs[iObs].Ndata; iData++)
      if (pObs[iObs].flux[iData]>0)
	{
	  dOminusC=
	    pObs[iObs].flux[iData] // observed flux
	    -pObs[iObs].ModelFlux[iData] // MODEL thermal flux
	    *dScaleSquared // mesh scaling factor (BEST) squared
	    -pObs[iObs].ReflectedFlux[iData];
	  
	  *pdChiSq+=dOminusC*dOminusC
	    /pObs[iObs].eflux[iData] // observed flux uncertainty
	    /pObs[iObs].eflux[iData];
	}
}

/* this function calculates the best scale that 
minimizes the ChiSquared of the data
MDB and Victor Nice Feb 19, 2012 

bugfix: MDB MOMO May 30, 2012
  added control on positivity of flux, to avoid using visibilities
  which are given with minus sign in the observation file.
*/
int CalcBestScale(OBS* pObs, int iNobs, double* pdScale)
{
  double dScale=0,
    dNumerator=0, dDenominator=0;
  int    iObs, iData;
  
  for (iObs=0; iObs<iNobs; iObs++)
    for (iData=0; iData<pObs[iObs].Ndata; iData++)
      if (pObs[iObs].flux[iData]>0)
	{
	  dNumerator+=
	    (pObs[iObs].flux[iData]-  // observed flux
	     pObs[iObs].ReflectedFlux[iData])*  // refelcted MODEL flux
	    pObs[iObs].ModelFlux[iData]/ // thermal MODEL flux
	    pObs[iObs].eflux[iData]/ // observed flux uncertainty
	    pObs[iObs].eflux[iData];
	
	  dDenominator+=
	    pObs[iObs].ModelFlux[iData]* // thermal MODEL flux
	    pObs[iObs].ModelFlux[iData]/ // squared
	    pObs[iObs].eflux[iData]/ // observed flux uncertainty
	    pObs[iObs].eflux[iData];
	}

  pdScale[0]=sqrt(dNumerator/dDenominator);
}

int compare_ephJD(const void *a, const void *b)
{
  EPH *eph_a = (EPH *) a;
  EPH *eph_b = (EPH *) b;
  
  if (eph_a->JD > eph_b->JD) return 1;
  else if (eph_a->JD == eph_b->JD) return 0;
  else return -1;
}


/* 
this function merges the obs with the Ephemerides.
i.e. it create an ephem point at the epoch of the observations

MDB March 04, 2012 fixed a bug in the chaining of obs with ephs
inserted a break in the eph loop and I check if 
 if (ieph==Noldeph) NeEphtoAdd++; 
to add a new ephemeris point.
Now the chaining works ok.
*/
int MergeEphObs(EPH** lpeph, long *lpNeph, 
		OBS* obs, long Nobs)
{
  EPH*  eph;
  long  Neph;
  EPH*  oldeph;
  long  Noldeph;

  long iobs, ieph, ioldeph, NeEphtoAdd=0;

  // saves locally the pointer to the old ephemerides
  oldeph=*lpeph;
  // and the number of ephs
  Noldeph=*lpNeph;


  // search if there is an ephemerid at one observation epoch
  // to determine how many new ephemeris points are needed
  for (iobs=0; iobs<Nobs; iobs++)
    {
      for(ieph=0; ieph<Noldeph; ieph++)
	{
	  if (fabs(oldeph[ieph].JD-obs[iobs].JD)<1.e-9) // if the difference is smaller than ~0.0001 sec..
	    {
	      // link the observation to the ephems
	      oldeph[ieph].pobs=&obs[iobs];
	      // and link the ephem to the observation
	      obs[iobs].peph=&oldeph[ieph]; 
	      printf("Linked obs %d to eph %d\n",iobs,ieph);
	      break; //...skip looping on the ephemerides
	    }	  
	}
      if (ieph==Noldeph) NeEphtoAdd++; // add to the number
    }
  printf("There are %ld ephemeris points and %ld new to be added\n",Noldeph, NeEphtoAdd);

  // ---- realloc the ephemerides buffer ! ------------------------------------------------
  // we alloc actually a new vector where we copy ephemerides
  Neph=Noldeph+NeEphtoAdd;
  if ((eph=(EPH*)malloc(sizeof(EPH)*Neph))==NULL)
    {
      printf("E: MergeEphObs err 01 Malloc error\n");
      return -1;
    }

  // fill the new eph buffer --------------------------------------------------------------
  
  // first put the ephemerides
  for (ioldeph=ieph=0; ioldeph<Noldeph; ioldeph++)
    {
      memcpy(&eph[ieph], &oldeph[ioldeph], sizeof(EPH));
      //      printf("ieph----%d\n",ieph);
      ieph++;
    }

  // then copies the observations into the eph structure
  for (iobs=0; iobs<Nobs; iobs++) // skipping those alredy linked if any
    {
      if (obs[iobs].peph!=NULL) continue; // found a link so it skip it
      eph[ieph].JD=obs[iobs].JD;
      memcpy(&eph[ieph].vs, &obs[iobs].vs, sizeof(VECTOR));
      // copy obs flags to ephems flags !!! OVEWRITE it!!!
      eph[ieph].flags=obs[iobs].flags; 
      eph[ieph].pobs=&obs[iobs]; // link the pointers
      obs[iobs].peph=&eph[ieph]; // link the pointers
      //      printf("ieph----%d\n",ieph);
      ieph++;
    }  

  // free the mem from the old eph
  free(oldeph);
  // pass back the pointers
  *lpeph=eph;
  *lpNeph=Neph;

  // signal to the caller that everythying went well
  return 0;
}

int inputdata(TPM* ptpm, 
	      EPH **lpeph,
	      long* pNeph,
	      OBS **lpobs, 
	      long* pNobs,
	      char *pObsFileName, 
	      char *pSpinFileName,
              int   usenormalfile)
{
  double    lscale=1.0;
  long	    res=0,i=0;
  char	    fname[255];
  EPH       *peph=NULL;  // ephemerides vector
  long	    neph=0;       // number of ephemerides
  OBS	    *pobs=NULL;  // observation vector
  long 	    nobs=0;       // number of observations
  FILE      *pSpinFile=NULL;

  // READ SHAPE -----------------------------------------------------------------------
  scanf("%s", fname);
  //	printf("%s\n",fname);
  if (usenormalfile)
    {
      if ( (res=GenFacetsNormals(fname, &ptpm->mesh)) <=0)
	{
	  printf("inputdata: GenFacetsNormals FAILED\n");
	  return -1;
	}
      printf("Normals %s loaded OK fn=%d; vn=%d\n",fname,ptpm->mesh.fn,ptpm->mesh.vn);
    }else
    {
      if ( (res=GenMeshWF(fname, &ptpm->mesh)) <=0)
	{
	  printf("inputdata: GenMeshWF FAILED\n");
	  return -1;
	}
      printf("mesh %s loaded OK fn=%d; vn=%d\n",fname,ptpm->mesh.fn,ptpm->mesh.vn);
    }
	
  // READ EPHEMERIDES ------------------------------------------------------------------
   scanf("%s", fname);
  if ( (res=ReadEphemerides(fname, &peph, &neph))<0)
    {
      printf("inputdata: ReadEphemerides FAILED\n");
      return -2;
    }
  printf("Ephemerides %s loaded OK N=%ld; elapsed time %f days\n",
	 fname,neph,peph[neph-1].JD-peph[0].JD);      

  // INITIALIZE THE POLE PARAMETERS of the MESH from the spinfile
  if (pSpinFileName)
    {
      pSpinFile=fopen(pSpinFileName, "rt");
      if (!pSpinFile)
	{
	  printf("Spinfile not found, WTF!\n");
	  exit(-1);
	}
      if (fscanf(pSpinFile,"%lf %lf %lf %lf %lf", &ptpm->mesh.lambda, &ptpm->mesh.beta, 
		 &ptpm->mesh.P, &ptpm->mesh.t0, &ptpm->mesh.phi0)!=5)
	return -1;
      ptpm->mesh.lambda/=RADEG;
      ptpm->mesh.beta/=RADEG;
      ptpm->mesh.P/=24.0;
      ptpm->mesh.phi0/=RADEG;
      if (pSpinFile) fclose(pSpinFile);
    }else
    {
      ptpm->mesh.lambda=0/RADEG;
      ptpm->mesh.beta=90/RADEG;
      ptpm->mesh.P=6.0/24.0;
      ptpm->mesh.phi0=0./RADEG;
      ptpm->mesh.t0=0; //JD
    }
  printf("lambda=%lf beta=%lf P=%lf t0=%lf phi0=%lf\n", 
	 ptpm->mesh.lambda, ptpm->mesh.beta, ptpm->mesh.P, 
	 ptpm->mesh.t0, ptpm->mesh.phi0);

// reads the asteroid physical parameters ----------------------------------------------
  scanf("%lf %lf %lf %lf %lf", &ptpm->epsilon, &ptpm->Gamma, &ptpm->A, 
	&ptpm->gamma, &ptpm->density);
  printf("Physical Parameters: emissivity=%f(SI); Gamma=%f(SI); A=%f CRATERS angle=%f; density=%f\n",
	 ptpm->epsilon, ptpm->Gamma, ptpm->A, ptpm->gamma, ptpm->density);
  
  // READ OBSERVATIONS-------IF A VALID POINTER IS PASSED ------------------------------------------
  //  scanf("%s", fname);
  if (pObsFileName)
    {
      if ( (res=ReadObs(pObsFileName,  &pobs, &nobs))<0)
	{
	  printf("inputdata: ReadObs FAILED\n");
	  FreeObs(pobs,nobs);
	  return -3;
	}
      printf("Observations %s loaded OK N=%ld; elapsed time %f days\n",
	     pObsFileName,nobs,pobs[nobs-1].JD-pobs[0].JD);
      


///////// INPUT of DATA completed here ////////////////////////////////////////////////

////// Merges the Obs and the Eph by creating an Eph point also on observations ///////
      if ((MergeEphObs(&peph, &neph, pobs, nobs)<0))
	{
	  printf("E: input data: MergeEphObs failed\n");
	  return -10;
	}
    }

  // sends back pointers
  *lpeph=peph;
  *pNeph=neph;
  *lpobs=pobs;
  *pNobs=nobs;

  // sort the ephemeris vector using the C standard library qsort
  //--III.4.5.2-- using user (i.e. me) made compare_ephJD function
  
  qsort(peph, neph, sizeof(EPH), compare_ephJD);
  
  for(i=0; i<neph; i++)
    {
      if (peph[i].pobs!=NULL)
	printf("eph: %d %lf %d linked_obs: %lf %d %d\n", i, peph[i].JD, peph[i].flags, 
	       peph[i].pobs->JD, peph[i].pobs->Ndata, peph[i].pobs->flags);
      else
	printf("eph: %d %lf %d\n", i, peph[i].JD, peph[i].flags);
    }
  

  return 0;// return OK
}


void calcGeometry(VECTOR obsvr, VECTOR obsvDelta, double *pr, double *pDelta, double *palpha)
{

  double      DeltaD=NORMA(obsvDelta);
  VECTOR      obsNDelta={obsvDelta.x/DeltaD, obsvDelta.y/DeltaD, obsvDelta.z/DeltaD};
  double      Deltar=NORMA(obsvr);
  VECTOR      obsNr={obsvr.x/Deltar, obsvr.y/Deltar, obsvr.z/Deltar};
  int         i;

  pr[0]   = Deltar;
  *pDelta = DeltaD;
  printf("scalar product = %lf\n",SCALAR(obsNDelta,obsNr));
  *palpha = acos(SCALAR(obsNDelta,obsNr))*RADEG; 
  
}

main(int argc, char* argv[])
{
  int      usenormalfile=0;// put it to one to read Joseph NORMAL FILES instead of a obj file
  TPM	   tpm;
  long	   npts=300;
  long	   N;
  double   dt;
  OBS	   *pobs=NULL;
  long	   Nobs=0;
  EPH      *peph=NULL;
  long     Neph=0;
  long	   i, iObs, iData;
  double   *pfc=NULL;
  double   scale=1.0, chisq=0, residuals=0;
  double   volume=0;
  double   surface=0;
  time_t   seconds;
  double   dSunLuminosity=1.0;
  int      NTslabs=32;
  double   dx=0.25; // this is the value of dx in depth = 0.25 of the heat length_scale = sqrt(kappa/rho/c/omega)
  double   tr=1.0; // temperature threshold is set to 1 deg
  double   ti=-1.0;
  int      lcmode=0, NValidObs=0;
  double   JD0, JD1;
  double   lame_mu=28.5,     // Lame's parameter (GPa) default value
           lame_lambda=35.5, // Lame's parameter (GPa) default value
           alpha=0.0;        // Thermal expansion coefficeint (if 0, Duhamel-Neumann equation deactivated)
  double   rho=2000.0, c=700.0;
  long     NpointsPerRotation=360; // a calculation point every deg //DEFAULT
  char     *pObsFileName=NULL; // reset the filename to NULL
  double   BeamingParameter=1.0;
  
  //Victor, 2012-02-14
  char     *pCCFileName=NULL; //filename of color correction table-containing file
  double   H=-1.0,G=0.15, Rp=1.5;
  double   r,Delta,alfa;
  char     *pSpinFileName=NULL;
  char     lmfit=0;
  char     *pLMflags=NULL;
  int      HDcrater=0;
  int      noScaleChi2=0;
  double   errorScale=1.0;
  double   fluxScale=1.0;
  int      iNRingsInCraters=4;

  printf("%s",VERSION);
  //	SCANS the arguments
  while (argc>1)
    {
      if (argv[1][0]=='-')
	{
	  switch (argv[1][1])
	    {	      
	    case 'h'://-h print the help of the TPM
	        printf("%s\n",USAGE); exit(0);
	      break;
	    case 'I'://-I <Nrings> :: change the number of rings by which craters are discretized
	      argc--; argv++; iNRingsInCraters=atoi(argv[1]);
	      printf("Warning: number of of Rings In Craters changed to:%d\n",iNRingsInCraters);	      
	      break;
	    case 'l'://-l <luminosity_factor> change Sun Luminosity. The new luminosity is 1373*luminosty_factor W m-2
	      argc--; argv++; dSunLuminosity=atof(argv[1]);
	      printf("Warning: Luminosity of the Sun is scaled by %lf: i.e. %lf W m-2\n",
		     dSunLuminosity, dSunLuminosity*SOLARC);
	      break;
	    case 'n'://-n <Nslabs> Change number of slabs in the subsurface (default is 32)
	      argc--; argv++; NTslabs=atoi(argv[1]);
	      printf("Warning: number of slabs changed to:%d\n",NTslabs);
	      break;
	    case 'x'://-x <dx> Change the dx (steps in depth) in unity of the heat penetration depth (default is 0.25)
	      argc--; argv++; dx=atof(argv[1]);
	      printf("Warning: dx changed to %lf\n", dx);
	      break;
	    case 't'://-t <delta_Temperature> Change the temperature threshold to check for stability (unit K)
	      argc--; argv++; tr=atof(argv[1]);
	      printf("Warning: temperature thershold changed to %lf\n", tr);
	      break;
	    case 'i'://-i <Temperature> Force all facets and all crater facets to have the same initial <Temperature> (unit K)
	      argc--; argv++; ti=atof(argv[1]);
	      printf("Warning: initial temperature set to %lf\n", ti);
	      break;
	    case 'L'://-L <mode> disable thermal modeling and enable lightcurve modeling (don't use it)
	      argc--; argv++; lcmode=atoi(argv[1]);
	      printf("Warning: Visble Lightcurve generation mode %d\n", lcmode);
	      break;
	    case 'T'://-T <filename> Facet temperature are initiallized to the values in the file
	      argc--; argv++; strcpy(tpm.initialtempsfilename,argv[1]);
	      ti=-2.0;
	      printf("Warning: Temperatures will be initalized to those from file: %s\n", tpm.initialtempsfilename);
	    case 'a'://-a <JD0> <JD1> Temperatures will be initalized using avarge insolation between <JD0> and <JD1>
	      argc--; argv++; JD0=atof(argv[1]);
	      argc--; argv++; JD1=atof(argv[1]);
	      ti=-3.0;
	      printf("Warning: Temperatures will be initalized using avarge insolation between %lf and %lf\n",JD0,JD1);
	      break;
	    case 'm':
	      argc--; argv++; lame_mu=atof(argv[1]);
	      argc--; argv++; lame_lambda=atof(argv[1]);
	      printf("Warning: Lame''s parameters changed to lambda=%lf and mu=%lf\n",lame_lambda,lame_mu);
	      break;
	    case 'A':
	      argc--; argv++; alpha=atof(argv[1]);
	      printf("Warning: thermal expansion coefficient changed to alpha=%lf (K-1)\n",alpha);
	      if (alpha != 0.0) printf("Thermal expansion (Duhamel-Neuman equation) activated\n");
	      break;
	    case 'c':
	      argc--; argv++; c=atof(argv[1]);
	      printf("Warning: heat capacity changed to c=%lf \n",c);
	      break;
	    case 'r':
	      argc--; argv++; rho=atof(argv[1]);
	      printf("Warning: density changed to rho=%lf (kg m-3)\n",rho);
	      break;
	    case 'N'://-N interpret the shape file as a bunch of normals and not as an .obj (shape must be convex)
	      usenormalfile=1;
	      printf("Warning: Using a files of NORMALS::: no OBJ file\n",rho);
	      break;
	    case 'p'://-p <Npoints_per_rotation> Number of ministeps/rotation (default 360). Ephemeris are interpolated to have <Npoints_per_rotation>
	      argc--; argv++; NpointsPerRotation=atoi(argv[1]);
	      printf("Warning: Number of points per rotation changed to %ld\n",NpointsPerRotation);
	      break;
	    case 's'://-s <scale> Mesh vectors are scaled by the factor <scale> The facet area is multiplied by <scale>^2
	      argc--; argv++; scale=atof(argv[1]);
	      printf("Warning: mesh scale factor %lf\n",scale);
	      break;
	    case 'o'://-o <filename> Observations from <filename> are linked to the thermal model. Model fluxes are compared to observed ones
	      argc--; argv++; pObsFileName=argv[1]; // store the pointer to the filename
	      printf("Using observations file %s\n",pObsFileName);
	      break;
	    case 'b'://-b <eta> The default beaming paramter of 1 is changed to the value of <eta> This chnages the surface temperature.
	      argc--; argv++; BeamingParameter=atof(argv[1]);
	      printf("Using a Beaming Parameter value of  %lf\n",BeamingParameter);
	      break;
	    case 'C'://-C <filename> Model fluxes are color corrected using a color correction file
	      //Load a color correction table
	      argc--; argv++; pCCFileName=argv[1]; //store pointer to the filename
	      printf("Using color correction table in file %s\n",pCCFileName);
	      break;
	    case 'H'://-H <H> The <H> value of the H,G system is used to calculate model reflected light component
	      //Input H value
	      argc--; argv++; H=atof(argv[1]); //
	      //	      printf("Using color correction table in file %s\n",pCCFileName);
	      break;
	    case 'G'://-G <G> The <G> value of the H,G system is used to calculate model reflected light component (default 0.15)
	      //Input G value
	      argc--; argv++; G=atof(argv[1]); 
	      //printf("Using color correction table in file %s\n",pCCFileName);
	      break;
	    case 'R'://-R <R_p> Ratio between the IR and the visibile albedo 
	      //Input R_p value
	      argc--; argv++; Rp=atof(argv[1]); //store pointer to the filename
	      //printf("Using color correction table in file %s\n",pCCFileName);
	      break;
	    case 'S'://-S <filename> Uses spin data from filename. Format: see J. Duerch's DAMIT (lambda, beta, period, JD0, phi0)
	      argc--; argv++; 
	      pSpinFileName=(char*)malloc(strlen(argv[1])+1);
	      strcpy(pSpinFileName, argv[1]);
	      printf("using spinvector file %s (DAMIT format)\n", pSpinFileName);
	      break;
	    case 'F': 
	      lmfit=1;
	      argc--; argv++;
	      pLMflags=(char*)malloc(strlen(argv[1]));
	      strcpy(pLMflags, argv[1]);
	      printf("Performing a Levenberg Marquard fit (JEE); %s\n",pLMflags);	      
	      break;
	    case 'f'://-f Performs full blown heat diffusion in crater tiles instead of using J. Lagerros approximation
	      HDcrater=1;
	      printf("Using full heat diffusion model in craters (no Lagerros' approx)\n");
	      break;
	    case 'j'://-j Do not use the optimized scale for chi2 and residuals output 
	      printf("Warning: optimized scale will NOT be used for chi2 and residuals \n");
	      noScaleChi2=1;
	      break;
	    case 'e'://-e <scale> Multiply observed flux uncertanties by <scale>
	      //Input of the error Scale Factor
	      argc--; argv++; errorScale=atof(argv[1]); 
	      printf("Warning: TPM will multiply measured Errors by %lf factor \n",errorScale);	      
	      break;
	    case 'E'://-e <scale> flux uncertanties = flux * <scale>
	      //Input of the error Scale Factor
	      argc--; argv++; errorScale=-atof(argv[1]); 
	      printf("Warning: TPM will assign Errors to be %lf factor of flxes\n",-errorScale);	      
	      break;
	    case 'X'://-X <scale> Multiply observed fluxes by <scale>
	      //Input of the flux Scale Factor
	      argc--; argv++; fluxScale=atof(argv[1]); 
	      printf("Warning: TPM will multiply measured Fluxes by %lf factor \n",fluxScale);	      
	      break;
	    default:
	      printf("unknown option -%s\n",&argv[1][1]);
	      return 0;
	      break;
	    } //switch
	}
      argc--; argv++;
    }

  seconds = time (NULL);

  memset(&tpm,0,sizeof(TPM));
  if (inputdata(&tpm, &peph, &Neph, &pobs, &Nobs, pObsFileName, pSpinFileName, usenormalfile) < 0)
    {
      printf("runtpm: input data failed\n");
      exit(-1);
    }

  // ------- initialize the tpm Structure ------------------------------------
  //	tpm.A=0.1; tpm.Gamma=300.0; tpm.epsilon=1.0;
  tpm.JD0=tpm.JD1=0;
  if (ti==-3) 
    {
      tpm.JD0=JD0;
      tpm.JD1=JD1;
    }
  tpm.Tsid=tpm.mesh.P*3600.0*24.0;
  tpm.NTslabs=NTslabs; tpm.dx=dx; 
  tpm.dt=tpm.dtMAX=tpm.Tsid/NpointsPerRotation; // time step is in seconds here 
  tpm.SunLuminosity=dSunLuminosity;
  tpm.tr=tr;
  tpm.ti=ti;
  tpm.mu=lame_mu;
  tpm.lambda=lame_lambda;
  tpm.alpha=alpha;
  tpm.rho=rho;
  tpm.c=c;
  tpm.kappa=tpm.Gamma*tpm.Gamma/tpm.rho/tpm.c;
  printf("TPM structure initalisation completed\n");
  tpm.heatlen=sqrt(tpm.kappa/tpm.rho/tpm.c/2/3.1415*tpm.Tsid);
  tpm.dz=tpm.heatlen*tpm.dx;
  tpm.mode=1;// lagerros approx TBD: let the user change it.
  tpm.MeshScaleFactor=1.0;
  //Victor: what is going on with the scale and the mesh? 

  //Input parameters for the FitLM option:
  tpm.BeamingParameter=BeamingParameter;    //default is 1.0
  tpm.D2=scale*scale;                       //default is scale = 1.0
  tpm.Rp=Rp;                                //default is 1.5
  tpm.H=H;                                  //default is -1.0
  tpm.G=G;                                  //default is 0.15
  tpm.iNRingsInCraters=iNRingsInCraters;    //default is 4


// scale the mesh if required --------------------------------------------------
  
  if ((scale!=1) && lmfit<1)
    {
      printf("LMfit not activated\n");
      if (!usenormalfile) 
	{
	  volume=meshvolume(&tpm.mesh);
	  printf("Initial volume of the mesh %lf ", volume);
	  scalemesh(&tpm.mesh, scale);
	  volume=meshvolume(&tpm.mesh);
	  printf("Now SCALED: volume of the mesh %lf\n", volume);
	}
      else
	{
	for(i=0; i<tpm.mesh.fn; i++)
	  tpm.mesh.n[i].mod=tpm.mesh.n[i].mod*scale*scale;   
	}  
      scale=1;
    }

  // --------------------
  // It allocates the Buffers for the temperatures for each facet
  //----------------------
  if (AllocTempB(&tpm)<=0) return -1;
  printf("Mesh temperature buffers allocated OK\n");
  if ((tpm.density>0) && (tpm.gamma>0))
    {
      putCraters(&tpm, tpm.gamma, tpm.density);
      printf("Craters added: Hapke thetabar=%5.2f\n", tpm.crat[0].thetabar);
    }

  /*
// thermalmechanical expansion of the rocks -------------------------------------------
  if (tpm.alpha>0) 
    {
      printf("Thermal expansion coefficient is %lf [m-1]\n", tpm.alpha);
      if (AllocDisplacementB(&tpm)<=0) {FreeDisplacementB(&tpm); return -33;}
      printf("Displacements buffers allocated OK\n");
      printf("Duhamel-Neumann equation will be solved in order to calculate the thermal displacement\n");
      printf("see Capek and Vokrouhlicky. Thermal stresses in small meteoroids. A&A (2010) vol. 519 pp. 75\n");
      printf("THERMO-MECHANICAL PARAMETERS:\n");
      printf("kappa=%lf, rho=%lf, c=%lf, heat penet. depth=%lf\nalpha=%lf, lambda=%lf, mu=%lf, dz=%lf\n",
	     tpm.kappa, tpm.rho, tpm.c,tpm.heatlen, tpm.alpha, 
	     tpm.lambda,tpm.mu, tpm.dz);
    }
  */

  //We point to the color correction table with tpm.pCCorr
  //and load it
  if (pCCFileName)
    {
      tpm.pCCorr=(ColorCorrection*)malloc(sizeof(ColorCorrection));
      if(!tpm.pCCorr)
	{
	  printf("runtpm main: Color Correction malloc error\n");
	}

      ColorCorrectionRead(pCCFileName,tpm.pCCorr);
      
    }


  //     If the H value is specified and the Rp is greater than 0, 
  //     add the reflected light scaled by the value at 3.35 um
  
  if((tpm.H>0.0) && (tpm.Rp>0.0))
    {
      //we calculate the reflected light component from H,r,Delta,alfa 	  
      for (iObs=0; iObs<Nobs; iObs++)
	{
	  calcGeometry(pobs[iObs].vs, pobs[iObs].vo,&r,&Delta,&alfa);
	  printf("iobs = %d; r = %lf, Delta = %lf, alfa = %lf\n",iObs,r,Delta,alfa);
	  AddReflComp_Jy(pobs[iObs].lambda,pobs[iObs].ReflectedFlux,pobs[iObs].Ndata,
			 tpm.H,tpm.G,r,Delta,alfa,tpm.Rp);
	}
    }
  else
    {
      printf("Warning: no input values for H, and pIR/pV were given\n");
    }


  // MDB and Victor on Fucking April 6th in nice. 
  // Marco took a nap snoring like a devil. 
  // print facets normals to test if we get the same answer using the 
  // bunch of normal file of the obj triangles i.e. the mesh  
  /*
  printf("n> %d\n", tpm.mesh.fn);
  for(i=0; i<tpm.mesh.fn; i++)
    {
    printf("n> %lf\nn> %lf %lf %lf\n",
	   tpm.mesh.n[i].mod/2,
	   tpm.mesh.n[i].vec.x,  tpm.mesh.n[i].vec.y,  tpm.mesh.n[i].vec.z);
    }  
  */

  // scale the error = flux scale //MDB with VIC 24 may 2016 Nice
  if (errorScale<0)
    {
      errorScale*=-1;
      printf("Warning: scaling flux errors ... ");
      for (iObs=0; iObs<Nobs; iObs++)
	for (iData=0; iData<pobs[iObs].Ndata; iData++)
	  pobs[iObs].eflux[iData]=pobs[iObs].flux[iData]*errorScale;
      printf("done\n");
      errorScale=1;
    }

  // scale the error if errorScale!=1 //MDB 19 Jan 2014 Nice
  if (errorScale!=1)
    {
      printf("Warning: scaling flux errors ... ");
      for (iObs=0; iObs<Nobs; iObs++)
	for (iData=0; iData<pobs[iObs].Ndata; iData++)
	  pobs[iObs].eflux[iData]*=errorScale;
      printf("done\n");
    }

  // scale the fluxes if fluxScale!=1 //MDB 20 Jan 2014 Nice
  if (fluxScale!=1)
    {
      printf("Warning: scaling fluxes ... ");
      for (iObs=0; iObs<Nobs; iObs++)
	for (iData=0; iData<pobs[iObs].Ndata; iData++)
	  pobs[iObs].flux[iData]*=fluxScale;
      printf("done\n");
    }
  
  if (lmfit)
    {
      TPM_LMFit(&tpm, peph, Neph, pobs, Nobs, pLMflags);
    } else
    {
      if (lcmode)
	{
	  // TBD LC mode to be resumed
	  //printf("Starting TPM with %ld points and %ld observations VISUAL LC MODE\n", N, Nobs);
	  //runlc(&tpm, pjd, psa, N, pobs, Nobs, &pfc, lcmode);
	  //chisq=lcchisq(pfc, pobs, Nobs);
	} else
	{
	  printf("Starting TPM with %ld Ephemeris points and %ld observations\n", Neph, Nobs);
	  if ((tpm.Gamma <=0) && ((tpm.gamma<=0) || (tpm.density<=0))) // no TI, no crater no lagerros
	    { 
	      TPM_noTI_noCrat_Flux(&tpm, peph, Neph, pobs, Nobs);
	    } 
	  if ((tpm.Gamma <=0) && ((tpm.gamma>0) && (tpm.density>0)) &&
	      (HDcrater==0) ) // no TI, and craters with Lagerros approx
	    { 
	      TPM_noTI_Crat_Lagerros_Flux(&tpm, peph, Neph, pobs, Nobs);
	    } 
	  if ((tpm.Gamma <=0) && ((tpm.gamma>0) && (tpm.density>0)) &&
	      (HDcrater==1) ) // no TI, and craters
	    { 
	      TPM_noTI_Crat_Flux(&tpm, peph, Neph, pobs, Nobs);
	    } 
	  if ((tpm.Gamma >0) && ((tpm.gamma<=0) || (tpm.density<=0)) ) // TI, no crater
	    { 
	      TPM_TI_noCrat_Flux(&tpm, peph, Neph, pobs, Nobs);
	    } 
	  if ((tpm.Gamma >0) && ((tpm.gamma>0) && (tpm.density>0)) &&
				 (HDcrater==0) ) // TI and craters with Lagerros approx
	    { 
	      TPM_TI_Crat_Lagerros_Flux(&tpm, peph, Neph, pobs, Nobs);
	    } 
	  if ((tpm.Gamma >0) && ((tpm.gamma>0) && (tpm.density>0)) &&
				 (HDcrater==1) ) // TI and craters FULL heat diffusion in crater
	    { 
	      TPM_TI_Crat_Flux(&tpm, peph, Neph, pobs, Nobs);
	    } 

	  printf("Done TPM\n"); 

	  /*
	    AFTER TPM has run
	  */
	  if (pobs)
	    {
	      if ((tpm.mesh.v)) 	    
		volume=meshvolume(&tpm.mesh);
	      else 
		volume=0;
	      surface=meshsurface(&tpm.mesh);
	    } else
	    {
	      volume=0;
	      surface=0;
	    }

	  CalcBestScale(pobs, Nobs, &scale);
	  if (noScaleChi2) 
	    {
	      CalcChiSquared(pobs, Nobs, 1.0, &chisq);
	      CalcResiduals(pobs, Nobs, 1.0, &residuals);
	    } else
	    {
	      CalcChiSquared(pobs, Nobs, scale, &chisq);
	      CalcResiduals(pobs, Nobs, scale, &residuals);
	    }
	  
	  //	  printf("\n\tGamma\t tbar\t scale\t\t eff_diam\t beaming parameter\t chi^2\tresiduals\n");
	  printf("h> %9s %5s %5s %5s %5s %10s %10s %10s %6s %10s %10s\n", 
	    "Gamma", "t.Bar", "gamma", "rhoC", "A", "scale", "Dv", "Ds", "eta", "chi^2", "RMS");
	  if (tpm.crat)
	    printf("r> %9.1lf %5.1lf %5.1lf %5.2lf %5.2lf %10.8lf %10.3lf %10.3lf %6.3lf %10.3lf %10.5lf\n", 
		   tpm.Gamma, tpm.crat[0].thetabar, 
		   tpm.gamma*RADEG, tpm.density, tpm.A,  
		   scale, 2.0*pow(3.0*volume/4.0/DPI,1.0/3.0)*scale, 
		   2.0*sqrt(surface/4.0/DPI)*scale,
		   tpm.BeamingParameter, chisq, residuals);
	  else
	    printf("r> %9.1lf %5.1lf %5.1lf %5.2lf %5.2lf %10.8lf %10.3lf %10.3lf %6.3lf %10.3lf %10.5lf\n", 
		   tpm.Gamma, 0.0, 
		   tpm.gamma*RADEG, tpm.density, tpm.A,  
		   scale, 2.0*pow(3.0*volume/4.0/DPI,1.0/3.0)*scale, 
		   2.0*sqrt(surface/4.0/DPI)*scale,
		   tpm.BeamingParameter, chisq, residuals);
	}
    }



  //cleanup -----------
  FreeTempB(&tpm); 
  printf("Mesh temperature buffers freed OK\n");
  if (peph) free(peph); 
  printf("Ephemeris vector freed OK\n");
  if (pobs) FreeObs(pobs,Nobs); 
  printf("Observation vector freed OK\n");
  if (pSpinFileName) free(pSpinFileName);
  printf("pSpinFileName freed OK\n");
  if (tpm.pCCorr) 
    {
      ColorCorrectionFree(tpm.pCCorr);
      free(tpm.pCCorr);
      printf("ColorCorrection Table freed OK\n");
    }

  // TBD: Mesh, Craters etc to be freed !!!
  

  printf ("TPM run in %ld seconds\n", time(NULL)-seconds);

}

/*
	to prepare the ephemerides
 /cygdrive/z/fitTM/hor2vec 3.iras.eph.txt 2445530 2445573.5 > 3.iras.eph
	to extract obs from the IRAS catalogue and create the obs file
/cygdrive/z/fitTM/extiras juno | awk '{print $3,$4,$5,$6}' >3.iras.obs
	to compile
gcc -O2 runtpm.c mesh.c tpm.c ephems.c obs.c craters.c -oruntpm.exe
	under linux
gcc -O3 -lm runtpm.c mesh.c tpm.c ephems.c obs.c craters.c -oruntpm
	to run it
objfile                  eph           obs         lambda  beta   P(hrs)            phi0   Epoch                  Gamma A  scale and send to the tpm stdin
 echo 3.tri_1.obj 3.iras.eph 3.iras.obs 103.32 26.76 7.20953079 180.0 2434752.779320 150.0 0.1 200.0 | /cygdrive/z/fitTM/runtpm.exe
*/

