//#include <sys\stat.h>       /* S_ constant definitions */
//#include <fcntl.h>          /* O_ constant definitions */
#include <time.h>
#include <float.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef TPM_H
  #include "tpm.h"
#endif

int SaveTemp(TPM* ptpm)
{
  if (!ptpm->pTold) // if not... 
    { // it assumes that the vector has to be initalized
      printf("MaxTemp: allocating vector to MaxTemp....");
      if ((ptpm->pTold=(double*)malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
	{
	  // the alloc did not succed
	  printf("ALLOC FAILED\n");
	  return -1;
	}
      printf("oldTemperatures alloc Done\n");
    }
  memcpy(ptpm->pTold, ptpm->pTemp, sizeof(double)*ptpm->mesh.fn);
  return 0;
}

int MaxTempDiff(TPM* ptpm, double* ptDiff)
{
  int ifacet;
  double tDiff=0;

  *ptDiff=0;

  if (!ptpm->pTold) // if not... 
    return -1;

  for (ifacet=0; ifacet<ptpm->mesh.fn; ifacet++)// for each facet
    {
      tDiff=fabs(ptpm->pTemp[ifacet]-ptpm->pTold[ifacet]);
      if (tDiff > *ptDiff) *ptDiff = tDiff;
    }
}

int MaxTemp(TPM* ptpm, double** lpMaxTemp)
{
  long ifacet;
  double *pMaxTemp;
  
  // get the pointer
  pMaxTemp=*lpMaxTemp;

  // check if the vector points to something 
  if (!pMaxTemp) // if not... 
    { // it assumes that the vector has to be initalized
      printf("MaxTemp: allocating vector to MaxTemp....");
      if ((pMaxTemp=(double*)malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
	{
	  // the alloc did not succed
	  printf("ALLOC FAILED\n");
	  return -1;
	}
      printf("Done\n");
      // OK... so now put the temperatures of the ptpm in the MaxTemp vector
      for (ifacet=0; ifacet<ptpm->mesh.fn; ifacet++)
	{
	  pMaxTemp[ifacet]=ptpm->pTemp[ifacet];
	}
      // pass back the pointer
      *lpMaxTemp=pMaxTemp;
      // return success
      return 0;
    }
  else // the vector is already allocated
    {
      for (ifacet=0; ifacet<ptpm->mesh.fn; ifacet++)// for each facet
	  pMaxTemp[ifacet]=max(pMaxTemp[ifacet],ptpm->pTemp[ifacet]);
      // put the max between the stored T and the actual T
    }
}

int MinTemp(TPM* ptpm, double** lpMinTemp)
{
  long ifacet;
  double *pMinTemp;
  
  // get the pointer
  pMinTemp=*lpMinTemp;

  // check if the vector points to something 
  if (!pMinTemp) // if not... 
    { // it assumes that the vector has to be initalized
      printf("MinTemp: allocating vector to MinTemp....");
      if ((pMinTemp=(double*)malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
	{
	  // the alloc did not succed
	  printf("ALLOC FAILED\n");
	  return -1;
	}
      printf("Done\n");
      // OK... so now put the temperatures of the ptpm in the MinTemp vector
      for (ifacet=0; ifacet<ptpm->mesh.fn; ifacet++)
	{
	  pMinTemp[ifacet]=ptpm->pTemp[ifacet];
	}
      // pass back the pointer
      *lpMinTemp=pMinTemp;
      // return success
      return 0;
    }
  else // the vector is already allocated
    {
      for (ifacet=0; ifacet<ptpm->mesh.fn; ifacet++)// for each facet
	  pMinTemp[ifacet]=min(pMinTemp[ifacet],ptpm->pTemp[ifacet]);
      // put the min between the stored T and the actual T
    }
}

int AllocTempB(TPM* ptpm) 
{
  int	i, j;
  
  ptpm->pTB = (double**)malloc(sizeof(double*)*ptpm->mesh.fn);
  if (!ptpm->pTB)
    {
      printf("AllocTempB E01 Cannot allocate Temperature Buffer\n");
      return -1;
    }
  
  for (i=0; i<ptpm->mesh.fn; i++) 
    {
      ptpm->pTB[i]=(double*)malloc(sizeof(double)*ptpm->NTslabs);
      if (!ptpm->pTB[i]) i = ptpm->mesh.fn+10;
		 
      for (j=0; j<ptpm->NTslabs; j++)	// initialize temperature buffer to zero
	ptpm->pTB[i][j]=0.0;
    }
  
  ptpm->pTB_dt=(double*)malloc(sizeof(double)*ptpm->NTslabs);
  if (!ptpm->pTB_dt) i = ptpm->mesh.fn+10;
  
  for (j=0; j<ptpm->NTslabs; j++)		// initialize temperature buffer to zero
    ptpm->pTB_dt[j]=0.0;
  
  if (i > ptpm->mesh.fn)
    {
      printf("AllocTempB E02 Cannot allocate Temperature Buffer\n");
      FreeTempB(ptpm);
      return -1;
    }
  
  return 1;  
}

int FreeTempB(TPM* ptpm)
{
  int	i;
  
  for (i=0; i<ptpm->mesh.fn; i++)			// frees Buffers for each facet
    if (!ptpm->pTB[i]) free(ptpm->pTB[i]);		
  
  if (!ptpm->pTB) free(ptpm->pTB);			// frees main pointer
  if (!ptpm->pTB_dt) free(ptpm->pTB_dt);
}

// this function allocates the crat vector and initialize 
// the crater structure
//
// of course you should have defined the crater properties 
// by using the IDLsetAsteroidParam(),
// however, you may want to change them provding additional
// parameter to this function
// if Crater opening angle is less than or equal zero 
// this function performs the correct (I hope) deallocation of
// craters and just return. IDLputCraters() can therefore be used
// to wipe crters out the asteroid's surface
int putCraters(TPM* ptpm, double gamma, double density)
{
  int			i;
  VECTOR		vSouth;	// define the south direction for each crater
  VECTOR		*v1, vtpm, vX={1.0, 0.0, 0.0};

  ptpm->gamma   = gamma/RADEG;
  ptpm->density = density;
  
  if (ptpm->crat!=NULL) //need to be dealocated
    {
      for (i=0; i<ptpm->Ncrat; i++)
	freecrater(&ptpm->crat[i]);
      free(ptpm->crat);
      ptpm->crat=NULL;
      ptpm->Ncrat=0;
    }

  if (ptpm->LUT) 
    {
      cratShadowLUTdelete(ptpm->LUT);	// destroy the Lookup table
      ptpm->LUT=NULL;
    }

// if the crater opening angle is less than or equal to zero 
// craters have not to be created on the surface of the asteroid
  if (ptpm->gamma <=0)	
    {
      return -1;					// just return
    }

  ptpm->crat=(CRATER*)malloc(sizeof(CRATER)*ptpm->mesh.fn);
  if (!ptpm->crat) 
    {
      printf("putCraters E01: crater vector alloc failed\n");
      return -1;			// failed to alloc crater vector 
    }
    
  ptpm->Ncrat=ptpm->mesh.fn;		// there is one crater for each tile of the mesh
  if ((ptpm->mesh.c!=NULL) && (ptpm->mesh.v!=NULL))	// if centers and vertices are present 
    {							// then make craters with int the usual  way
      v1=ptpm->mesh.v-1;   // make base-1 vector arithmetic
      for (i=0; i<ptpm->Ncrat; i++)
	{
	  vtpm.x=ptpm->mesh.c[i].x-v1[ptpm->mesh.f[i].v1].x;
	  vtpm.y=ptpm->mesh.c[i].y-v1[ptpm->mesh.f[i].v1].y;
	  vtpm.z=ptpm->mesh.c[i].z-v1[ptpm->mesh.f[i].v1].z;
	  normalize(vtpm, &vSouth);
	  
	  initcrater(&ptpm->crat[i],	// point to the crater vector, one crater for each mesh tile
		     ptpm->iNRingsInCraters,				// Default is 4 ring-crater
		     ptpm->gamma,		// crater openeing angle in radians 
		     1.0,				// radius of the crater
		     ptpm->A,			// bond albedo
		     ptpm->epsilon,	// emissivity
		     ptpm->Gamma,		// surface thermal inertia
		     ptpm->density,	// surface density of craters
		     ptpm->mesh.n[i].vec,		// normal of the crater=normal to the tile
		     vSouth);
	}
    } else	// centers are not allocated (-> Mikko's like vector of areas is used)
    {
      // since vertex mesh is not defined, we need to define the south direction a different way
      // comments: it looks like it works...
      // MDB040827 Novi-> To
      // 
      // Praha 1/11/11. Josef thinks that it wold be better to use the vZ instead of vX.
      // this is because in their convex inversion there should be no normals parallel to the Z
      // however, since we are not sure... we leave as it is.
      for (i=0; i<ptpm->Ncrat; i++)
	{
	  // the South crater on each tile is defined as the vector product of the normal
	  // and the x axis. Such vector is orthogonal to the normal of the tile.
	  vtpm.x=VECx(ptpm->mesh.n[i].vec, vX);
	  vtpm.y=VECy(ptpm->mesh.n[i].vec, vX);
	  vtpm.z=VECz(ptpm->mesh.n[i].vec, vX);
	  normalize(vtpm, &vSouth);
	  
	  initcrater(&ptpm->crat[i],	// point to the crater vector, one crater for each mesh tile
		     ptpm->iNRingsInCraters,				// Default is 4 ring-crater
		     ptpm->gamma,		// crater openeing angle in radians 
		     1.0,				// radius of the crater
		     ptpm->A,			// bond albedo
		     ptpm->epsilon,	// emissivity
		     ptpm->Gamma,		// surface thermal inertia
		     ptpm->density,	// surface density of craters
		     ptpm->mesh.n[i].vec,		// normal of the crater=normal to the tile
		     vSouth);
	}
    }

  cratShadowLUTcreate(&ptpm->crat[0], &ptpm->LUT);
  return 1;
}

int PrintTemp(TPM* ptpm, EPH *peph, FILE** ppfh)
{
  int     ifacet, ix;
  FILE*	  pfh;
  double  lambda, beta, rxy;
  VECTOR  vSun;

  pfh=ppfh[0];
  printf("PrintTemp.. ");
  if (pfh==NULL)
    {
      if ( (pfh=fopen("tmap_temps.dat", "wt"))==NULL)
	{
	  fprintf(stderr, "PrintTemp: cannot open tmap_temps.dat file\n");	  
	  return -1;
	}
      // pass back pointer
      *ppfh=pfh;
    }
  
  memcpy(&vSun, &peph->vs, sizeof(VECTOR));
  Eclip2Ast(&vSun, ptpm->mesh.lambda, ptpm->mesh.beta, 
	    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, peph->JD);

  if (ptpm->mesh.c)
    {

      for (ifacet=0; ifacet<ptpm->mesh.fn; ifacet++)	// for each facet
	{
	  rxy=sqrt(ptpm->mesh.c[ifacet].x*ptpm->mesh.c[ifacet].x + ptpm->mesh.c[ifacet].y*ptpm->mesh.c[ifacet].y);
	  lambda=atan2(ptpm->mesh.c[ifacet].y, ptpm->mesh.c[ifacet].x)*RADEG;
	  if (lambda<0) lambda+=360.0;
	  if (rxy!=0)
	    beta=atan(ptpm->mesh.c[ifacet].z/rxy)*RADEG;
	  else
	    {
	      if (ptpm->mesh.c[ifacet].z>0) beta=90.0;
	      else beta=-90.0;
	    }
	  fprintf(pfh, "%lf\t%lf\t%lf", peph->JD, ptpm->dt, ptpm->dx);
	  fprintf(pfh, "%5d %6.1lf %6.1lf\t%d\t", ifacet, lambda, beta, ptpm->NTslabs);
	  
	  // calcola la SubSolar Longitude and Latitude
	  rxy=sqrt(vSun.x*vSun.x + vSun.y*vSun.y);
	  lambda=atan2(vSun.y, vSun.x)*RADEG;
	  if (lambda<0) lambda+=360.0;
	  if (rxy!=0)
	    beta=atan(vSun.z/rxy)*RADEG;
	  else
	    {
	      if (vSun.z>0) beta=90.0;
	      else beta=-90.0;
	    }

	  //      fprintf(pfh, "%20.10lf\t", pobs->JD);
	  fprintf(pfh, "%6.1lf %6.1lf ", lambda, beta);

	  // Print the TEMPERATURES

	  if (ptpm->Gamma>0)		// if there is a buffer of temps towards the core
	    {
	      for (ix=0; ix<ptpm->NTslabs; ix++)
		fprintf(pfh, " %7.3lf", ptpm->pTB[ifacet][ix]);	
	      if (ptpm->crat)		// if there is a buffer of temps towards the core
		{
		  fprintf(pfh,"\nCRAT> %lf\t", peph->JD);
		  for (ix=0; ix<ptpm->crat[ifacet].Nt; ix++)
		    fprintf(pfh, " %5.1lf", ptpm->crat[ifacet].pT[ix]);
		}		  
	    } else
	    {
	      fprintf(pfh, " %5.1lf", ptpm->pTemp[ifacet]);		
	    }
	  fprintf(pfh, "\n");	
	}
    } 
  else
    { // the centers of the facets are not defined...
      // put lambda and beta to unphysical values
      lambda=-400;
      beta=-400;
      for (ifacet=0; ifacet<ptpm->mesh.fn; ifacet++)	// for each facet
	{
	  fprintf(pfh, "%lf\t%lf\t%lf", peph->JD, ptpm->dt, ptpm->dx);
	  fprintf(pfh, "%5d %6.1lf %6.1lf\t%d\t", ifacet, lambda, beta, ptpm->NTslabs);
	  
	  if (ptpm->Gamma>0)		// if there is a buffer of temps towards the core
	    {
	      for (ix=0; ix<ptpm->NTslabs; ix++)
		fprintf(pfh, " %5.1lf", ptpm->pTB[ifacet][ix]);		
		} else
	    {
	      fprintf(pfh, " %5.1lf", ptpm->pTemp[ifacet]);		
	    }
	  fprintf(pfh, "\n");	
	}
    }	
  printf("... print temp done\n");
}

int LoadTemp(TPM* ptpm)
{
  FILE* pfh=NULL;
  unsigned int fn=0, NTslabs=0, i, j, ix;
  float *fdum=NULL;

  printf("Loading Temperatures... from %s\t", ptpm->initialtempsfilename);

  if ( (fdum=malloc(sizeof(float)*10000))==NULL)
    {
      fprintf(stderr, "LoadTemp: cannot allocate FDUMMY buffer\n");	  
      return -1;
    }

  if ( (pfh=fopen(ptpm->initialtempsfilename, "rb"))==NULL) //open for reading in binary mode
    {
      fprintf(stderr, "LoadTemp: cannot open tmap_temps.bin file\n");	  
      if (fdum) free(fdum);
      return -1;
    }
  fread(&fn, sizeof(ptpm->mesh.fn),1,pfh);//read the number of facets
  fread(&NTslabs, sizeof(ptpm->NTslabs),1,pfh);//read the number of slabs
  fread(fdum, sizeof(double),1,pfh);//read JD (double)
  
  if( (fn!=ptpm->mesh.fn) || (NTslabs!=ptpm->NTslabs))
    {
      fprintf(stderr, "LoadTemp: data are not compatible: stop\n");	  
      if (pfh) fclose(pfh); // attempts to close the file.
      if (fdum) free(fdum);
      return -2; 
    }
   
  fread(fdum, sizeof(float), 2, pfh);//read dt and dx

  for (i=0; i<ptpm->mesh.fn; i++)	// for each facet
    {

      fread(&j, sizeof(j),1,pfh);//read facet index
      fread(fdum, sizeof(float), 3, pfh);//load lambda beta etc ... won't be used though
      
      if (ptpm->Gamma>0)		// if there is a buffer of temps towards the core
	{
	  fread(fdum,sizeof(float),ptpm->NTslabs,pfh);// load the whole temperatures depth buffer
	  for (ix=0; ix<ptpm->NTslabs; ix++)
	    ptpm->pTB[i][ix]=(double)fdum[ix]; // store back in the TPM /// WOOWOWOWOW
	} else
	{
	  fread(fdum,sizeof(float),1,pfh);
	  ptpm->pTemp[i]=(double)fdum[0];
	}
    }	

  if (pfh) fclose(pfh); // attempts to close the file.
  if (fdum) free(fdum);
  printf("DONE.. OK\n");
}

int	DumpTemp(TPM* ptpm, double JD, FILE** ppfh)
{
	int		i, ix;
	FILE*		pfh;
	double		lambda, beta, rxy;
	float  fdum[512];

	pfh=ppfh[0];

	if (pfh==NULL)
	{
	  if ( (pfh=fopen("tmap_temps.bin", "wb"))==NULL) //open for writing in binary mode
	  {
	    fprintf(stderr, "DumpTemp: cannot open tmap_temps.bin file\n");	  
	    return -1;
	  }
	  // pass back pointer
	  *ppfh=pfh;
	}

	fwrite(&ptpm->mesh.fn, sizeof(ptpm->mesh.fn),1,pfh);//write the number of facets
	fwrite(&ptpm->NTslabs, sizeof(ptpm->NTslabs),1,pfh);//write the number of slabs
	fwrite(&JD, sizeof(JD),1,pfh);//write JD (double)
	fdum[0]=(float)ptpm->dt;
	fdum[1]=(float)ptpm->dx;
	fwrite(fdum, sizeof(float), 2, pfh);//write dt and dx
	
	for (i=0; i<ptpm->mesh.fn; i++)	// for each facet
	{
	  rxy=sqrt(ptpm->mesh.c[i].x*ptpm->mesh.c[i].x + ptpm->mesh.c[i].y*ptpm->mesh.c[i].y);
	  lambda=atan2(ptpm->mesh.c[i].y, ptpm->mesh.c[i].x)*RADEG;
	  if (lambda<0) lambda+=360.0;
	  if (rxy!=0)
	    beta=atan(ptpm->mesh.c[i].z/rxy)*RADEG;
	  else 
	  { 
	    if (ptpm->mesh.c[i].z>0) beta=90.0;
	    else beta=-90.0;
	  }
//fprintf(pfh, "%4d  %6.1lf %6.1lf %6.1lf %6.1lf %6.1lf\t", i, ptpm->mesh.c[i].x, ptpm->mesh.c[i].y, ptpm->mesh.c[i].z,lambda, beta);

	  fwrite(&i, sizeof(i),1,pfh);//write facet index
	  fdum[0]=(float)lambda;
	  fdum[1]=(float)beta;
	  fdum[2]=(float)ptpm->mesh.n[i].mod/2;
	  fwrite(fdum, sizeof(float), 3, pfh);//write lambda and beta of the facet
	
		if (ptpm->Gamma>0)		// if there is a buffer of temps towards the core
		{
		  for (ix=0; ix<ptpm->NTslabs; ix++)
		    fdum[ix]=(float)ptpm->pTB[i][ix];
		  fwrite(fdum,sizeof(float),ptpm->NTslabs,pfh);
		} else
		{
		  fdum[0]=(float)ptpm->pTemp[i];
		  fwrite(fdum,sizeof(float),1,pfh);
		}
	}	
}

int PrintCraterTempStatistics(TPM* ptpm, double JD, FILE** ppfh)
{
  int     i, ix;
  FILE*	  pfh;
  double  lambda, beta, rxy;
  double  Tave, Tstddev, Tmax, Tmin;

  pfh=ppfh[0];
  printf("Printing facets and craters temperature Statistics ... ");

 if (pfh==NULL)
    {
      if ( (pfh=fopen("Tmap_Crat_Temps_Stat.txt", "wt"))==NULL)
	{
	  fprintf(stderr, "Tmap_Crat_TempsStatistics: cannot open Tmap_Crat_Temps_Stat file\n");	  
	  return -1;
	}
      // pass back pointer
      *ppfh=pfh;
    }
  
  if (ptpm->mesh.c)
    {
      for (i=0; i<ptpm->mesh.fn; i++)	// for each facet
	{
	  rxy=sqrt(ptpm->mesh.c[i].x*ptpm->mesh.c[i].x + ptpm->mesh.c[i].y*ptpm->mesh.c[i].y);
	  lambda=atan2(ptpm->mesh.c[i].y, ptpm->mesh.c[i].x)*RADEG;
	  if (lambda<0) lambda+=360.0;
	  if (rxy!=0)
	    beta=atan(ptpm->mesh.c[i].z/rxy)*RADEG;
	  else
	    {
	      if (ptpm->mesh.c[i].z>0) beta=90.0;
	      else beta=-90.0;
	    }
	  //		fprintf(pfh, "%4d  %6.1lf %6.1lf %6.1lf %6.1lf %6.1lf\t", i, ptpm->mesh.c[i].x, ptpm->mesh.c[i].y, ptpm->mesh.c[i].z,lambda, beta);
	  fprintf(pfh, "%lf\t%lf\t%lf", JD, ptpm->dt, ptpm->dx);
	  fprintf(pfh, "%5d %6.1lf %6.1lf\t%d\t", i, lambda, beta, ptpm->NTslabs);
	  
	  fprintf(pfh, " %5.1lf", ptpm->pTemp[i]);		
	  CraterCalcTempreatureStatistics(&ptpm->crat[i], &Tave, &Tstddev, &Tmax, &Tmin);
	  fprintf(pfh, " %5.1lf %5.1lf %5.1lf %5.1lf ", Tave, Tstddev, Tmax, Tmin);		

	  fprintf(pfh, "\n");	
	}
    } 
  else
    { // the centers of the facets are not defined...
      // put lambda and beta to unphysical values
      lambda=-400;
      beta=-400;
      for (i=0; i<ptpm->mesh.fn; i++)	// for each facet
	{
	  fprintf(pfh, "%lf\t%lf\t%lf", JD, ptpm->dt, ptpm->dx);
	  fprintf(pfh, "%5d %6.1lf %6.1lf\t%d\t", i, lambda, beta, ptpm->NTslabs);
	  
	  if (ptpm->Gamma>0)		// if there is a buffer of temps towards the core
	    {
	      for (ix=0; ix<ptpm->NTslabs; ix++)
		fprintf(pfh, " %5.1lf", ptpm->pTB[i][ix]);		
		} else
	    {
	      fprintf(pfh, " %5.1lf", ptpm->pTemp[i]);		
	    }
	  fprintf(pfh, "\n");	
	}
    }	
  printf("... print temp done\n");
}


int PrintFacetFlux(TPM* ptpm, OBS *pobs, FILE** ppfh, int FLAGS)
{
  int    ifacet, idata;
  FILE*  pfh;
  double lambda, beta, rxy;
  double lambda5=0; //pobs->lambda*pobs->lambda*pobs->lambda*pobs->lambda*pobs->lambda;
  double *pflux=NULL, *pfluxCrater=NULL;
  double DeltaD=NORMA(pobs->vo);
  double rD=NORMA(pobs->vs);
  VECTOR obsN={pobs->vo.x/DeltaD, pobs->vo.y/DeltaD, pobs->vo.z/DeltaD};
  VECTOR sunN={pobs->vs.x/rD, pobs->vs.y/rD, pobs->vs.z/rD};
  VECTOR vNorthMesh={0,0,1};
  VECTOR vSouthFacet;
  VECTOR vWestFacet;
  VECTOR vProjObsFacet;
  VECTOR vtmp;
  double DeltaQAU = DeltaD*DeltaD*AU*AU;
  double ps=0, B=0, psOnWest, psOnSouth;
  double ccfactor=1;

  pfh=ppfh[0];
  printf("running PrintFacetFlux\n");
  
  if (pfh==NULL)
    {
      if ( (pfh=fopen("tmap_flux.dat", "wt"))==NULL)
	{
	  fprintf(stderr, "PrintFacetFlux: cannot open tmap_flux.dat file\n");	  
	  return -1;
	}
      // pass back pointer
      *ppfh=pfh;
    }

  // allocate the flux vector for all wavelegnths
  if ( (pflux=(double*)malloc(pobs->Ndata*sizeof(double)))==NULL)
    {
      fprintf(stderr, "PrintFacetFlux: cannot allocate flux vector\n");	  
      return -3;
    }
  // allocate the Crater flux vector for all wavelegnths
  if ( (pfluxCrater=(double*)malloc(pobs->Ndata*sizeof(double)))==NULL)
    {
      fprintf(stderr, "PrintFacetFlux: cannot allocate flux vector\n");	  
      return -3;
    }
  // loop over facets
  for (ifacet=0; ifacet<ptpm->mesh.fn; ifacet++)
    {
      /* CALC GEOMETRY */
      
      rxy=sqrt(ptpm->mesh.c[ifacet].x*ptpm->mesh.c[ifacet].x + ptpm->mesh.c[ifacet].y*ptpm->mesh.c[ifacet].y);
      lambda=atan2(ptpm->mesh.c[ifacet].y, ptpm->mesh.c[ifacet].x)*RADEG;
      if (lambda<0) lambda+=360.0;
      if (rxy!=0)
	beta=atan(ptpm->mesh.c[ifacet].z/rxy)*RADEG;
      else
	{
	  if (ptpm->mesh.c[ifacet].z>0) beta=90.0;
	  else beta=-90.0;
	}
      fprintf(pfh, "%20.10lf\t", pobs->JD);
      fprintf(pfh, "%5d %6.1lf %6.1lf ", ifacet, lambda, beta);
      
      /*
      rxy=sqrt(ptpm->mesh.n[ifacet].vec.x*ptpm->mesh.n[ifacet].vec.x + ptpm->mesh.n[ifacet].vec,y*ptpm->mesh.n[ifacet].vec.y);
      lambda=atan2(ptpm->mesh.n[ifacet].vec.y, ptpm->mesh.n[ifacet].vec.x)*RADEG;
      if (lambda<0) lambda+=360.0;
      if (rxy!=0)
	beta=atan(ptpm->mesh.n[ifacet].vec.z/rxy)*RADEG;
      else
	{
	  if (ptpm->mesh.n[ifacet].vec.z>0) beta=90.0;
	  else beta=-90.0;
	}
      fprintf(pfh, "%20.10lf\t", pobs->JD);
      fprintf(pfh, "%5d %6.1lf %6.1lf %6.1lf\t", ifacet, lambda, beta, ptpm->pTemp[ifacet]);
      */
      
      ps=SCALAR(ptpm->mesh.n[ifacet].vec,obsN);

      // calcola il vettore West on the facet 
      vtmp.x=VECx(ptpm->mesh.n[ifacet].vec,vNorthMesh);
      vtmp.y=VECy(ptpm->mesh.n[ifacet].vec,vNorthMesh);
      vtmp.z=VECz(ptpm->mesh.n[ifacet].vec,vNorthMesh);

      // Vest deve essere Normalizzato.
      // questo e' by definition nel piano della faccetta.
      normalize(vtmp, &vWestFacet);

      // calcola il vettore Sud proiettato sulla faccetta
      // no need to normalize it as Normal and West are unit long
      // and perperndicular.
      vSouthFacet.x=VECx(ptpm->mesh.n[ifacet].vec, vWestFacet);
      vSouthFacet.y=VECy(ptpm->mesh.n[ifacet].vec, vWestFacet);
      vSouthFacet.z=VECz(ptpm->mesh.n[ifacet].vec, vWestFacet);
      

      // calcola il vettore proiezione di obsN nel piano della faccetta
      // questo e' obsN - (Normal*obsN)Normal
      vtmp.x=obsN.x-ps*ptpm->mesh.n[ifacet].vec.x;
      vtmp.y=obsN.y-ps*ptpm->mesh.n[ifacet].vec.y;
      vtmp.z=obsN.z-ps*ptpm->mesh.n[ifacet].vec.z;

      // normalizzi sto vettore
      normalize(vtmp, &vProjObsFacet);
      
      // angolo del vettore con il vettore est 
      psOnWest=SCALAR(vProjObsFacet,vWestFacet);
      // angolo del vettore con il nord locale
      psOnSouth=SCALAR(vProjObsFacet,vSouthFacet);      

      beta=90-acos(ps)*RADEG;
      lambda=atan2(psOnWest,psOnSouth)*RADEG + 180.0;
      //      fprintf(pfh, "%20.10lf\t", pobs->JD);
      fprintf(pfh, "%6.1lf %6.1lf %6.1lf %6.1lf\t", lambda, beta, DeltaD, ptpm->pTemp[ifacet]);

      /* RESET FLUX BUFFERS */
      memset(pflux,0,sizeof(double)*pobs->Ndata);      // reset the flux vector to zero
      memset(pfluxCrater,0,sizeof(double)*pobs->Ndata);      // reset the Craterflux vector to zero
      ps=SCALAR(ptpm->mesh.n[ifacet].vec,obsN);

      /* GO FOR FLUX CALC */
     if (ps>0)   // if the facet is NOT tilted away from the observer 
	{
	  if ((ptpm->density>0) && (ptpm->gamma>0)) // if there are craters 
	    {
	      // get the crater flux for all wavelenghts
	      if ( (FLAGS & LAGERROSCRATER)==LAGERROSCRATER)
		GetCraterFluxLagerros(&ptpm->crat[ifacet], obsN, pobs->lambda, pfluxCrater, 
				      pobs->Ndata, ptpm->LUT, pobs->FilterIndex, ptpm->pCCorr); 
	      if ( (FLAGS & HEATDIFFUSIONCRATER)==HEATDIFFUSIONCRATER)
	        getcraterflux(&ptpm->crat[ifacet], obsN, pobs->lambda, pfluxCrater, 
			      pobs->Ndata, ptpm->LUT, pobs->FilterIndex, ptpm->pCCorr); // get the crater flux for all wavelenghts
	      
	      for (idata=0; idata<pobs->Ndata; idata++) // for each wavelength
		{
		  /* get the color correction for the flat area if necessary */
		  if ((pobs->FilterIndex[idata] >= 0) && (ptpm->pCCorr))
		    {
		      //there is a color correction to perform:
		      //we round the float by casting
		      if((int)ptpm->pTemp[ifacet] > 999) printf("Temperature too high!\n");
		      
		      //we extract the color corr. factor corresponding to this filter and this T
		      ccfactor = ptpm->pCCorr->CCTable[pobs->FilterIndex[idata]][(int)ptpm->pTemp[ifacet]];
		    } else ccfactor=1;
		  
		  lambda5=pobs->lambda[idata]*pobs->lambda[idata]*pobs->lambda[idata]*
		    pobs->lambda[idata]*pobs->lambda[idata];	      
		  B=NIST_C1/lambda5/(exp(NIST_C2/(pobs->lambda[idata]*ptpm->pTemp[ifacet]))-1)/PI; //in steradians (diveded by pi)
		  pflux[idata]=(B  
				* ptpm->epsilon 
				* (1.0-ptpm->density)
				* ccfactor
				+ pfluxCrater[idata]*ptpm->density/ptpm->crat[ifacet].projArea/ps)/
		    DeltaQAU*pobs->lambda[idata]*pobs->lambda[idata]*0.33357e12;
		  
		  fprintf(pfh,"%7.2lf %le  ", pobs->lambda[idata], pflux[idata]);
		} 
	    } else // i.e. NO CRATERS
	    {
	      for (idata=0; idata<pobs->Ndata; idata++) // for each wavelength
		{
		  /* get the color correction for the flat area if necessary */
		  if ((pobs->FilterIndex[idata] >= 0) && (ptpm->pCCorr))
		    {
		      //there is a color correction to perform:
		      //we round the float by casting
		      if((int)ptpm->pTemp[ifacet] > 999) printf("Temperature too high!\n");
		      
		      //we extract the color corr. factor corresponding to this filter and this T
		      ccfactor = ptpm->pCCorr->CCTable[pobs->FilterIndex[idata]][(int)ptpm->pTemp[ifacet]];
		    } else ccfactor=1;

		  lambda5=pobs->lambda[idata]*pobs->lambda[idata]*pobs->lambda[idata]*
		    pobs->lambda[idata]*pobs->lambda[idata];	      
		  B=NIST_C1/lambda5/(exp(NIST_C2/(pobs->lambda[idata]*ptpm->pTemp[ifacet]))-1)/PI; //in steradians (diveded by pi)
		  pflux[idata]=B*ptpm->epsilon/DeltaQAU*ccfactor*
		    pobs->lambda[idata]*pobs->lambda[idata]*0.33357e12;

		  fprintf(pfh,"%7.2lf %le  ", pobs->lambda[idata], pflux[idata]);
		}
	    }
	} // if ps>0 i.e. facet not tilted away
     else
       { 
	 for (idata=0; idata<pobs->Ndata; idata++) // for each wavelength
	   fprintf(pfh,"%7.2lf %le  ", pobs->lambda[idata], pflux[idata]);
       } // no flux from this facet

     fprintf(pfh, "\n");		         
    } // end loop over facets
  
    	
      /* Clean UP */
      if (pflux) free(pflux);
      if (pfluxCrater) free(pfluxCrater);
}


/* 
this function calculates and write in a binary file the flux from each facet
the thermalmodeler calls the fucntion for each observation
and a plane is crated for each wavelenght
Modification history: crated the for A. Matter PhD thesis in 2010

6 march 2012 ESAC Villafranca del Castillio (Espagne)

19 april 2013 OCA: added FLAGS to take into account of LAGERRROS CRATER MODE or NOT
added the ColorCorrection term
*/
int DumpFacetFlux(TPM* ptpm, OBS *pobs, FILE** ppfh, int FLAGS)
{

  unsigned int ifacet, idata;
  FILE*  pfh;
  double lambda, beta, rxy;
  double lambda5=0; //pobs->lambda*pobs->lambda*pobs->lambda*pobs->lambda*pobs->lambda;
  double *pflux=NULL;
  double *pfluxCrater=NULL;
  double DeltaD=NORMA(pobs->vo);
  VECTOR obsN={pobs->vo.x/DeltaD, pobs->vo.y/DeltaD, pobs->vo.z/DeltaD};
  double DeltaQAU = DeltaD*DeltaD*AU*AU;
  double ps=0, B=0;
  double flc=0;
  double ccfactor=1;

  pfh=ppfh[0];
  printf("running DumpFacetFlux\n");
  
  // write the general info about the object if the file is opened for the first time
  if (pfh==NULL)
    {
      if ( (pfh=fopen("tmap_flux.bin", "wb"))==NULL)
	{
	  fprintf(stderr, "DumpFacetFlux: cannot open tmap_flux.bin file\n");	  
	  return -1;
	}
      // pass back pointer
      *ppfh=pfh;
      // write the TPM relevant informations
      fwrite(&ptpm->mesh.fn,sizeof(ptpm->mesh.fn),1,pfh);//int
      fwrite(&ptpm->mesh.isconcave,sizeof(ptpm->mesh.isconcave),1,pfh);//int
      fwrite(&ptpm->mesh.lambda,sizeof(ptpm->mesh.lambda),1,pfh);//double
      fwrite(&ptpm->mesh.beta,sizeof(ptpm->mesh.beta),1,pfh);//double
      fwrite(&ptpm->mesh.P,sizeof(ptpm->mesh.P),1,pfh);//double
      fwrite(&ptpm->mesh.phi0,sizeof(ptpm->mesh.phi0),1,pfh);//double
      fwrite(&ptpm->mesh.t0,sizeof(ptpm->mesh.t0),1,pfh);//double
      fwrite(&ptpm->mesh.maxsize,sizeof(ptpm->mesh.maxsize),1,pfh); //double     
      printf("done writing header in the tmap_flux.bin  ");
    }

  fwrite(&pobs->JD, sizeof(double), 1, pfh); // write JD
  fwrite(&pobs->vo, sizeof(VECTOR), 1, pfh); // write vector to the observer
  fwrite(&pobs->vs, sizeof(VECTOR), 1, pfh); // write vector to the sun
  fwrite(&pobs->flags, sizeof(unsigned int), 1, pfh); // write obs flags
  fwrite(&pobs->Ndata, sizeof(int), 1, pfh); // write ndata
  fwrite(pobs->lambda, sizeof(double), pobs->Ndata, pfh); // write the wavelengths
  fwrite(pobs->flux, sizeof(double), pobs->Ndata, pfh); // write the observed flux
  fwrite(pobs->eflux, sizeof(double), pobs->Ndata, pfh); // write the uncertainty on the observed flux

  // allocate the flux vector for all wavelegnths
  if ( (pflux=(double*)malloc(pobs->Ndata*sizeof(double)))==NULL)
    {
      fprintf(stderr, "DumpFacetFlux: cannot allocate flux vector\n");	  
      return -3;
    }
  // allocate the Crater flux vector for all wavelegnths
  if ( (pfluxCrater=(double*)malloc(pobs->Ndata*sizeof(double)))==NULL)
    {
      fprintf(stderr, "DumpFacetFlux: cannot allocate flux vector\n");	  
      return -3;
    }

  // loop over facets
  for (ifacet=0; ifacet<ptpm->mesh.fn; ifacet++)
    {
      memset(pflux,0,sizeof(double)*pobs->Ndata);      // reset the flux vector to zero
      memset(pfluxCrater,0,sizeof(double)*pobs->Ndata);      // reset the Craterflux vector to zero
      ps=SCALAR(ptpm->mesh.n[ifacet].vec,obsN);

      //     printf("dumping flux of facet %d ", ifacet);
     if (ps>0)   // if the facet is NOT tilted away from the observer 
	{
	  //	  printf("visible ");
	  if ((ptpm->density>0) && (ptpm->gamma>0)) // if there are craters 
	    {
	      // get the crater flux for all wavelenghts
	      if ( (FLAGS & LAGERROSCRATER)==LAGERROSCRATER)
		GetCraterFluxLagerros(&ptpm->crat[ifacet], obsN, pobs->lambda, pfluxCrater, 
				      pobs->Ndata, ptpm->LUT, pobs->FilterIndex, ptpm->pCCorr); 
	      if ( (FLAGS & HEATDIFFUSIONCRATER)==HEATDIFFUSIONCRATER)
	        getcraterflux(&ptpm->crat[ifacet], obsN, pobs->lambda, pfluxCrater, 
			      pobs->Ndata, ptpm->LUT, pobs->FilterIndex, ptpm->pCCorr); // get the crater flux for all wavelenghts
	      
	      for (idata=0; idata<pobs->Ndata; idata++) // for each wavelength
		{
		  /* get the color correction for the flat area if necessary */
		  if ((pobs->FilterIndex[idata] >= 0) && (ptpm->pCCorr))
		    {
		      //there is a color correction to perform:
		      //we round the float by casting
		      if((int)ptpm->pTemp[ifacet] > 999) printf("Temperature too high!\n");
		      
		      //we extract the color corr. factor corresponding to this filter and this T
		      ccfactor = ptpm->pCCorr->CCTable[pobs->FilterIndex[idata]][(int)ptpm->pTemp[ifacet]];
		    } else ccfactor=1;
		  
		  lambda5=pobs->lambda[idata]*pobs->lambda[idata]*pobs->lambda[idata]*
		    pobs->lambda[idata]*pobs->lambda[idata];	      
		  B=NIST_C1/lambda5/(exp(NIST_C2/(pobs->lambda[idata]*ptpm->pTemp[ifacet]))-1)/PI; //in steradians (diveded by pi)
		  pflux[idata]=(B  
				* ptpm->epsilon 
				* (1.0-ptpm->density)
				* ccfactor
				+ pfluxCrater[idata]*ptpm->density/ptpm->crat[ifacet].projArea/ps)/
		    DeltaQAU*pobs->lambda[idata]*pobs->lambda[idata]*0.33357e12;
		} 
	    } else
	    {
	      for (idata=0; idata<pobs->Ndata; idata++) // for each wavelength
		{
		  /* get the color correction for the flat area if necessary */
		  if ((pobs->FilterIndex[idata] >= 0) && (ptpm->pCCorr))
		    {
		      //there is a color correction to perform:
		      //we round the float by casting
		      if((int)ptpm->pTemp[ifacet] > 999) printf("Temperature too high!\n");
		      
		      //we extract the color corr. factor corresponding to this filter and this T
		      ccfactor = ptpm->pCCorr->CCTable[pobs->FilterIndex[idata]][(int)ptpm->pTemp[ifacet]];
		    } else ccfactor=1;
		  
		  lambda5=pobs->lambda[idata]*pobs->lambda[idata]*pobs->lambda[idata]*
		    pobs->lambda[idata]*pobs->lambda[idata];	      
		  B=NIST_C1/lambda5/(exp(NIST_C2/(pobs->lambda[idata]*ptpm->pTemp[ifacet]))-1)/PI; //in steradians (diveded by pi)
		  pflux[idata]=B*ptpm->epsilon/DeltaQAU*ccfactor*
		    pobs->lambda[idata]*pobs->lambda[idata]*0.33357e12;
		}
	    }
	}
     printf("%d.", ifacet+1);
     fwrite(pflux,sizeof(double), pobs->Ndata, pfh); // flux from the facet/projected area vector  
    }
  printf("\n");

  free(pflux);
  free(pfluxCrater);
}

int DiffuseHeat(TPM* ptpm, double HelioD)
{
  int	i, ix;
  double  omega=D2PI/(ptpm->Tsid), sqrtomega=sqrt(omega);
  double  c1=omega/ptpm->dx/ptpm->dx;
  double  c2=sqrtomega/ptpm->Gamma/ptpm->dx;
  double	cons1 = ptpm->dt* omega/ptpm->dx/ptpm->dx;
  double	cons2 = 2.*ptpm->dt* sqrtomega/ptpm->Gamma/ptpm->dx;
  double	cons3 = (1.0 - ptpm->A)*SOLARC*ptpm->SunLuminosity/HelioD/HelioD; 
  double  dt=ptpm->dt, t=0;

	// do iterations!!!
  for (i=0; i<ptpm->mesh.fn; i++)	// for each facet
    {
      dt=ptpm->dt, t=0;
      ptpm->NiterDiffHeat=0;

      //      while (t<ptpm->dt)
      //     	{
	  //	  dt=min(dt, ptpm->tr/2/fabs(c1*(ptpm->pTB[i][1]-ptpm->pTB[i][0]) - 
	  //				     c2* ( ptpm->epsilon * SIGMA * 
	  //				   ptpm->pTB[i][0]*ptpm->pTB[i][0]*ptpm->pTB[i][0]*ptpm->pTB[i][0] - 
	  //				   cons3 * ptpm->mesh.ill[i])));
	  //	    printf("%lf\t",ptpm->pTB[i][1]-ptpm->pTB[i][0]);
	  cons1 = dt*omega/ptpm->dx/ptpm->dx;
	  cons2 = 2.*dt*sqrtomega/ptpm->Gamma/ptpm->dx;
	  // heat diffusion equation in the depth
	  for (ix=1; ix<(ptpm->NTslabs-1); ix++)
	    ptpm->pTB_dt[ix] = ptpm->pTB[i][ix] + cons1* (ptpm->pTB[i][ix+1] - 2*ptpm->pTB[i][ix] + ptpm->pTB[i][ix-1]);
	  // implicit Crank-Nicolson
	  //	  for (ix=1; ix<(ptpm->NTslabs-1); ix++)
	  //  ptpm->pTB_dt[ix] = ptpm->pTB[i][ix] + cons1/2* (ptpm->pTB_dt[ix+1] - 2*ptpm->pTB_dt[ix] + ptpm->pTB_dt[ix-1]+
	  //						     ptpm->pTB[i][ix+1] - 2*ptpm->pTB[i][ix] + ptpm->pTB[i][ix-1]);
	  
	  //calculate the surface temperature: boundary condition
	  ptpm->pTB_dt[0] = ptpm->pTB[i][0] + 
	    2.* cons1 * ( ptpm->pTB[i][1]-ptpm->pTB[i][0] ) -
	    cons2 * ( ptpm->epsilon * SIGMA * 
		      ptpm->pTB[i][0]*ptpm->pTB[i][0]*ptpm->pTB[i][0]*ptpm->pTB[i][0] - 
		      cons3 * ptpm->mesh.ill[i]);
	  
	  // impose boundary condition at latest elements of the buffer
	  ptpm->pTB_dt[ix] = ptpm->pTB_dt[ix-1] = ptpm->pTB[i][ix-2];

      	  for (ix=1; ix<(ptpm->NTslabs-1); ix++)
	  {
	    if (ptpm->pTB_dt[ix]<0) ptpm->pTB_dt[ix]=0;
	  }
	  
	  memcpy(ptpm->pTB[i], ptpm->pTB_dt, sizeof(double)*ptpm->NTslabs);
	  t+=dt;
	  ptpm->NiterDiffHeat++;
	  //	  printf("-Bryce_Better_Jewitt-> %ld \n",ptpm->NiterDiffHeat);
	  //}						// for each iteration
    } //for each facet
}

int calcTemp(TPM* ptpm, VECTOR sun)
{
  double	TSS;
  int		i;
  double	HelioD=NORMA(sun);
  VECTOR	sunN={sun.x/HelioD, sun.y/HelioD, sun.z/HelioD};

  if (meshillumination(&ptpm->mesh, sun)<=0)
    {
      printf("calcTemp E02:  meshillumination Failed\n");
      return -1;	// error
    }
  memset(ptpm->pTemp, 0, sizeof(float)*ptpm->mesh.fn);  
  
  TSS=562.341325*sqrt(sqrt((1-ptpm->A)*SOLARCONSTANT*ptpm->SunLuminosity/ptpm->epsilon
			   /STEFBOLTZ/ptpm->BeamingParameter   // 562.341325 is the fourth root of 1e11 (unit conversion)
			   )/ HelioD);
  
  for (i=0;i<ptpm->mesh.fn;i++)
    ptpm->pTemp[i]=TSS*sqrt(sqrt(ptpm->mesh.ill[i])); // TSS* (cos i)1/4
	
  return 1; // success
}
 
int calcTempTI(TPM* ptpm, VECTOR sun)
{
	double	TSS;
	int		i;
	double	HelioD=NORMA(sun);
	VECTOR	sunN={sun.x/HelioD, sun.y/HelioD, sun.z/HelioD};

// calculate the illumination of the Sun
	meshillumination(&ptpm->mesh, sun);	
// reset the temperature buffer	
        memset(ptpm->pTemp, 0, sizeof(float)*ptpm->mesh.fn);  

// calculate the heatdiffusion
	DiffuseHeat(ptpm, HelioD);

// copy temperatures back to temperature buffer
    for (i=0;i<ptpm->mesh.fn;i++)
		  ptpm->pTemp[i]=ptpm->pTB[i][0];
	return 1; // success
}

int calcTempCrat(TPM* ptpm, VECTOR sun)
{
  double	TSS;
  int		i;
  double	HelioD=NORMA(sun);
  VECTOR	sunN={sun.x/HelioD, sun.y/HelioD, sun.z/HelioD};


  meshillumination(&ptpm->mesh, sun);
  memset(ptpm->pTemp, 0, sizeof(float)*ptpm->mesh.fn);  
  TSS=562.341325*sqrt(sqrt((1-ptpm->A)*SOLARCONSTANT*ptpm->SunLuminosity/ptpm->epsilon
			   /STEFBOLTZ/ptpm->BeamingParameter   // 562.341325 is the fourth root of 1e11 (unit conversion)
			   )/ HelioD);

  for (i=0;i<ptpm->mesh.fn;i++)
    {
      if (ptpm->mesh.ill[i] <=0) continue;
      ptpm->pTemp[i]=TSS*sqrt(sqrt(ptpm->mesh.ill[i])); // TSS* (cos i)1/4  
      setcratertemp(&ptpm->crat[i],  sunN, HelioD,  ptpm->LUT); //// TBD -- PUT IT BACK
      //    SetCraterTempLagerros(&ptpm->crat[i],  sunN, HelioD,  ptpm->LUT);
    }
  return 1; // success
}

/*
int calcTempCratLag(TPM* ptpm, VECTOR sun)	// Lagerros Approximation
{
	double	TSS;
	int		i;
	double	HelioD=NORMA(sun);
	VECTOR	sunN={sun.x/HelioD, sun.y/HelioD, sun.z/HelioD};


	meshillumination(&ptpm->mesh, sun);
	for (i=0;i<ptpm->mesh.fn;i++)
		memset(ptpm->crat[i].pT, 0, ptpm->crat[i].Nt*sizeof(double));

  	TSS=562.341325*sqrt(sqrt(
						(1-ptpm->A)*SOLARCONSTANT/ptpm->epsilon
						/STEFBOLTZ/ptpm->eta   // 562.341325 is the fourth root of 1e11 (unit conversion)
						)/ HelioD);

	for (i=0;i<ptpm->mesh.fn;i++)
	{
		if (ptpm->mesh.ill[i]<=0) continue;
		setcratertemp(&ptpm->crat[i],  sunN, HelioD,  ptpm->LUT);
		multiplycratertemp(&ptpm->crat[i], 
			ptpm->pTemp[i]/(TSS*sqrt(sqrt(ptpm->mesh.ill[i]))));
	}

	return 1; // success
}
*/

int calcTempTICrat(TPM* ptpm, VECTOR sun)
{
	double	TSS;
	int		i;
	double	HelioD=NORMA(sun);
	VECTOR	sunN={sun.x/HelioD, sun.y/HelioD, sun.z/HelioD};

	meshillumination(&ptpm->mesh, sun);	
    memset(ptpm->pTemp, 0, sizeof(float)*ptpm->mesh.fn);  
	DiffuseHeat(ptpm, HelioD);
	for (i=0;i<ptpm->mesh.fn;i++)
	{
		ptpm->pTemp[i]=ptpm->pTB[i][0];
	}
	
	for (i=0; i<ptpm->mesh.fn; i++)
	{
		setcratertempHD(&ptpm->crat[i], sunN, HelioD, ptpm->LUT, ptpm->dx, ptpm->dt, ptpm->Tsid);
	}
	return 1; // success
}


/* ------------------------------------------------------------
   This function computes the flux from the object
-------------------------------------------------------------*/
int calcFlux(TPM* ptpm, VECTOR obsv, double* plambda, double* pflux, double* pcflux, int nlambda, int *pfiltind)
{
  double		B;
  int			i, l;
  double		DeltaD=NORMA(obsv);
  VECTOR		obsN={obsv.x/DeltaD, obsv.y/DeltaD, obsv.z/DeltaD};
  double		DeltaQAU = DeltaD*DeltaD*AU*AU;
  double		*flc;
  double		ps;
  double                ccfactor;


  if (nlambda > 5000) return -1;

  memset(pflux, 0, sizeof(double)*nlambda);
  memset(pcflux, 0, sizeof(double)*nlambda);

  for (l=0;l<nlambda; l++)
    {
      for (i=0;i<ptpm->mesh.fn;i++)
	{ 
	  ps=SCALAR(ptpm->mesh.n[i].vec,obsN); 
	  if (ps <=0) continue;
	  
	  B=NIST_C1/
	    plambda[l]/plambda[l]/plambda[l]/plambda[l]/plambda[l]/
	    (exp(NIST_C2/(plambda[l]*ptpm->pTemp[i]))-1)/PI; //in steradians (divided by pi)
	  if ((pfiltind[l] >= 0) && (ptpm->pCCorr))
	   {
	     //printf("CC> calcFlux(): correcting facet %d W%d flux with T=%lf for lambda =%lf ",i,pfiltind[l]+1,ptpm->pTemp[i],plambda[l]);
	     //there is a color correction to perform:
	     //we round the float by casting
	     if((int)ptpm->pTemp[i] > 999) printf("Temperature too high!\n");
	     
	     //we extract the color corr. factor corresponding to this filter and this T
	     ccfactor = ptpm->pCCorr->CCTable[pfiltind[l]][(int)ptpm->pTemp[i]];

	     pflux[l]+=((B*ptpm->epsilon*ps * ptpm->mesh.n[i].mod))*ccfactor;  // watch the factor 2 (LENGHT of the NORMAL)
	    }
	  else
	    {
	      //printf("No color correction performed\n");
	      //there is no color correction to perform
	      pflux[l]+=((B*ptpm->epsilon*ps * ptpm->mesh.n[i].mod));  
	      // watch the factor 2 (LENGHT of the NORMAL)
	    }
	}

      pflux[l] /=2;
      pflux[l] /= (DeltaQAU);
      pflux[l] *= (plambda[l]*plambda[l]*0.33357e12);			  //Jy
    }
  return 1; 
}


/* ------------------------------------------------------------
   This function computes the flux from the object and its derivatives 
wrt the parameters
-------------------------------------------------------------*/
int CalcFluxDerivatives(TPM* ptpm, OBS* pobs)
{
  double B,r_loc,Delta_loc,alfa_loc;
  int    idata, ifacet,icoef;
  double DeltaD=NORMA(pobs->vo);
  double S=NORMA(pobs->vs); // distance to the fucking Sun
  VECTOR obsN={pobs->vo.x/DeltaD, pobs->vo.y/DeltaD, pobs->vo.z/DeltaD};
  double DeltaQAU = DeltaD*DeltaD*AU*AU;
  double *flc;
  double ccfactor=1.0;
  double facetflux;
  double lambda5;
  double dfdt;
  double  omega=D2PI/(ptpm->Tsid), sqrtomega=sqrt(omega);

  S=SOLARC*ptpm->SunLuminosity/(S*S); // Solar constant at the asteroid

  memset(pobs->ModelFlux,     0, sizeof(double)*pobs->Ndata);
  memset(pobs->ReflectedFlux, 0, sizeof(double)*pobs->Ndata);

  for (idata=0;idata<pobs->Ndata; idata++)
    for (icoef=0;icoef<5;icoef++)
      pobs->dyda[idata][icoef] = 0.0;
  
  if (ptpm->crat) memset(pobs->CraterFlux, 0, sizeof(double)*pobs->Ndata);
  
  /* 
     If the H value is given as input and the Rp is greater than 0, 
     add the reflected light scaled by the value at 3.35 um
  */
  if((ptpm->H > 0.0) && (ptpm->Rp > 0.0))
    { 
      //we calculate the reflected light component from H,r,Delta,alfa 	 
      calcGeometry(pobs->vs,pobs->vo,&r_loc,&Delta_loc,&alfa_loc);
      //printf("iobs = %d; r = %lf, Delta = %lf, alfa = %lf Rp = %lf\n",
      //     idata,r_loc,Delta_loc,alfa_loc,ptpm->Rp);
      AddReflComp_Jy(pobs->lambda,pobs->ReflectedFlux,pobs->Ndata,
		     ptpm->H,ptpm->G,r_loc,Delta_loc,alfa_loc,ptpm->Rp);
    }
  else
    printf("Warning: no input values for H and/or pIR/pV. No reflec. light component\n");
  
  // calculate the visibility of each facet from the observer
  // the ptpm->mesh.vis[ifacet] contains the visbility of each facet from the
  // observer taking into account of blockers 
  //  ptpm->mesh.vis[ifacet] = SCALAR(normal to the facet, versor to the obs)*blockers
  meshvisibility(&ptpm->mesh, pobs->vo);
  
  for (idata=0;idata<pobs->Ndata; idata++)
    {
      lambda5=pobs->lambda[idata]*pobs->lambda[idata]*pobs->lambda[idata]*
	pobs->lambda[idata]*pobs->lambda[idata];
      
      for (ifacet=0;ifacet<ptpm->mesh.fn;ifacet++)
	{ 
	  // if the facet is not visbile or have zero temp, skip it
	  if ((ptpm->mesh.vis[ifacet]<=0) || (ptpm->pTemp[ifacet]<=0)) continue;  
	  
	  B=NIST_C1/lambda5/
	    (exp(NIST_C2/(pobs->lambda[idata]*ptpm->pTemp[ifacet]))-1)/PI; //in steradians (divided by pi)

	  if ((pobs->FilterIndex[idata] >= 0) && (ptpm->pCCorr))
	    {
	      //there is a color correction to perform:
	      //we round the float by casting
	      if((int)ptpm->pTemp[ifacet] > 999){ 
		printf("Warning: facet Temperature too high!\n");
		ptpm->pTemp[ifacet] = 999;
	      }

	      if((int)ptpm->pTemp[ifacet] >= 0)
		ccfactor = 
		  ptpm->pCCorr->CCTable[pobs->FilterIndex[idata]][(int)ptpm->pTemp[ifacet]];
	          //we extract the color corr. factor corresponding to this filter and this T
	      else{
		printf("Warning: Temperature<0; ccfactor =1\n");
		ccfactor=1;
	      }

	      // watch the factor 2 (LENGHT of the NORMAL)
	      facetflux=(B*ptpm->mesh.vis[ifacet]*
			 ptpm->mesh.n[ifacet].mod)*ccfactor;
	      pobs->ModelFlux[idata]+=facetflux;

	      // derivative of the reduced FacetFlux wrt the FacetTemperature
	      dfdt=facetflux*(B*ccfactor)*exp(NIST_C2/(pobs->lambda[idata]*ptpm->pTemp[ifacet]))
		/(ptpm->pTemp[ifacet]*ptpm->pTemp[ifacet]);
	    }
	  else
	    {
	      //there is no color correction to perform
	      // watch the factor 2 (LENGHT of the NORMAL)
	      facetflux=(B*ptpm->mesh.vis[ifacet]*
			 ptpm->mesh.n[ifacet].mod);
	      pobs->ModelFlux[idata]+=facetflux;
	      
	      // derivative of the reduced FacetFlux wrt the FacetTemperature
	      dfdt=facetflux*B*exp(NIST_C2/(pobs->lambda[idata]*ptpm->pTemp[ifacet]))
		/(ptpm->pTemp[ifacet]*ptpm->pTemp[ifacet]);
	    }

	  // derivative of the reduced term wrt the the beaming parameter
	  pobs->dyda[idata][1]-=dfdt*ptpm->pTemp[ifacet]/(4*ptpm->BeamingParameter);
	  // derivative with respect to the log of the beaming
	  //pobs->dyda[idata][1]-=dfdt*ptpm->pTemp[ifacet]/4*log(ptpm->BeamingParameter);
	      	      
	  // derivative of the reduced term wrt the the absorption (1-A)
	  pobs->dyda[idata][2]+=dfdt*S*ptpm->mesh.ill[ifacet]
	    /(4*ptpm->BeamingParameter*ptpm->epsilon*SIGMA*
	      ptpm->pTemp[ifacet]*ptpm->pTemp[ifacet]*ptpm->pTemp[ifacet]);
	  
	  // derivative of the reduced term wrt the value of the Thermal Inertia
	  if(ptpm->Gamma>0)
	    pobs->dyda[idata][3]+=dfdt*sqrtomega*(ptpm->pTB[ifacet][1]-ptpm->pTB[ifacet][0])
	      /ptpm->dx
	      /(4*ptpm->BeamingParameter*ptpm->epsilon*SIGMA*
		ptpm->pTemp[ifacet]*ptpm->pTemp[ifacet]*ptpm->pTemp[ifacet]);
	}
      
      pobs->ModelFlux[idata] *= (pobs->lambda[idata]*pobs->lambda[idata]*0.33357e12) // W/m-2/um -> Jy
	*ptpm->epsilon // emissivity
	/2/DeltaQAU    // factor 2 due to the fact that ptpm->mesh.n[ifacet].mod is twice the are of the facet
	*ptpm->D2;     // the Diameter^2 that is varied by the LM fit
      
      pobs->dyda[idata][0]=pobs->ModelFlux[idata]/ptpm->D2; // derivative of the flux with respect to the D^2
      //pobs->dyda[idata][0]=pobs->ModelFlux[idata]*log(ptpm->D2);    // derivative of the flux with respect to the log(D^2)

      pobs->dyda[idata][1]*=ptpm->epsilon*lambda5*pobs->lambda[idata]*ptpm->D2*0.33357e12
	/(2*DeltaQAU)*NIST_C2*PI/NIST_C1;
      pobs->dyda[idata][2]*=ptpm->epsilon*lambda5*pobs->lambda[idata]*ptpm->D2*0.33357e12
	/(2*DeltaQAU)*NIST_C2*PI/NIST_C1;
      pobs->dyda[idata][3]*=ptpm->epsilon*lambda5*pobs->lambda[idata]*ptpm->D2*0.33357e12
	/(2*DeltaQAU)*NIST_C2*PI/NIST_C1;
     
      pobs->dyda[idata][4]=pobs->ReflectedFlux[idata]/ptpm->Rp;
      //pobs->dyda[idata][4]=pobs->ReflectedFlux[idata]*log(ptpm->Rp);   // derivative of the flux with respect to the log(Rp)
    }
  return 1; 
}


//compliant with IDL planck_W_um     20 JUL 02
int calcFluxCrat(TPM* ptpm, VECTOR obsv, double* plambda, double* pflux, double* pcflux, int nlambda, 
		 int       *pfiltind,
		 ColorCorrection *pCCorr)
{
	double		B;
	int			i, l;
	double		DeltaD=NORMA(obsv);
	VECTOR		obsN={obsv.x/DeltaD, obsv.y/DeltaD, obsv.z/DeltaD};
	double		DeltaQAU = DeltaD*DeltaD*AU*AU;
	double		*flc;
	double		ps;
	int             k;


	if (nlambda > 5000) return -1;

	memset(pflux, 0, sizeof(double)*nlambda);
	memset(pcflux, 0, sizeof(double)*nlambda);
	flc=malloc(sizeof(double)*nlambda);

	for (i=0;i<ptpm->mesh.fn;i++)	// loops on facets
	{
		ps=SCALAR(ptpm->mesh.n[i].vec,obsN); 
		if (ps <=0) continue;

		getcraterflux(&ptpm->crat[i], obsN, plambda, flc, nlambda, ptpm->LUT, pfiltind, pCCorr);
		//		printf("bT> %ld %lf %lf %lf ",i,ptpm->pTemp[i], ps, flc[0]);
		//		for (k=0;k<ptpm->crat[i].Nt;k++) printf("%4d",(int)ptpm->crat[i].pT[k]);
		//		printf("\n");
		for (l=0;l<nlambda; l++)
		{
		  //		  printf("bT> %ld %lf -- %lf, %lf ",i,plambda[l],ptpm->pTemp[i],flc[l]);
		  B=NIST_C1/
		    plambda[l]/plambda[l]/plambda[l]/plambda[l]/plambda[l]/
		    (exp(NIST_C2/(plambda[l]*ptpm->pTemp[i]))-1)/PI; //in steradians (diveded by pi)
		  pflux[l]+=(B * ptpm->epsilon
			     * ps
			     * (1.0-ptpm->density)
			     + flc[l]*ptpm->density
			     / ptpm->crat[i].projArea)
		    * ptpm->mesh.n[i].mod;
		  pcflux[l]+= flc[l] / ptpm->crat[i].projArea * ptpm->mesh.n[i].mod;
		}
	}

	for (l=0;l<nlambda; l++)
	{
		pflux[l] /=2;
		pflux[l] /= (DeltaQAU);
		pflux[l] *= (plambda[l]*plambda[l]*0.33357e12);			  //Jy	
		pcflux[l] /=2;
		pcflux[l] /= (DeltaQAU);
		pcflux[l] *= (plambda[l]*plambda[l]*0.33357e12);		  //Jy	
	}
	free(flc);
	return 1; 
}

/* flux from a rough surface taking into accout the heat transfer
uses the Lagerros approx */
int CalcFluxCratLagerrosCAZZO(TPM* ptpm, VECTOR obsv, VECTOR sun, double* plambda, 
			 double* pflux, double* pcflux, int nlambda, 
			 int       *pfiltind,
			 ColorCorrection *pCCorr)

{
  double		B;
  int			i, ilambda, k;
  double		DeltaD=NORMA(obsv);
  VECTOR		obsN={obsv.x/DeltaD, obsv.y/DeltaD, obsv.z/DeltaD};
  double		DeltaQAU = DeltaD*DeltaD*AU*AU;
  double		HelioD=NORMA(sun);
  VECTOR		sunN={sun.x/HelioD, sun.y/HelioD, sun.z/HelioD};
  double		*flc;
  double		mu_obs, mu_sun;
  double		TSS;
  double                facetFluxNoGamma;
  double		facetBriTemp;
  double		TnoGamma;
  double		Tb_rough;
  double                lambda5;
  double                ccfactor=1.0;

  if (nlambda > 5000) return -1;
  
  memset(pflux, 0, sizeof(double)*nlambda);
  memset(pcflux, 0, sizeof(double)*nlambda);
  flc=malloc(sizeof(double)*nlambda);
  
  TSS=562.341325*sqrt(sqrt((1-ptpm->A)*SOLARCONSTANT*ptpm->SunLuminosity/ptpm->epsilon
			   /STEFBOLTZ/ptpm->BeamingParameter   // 562.341325 is the fourth root of 1e11 (unit conversion)
			   )/ HelioD);
  
  for (i=0;i<ptpm->mesh.fn;i++)	// loops on facets
    {
      // reset the flux from crater vector to zero. There is one element per wavelength
      memset(flc,0,sizeof(double)*nlambda);
      
      // calculate the direction cosine of the facet to the observer (mu_obs>0 the facet is visible)
      if ( (mu_obs=SCALAR(ptpm->mesh.n[i].vec,obsN))>0)
	{
	  // MDB 2012-11-06 Nice
	  // calculate the direction cosine of the facet to the SUN (mu_sun>0 the facet is illuminated)
	  // if the facet is visible and illuminated THEN we can use the Lagerros approximation
	  //==================================== FACET IS ILLUMINATED AND VISIBLE ====================
	  if ( (mu_sun=SCALAR(ptpm->mesh.n[i].vec,sunN))>0)
	    {
	      setcratertemp(&ptpm->crat[i], sunN, HelioD,  ptpm->LUT);
	      TnoGamma=(TSS*pow(mu_sun,0.25));
	      //	      printf("FR> %ld %lf %lf \n",i,ptpm->pTemp[i]/TnoGamma, acos(mu_sun)*180/3.1416);
	      /* Lagerros correction on the physical temperatures inside the crater */
	      for (k=0;k<ptpm->crat[i].Nt;k++) ptpm->crat[i].pT[k]*=(ptpm->pTemp[i]/TnoGamma);
	      // ******************************************************************************

	      // MDB and Victor 2013 march equinox: we do not color correct the flux from the crater for now
	      //	      	      GetCraterFluxLagerros(&ptpm->crat[i], obsN, plambda, flc, nlambda, ptpm->LUT, 
	      //				    pfiltind, pCCorr);

	      getcraterflux(&ptpm->crat[i], obsN, plambda, flc, nlambda, ptpm->LUT, 
		      		    pfiltind, NULL);
	      
		      //		      printf("bT> %ld %lf %lf %lf ",i,ptpm->pTemp[i], mu_obs, flc[0]);
		      //		      for (k=0;k<ptpm->crat[i].Nt;k++) printf("%4d",(int)ptpm->crat[i].pT[k]);
		      //		      printf("\n");
	      
	      for (ilambda=0;ilambda<nlambda; ilambda++) //loop on lambda 
		{
		  /*		  
		  // Lagerros correction on the brigthness temperatures corrected for thermal inertia
		  CraterBriTemp=NIST_C2/(plambda[ilambda]*log(1.0+NIST_C1*ptpm->epsilon/flc[ilambda]/PI/lambda5));
		  //printf("bT> %ld %lf -- %lf, %lf, %lf %lf :: %lf\n",i,plambda[ilambda],ptpm->pTemp[i],CraterBriTemp,TGamma_TnoGamma,flc[ilambda],mu_sun);
		  Tb_rough=TGamma_TnoGamma*CraterBriTemp;
		  //  MDB and Victor 2013 march equinox:
		  // If there is a filter index and a color correction pointer
		  ccfactor=1.0;
		  if ((pfiltind[ilambda] >= 0) && (ptpm->pCCorr))
		  {
		    if((int)Tb_rough > 999) 
		      printf("Warning: Temperature too high!\n");		  
		    else
		      ccfactor = ptpm->pCCorr->CCTable[pfiltind[ilambda]][(int)Tb_rough];
		  }
		  //		  flc[ilambda]=NIST_C1/lambda5/(exp(NIST_C2/(plambda[ilambda]*Tb_rough))-1)/PI*ptpm->epsilon*ccfactor;
		  */		  		  
		  // ******************************************************************************

		  B=NIST_C1/
		    plambda[ilambda]/plambda[ilambda]/plambda[ilambda]/plambda[ilambda]/plambda[ilambda]/
		    (exp(NIST_C2/(plambda[ilambda]*ptpm->pTemp[i]))-1)/PI; //in steradians (diveded by pi)
		  pflux[ilambda]+=(B * ptpm->epsilon
			           * mu_obs
			           * (1.0-ptpm->density)
			           + flc[ilambda]*ptpm->density
			           / ptpm->crat[i].projArea)
		    * ptpm->mesh.n[i].mod;
		  pcflux[ilambda]+=flc[ilambda] / ptpm->crat[i].projArea * ptpm->mesh.n[i].mod;
		}//loop on lambda
	    }	 // if  (mu_sun>0) i.e. facet is illuminated	  
	  //==================================== FACET IS NOT ILLUMINATED BUT STILL VISIBLE ====================
	  // MDB 2012-11-06 Nice
	  // HOWEVER, with non negligible thermal inertia a facet can irradiate heat even when non illiminated. 
	  // In this case, the Lagerros approximation is no longer valid, 
	  // but we need to get the flux from that facet.
	  // We use the planar approximation i.e. NO CRATER
	  // Victor20130319: color correction here
	  else { // if  (mu_sun>0) i.e. facet is illuminated
	    for (ilambda=0;ilambda<nlambda; ilambda++) //loop on lambda 
	      {

		// If there is a filter index and a color correction pointer
		ccfactor=1.0;
		if ((pfiltind[ilambda] >= 0) && (ptpm->pCCorr))
		  {
		    if((int)ptpm->pTemp[i] > 999) 
		      printf("Warning: Temperature too high!\n");		  
		    else
		      ccfactor = ptpm->pCCorr->CCTable[pfiltind[ilambda]][(int)ptpm->pTemp[i]];
		  }

		lambda5=plambda[ilambda]*plambda[ilambda]*plambda[ilambda]*plambda[ilambda]*plambda[ilambda];
		B=NIST_C1/lambda5/
		  (exp(NIST_C2/(plambda[ilambda]*ptpm->pTemp[i]))-1)/PI; //in steradians (diveded by pi)
		pflux[ilambda]+=B * ptpm->epsilon * mu_obs  * ptpm->mesh.n[i].mod*ccfactor;
	      }//loop on lambda	    
	  } 
	  
	}// if  (mu_obs>0) i.e. facet is visible
    } // facet loop

  for (ilambda=0;ilambda<nlambda; ilambda++)
    {
      pflux[ilambda] /=2;
      pflux[ilambda] /= (DeltaQAU);
      pflux[ilambda] *= (plambda[ilambda]*plambda[ilambda]*0.33357e12);		//Jy	
      pcflux[ilambda] /=2;
      pcflux[ilambda] /= (DeltaQAU);
      pcflux[ilambda] *= (plambda[ilambda]*plambda[ilambda]*0.33357e12);		//Jy	
    }

  free(flc);
  return 1; 
}

int CalcFluxCratLagerros(TPM* ptpm, VECTOR obsv, VECTOR sun, double* plambda, 
			 double* pflux, double* pcflux, int nlambda, 
			 int       *pfiltind,
			 ColorCorrection *pCCorr)

{
  double		B;
  int			i, ilambda, k;
  double		DeltaD=NORMA(obsv);
  VECTOR		obsN={obsv.x/DeltaD, obsv.y/DeltaD, obsv.z/DeltaD};
  double		DeltaQAU = DeltaD*DeltaD*AU*AU;
  double		HelioD=NORMA(sun);
  VECTOR		sunN={sun.x/HelioD, sun.y/HelioD, sun.z/HelioD};
  double		*flc;
  double		mu_obs, mu_sun;
  double		TSS;
  double                facetFluxNoGamma;
  double		facetBriTemp;
  double		TnoGamma;
  double		Tb_rough;
  double                lambda5;
  double                ccfactor=1.0;

  if (nlambda > 5000) return -1;
  
  memset(pflux, 0, sizeof(double)*nlambda);
  memset(pcflux, 0, sizeof(double)*nlambda);
  flc=malloc(sizeof(double)*nlambda);
  
  TSS=562.341325*sqrt(sqrt((1-ptpm->A)*SOLARCONSTANT*ptpm->SunLuminosity/ptpm->epsilon
			   /STEFBOLTZ/ptpm->BeamingParameter   // 562.341325 is the fourth root of 1e11 (unit conversion)
			   )/ HelioD);
  
  for (i=0;i<ptpm->mesh.fn;i++)	// loops on facets
    {
      // reset the flux from crater vector to zero. There is one element per wavelength
      memset(flc,0,sizeof(double)*nlambda);
      
      // calculate the direction cosine of the facet to the observer (mu_obs>0 the facet is visible)
      if ( (mu_obs=SCALAR(ptpm->mesh.n[i].vec,obsN))>0)
	{
	  // MDB 2012-11-06 Nice
	  // calculate the direction cosine of the facet to the SUN (mu_sun>0 the facet is illuminated)
	  // if the facet is visible and illuminated THEN we can use the Lagerros approximation
	  //==================================== FACET IS ILLUMINATED AND VISIBLE ====================
	  if ( (mu_sun=SCALAR(ptpm->mesh.n[i].vec,sunN))>0)
	    {
	      SetCraterTempLagerros(&ptpm->crat[i], sunN, HelioD,  ptpm->LUT);
	      TnoGamma=(TSS*pow(mu_sun,0.25));
	      //	      printf("FR> %ld %lf %lf \n",i,ptpm->pTemp[i]/TnoGamma, acos(mu_sun)*180/3.1416);
	      /* Lagerros correction on the physical temperatures inside the crater */
	      //	      for (k=0;k<ptpm->crat[i].Nt;k++) ptpm->crat[i].pT[k]*=(ptpm->pTemp[i]/TnoGamma);
	      // ******************************************************************************

	      // MDB and Victor 2013 march equinox: we do not color correct the flux from the crater for now
	      //	      	      GetCraterFluxLagerros(&ptpm->crat[i], obsN, plambda, flc, nlambda, ptpm->LUT, 
	      //				    pfiltind, pCCorr);

	      GetCraterFluxLagerros(&ptpm->crat[i], obsN, plambda, flc, nlambda, ptpm->LUT, 
	           		    pfiltind, NULL);
	      
	      for (ilambda=0;ilambda<nlambda; ilambda++) //loop on lambda 
		{
		  
		  lambda5=plambda[ilambda]*plambda[ilambda]*plambda[ilambda]*plambda[ilambda]*plambda[ilambda];
		  /*		  
		  // Lagerros correction on the brigthness temperatures corrected for thermal inertia
		  CraterBriTemp=NIST_C2/(plambda[ilambda]*log(1.0+NIST_C1*ptpm->epsilon/flc[ilambda]/PI/lambda5));
		  //printf("bT> %ld %lf -- %lf, %lf, %lf %lf :: %lf\n",i,plambda[ilambda],ptpm->pTemp[i],CraterBriTemp,TGamma_TnoGamma,flc[ilambda],mu_sun);
		  Tb_rough=TGamma_TnoGamma*CraterBriTemp;
		  //  MDB and Victor 2013 march equinox:
		  // If there is a filter index and a color correction pointer
		  ccfactor=1.0;
		  if ((pfiltind[ilambda] >= 0) && (ptpm->pCCorr))
		  {
		    if((int)Tb_rough > 999) 
		      printf("Warning: Temperature too high!\n");		  
		    else
		      ccfactor = ptpm->pCCorr->CCTable[pfiltind[ilambda]][(int)Tb_rough];
		  }
		  //		  flc[ilambda]=NIST_C1/lambda5/(exp(NIST_C2/(plambda[ilambda]*Tb_rough))-1)/PI*ptpm->epsilon*ccfactor;
		  */		  		  
		  // ******************************************************************************

		  B=NIST_C1/lambda5/
		    (exp(NIST_C2/(plambda[ilambda]*TnoGamma))-1)/PI; //in steradians (diveded by pi) FLUX From facet with no Gamma
 		  facetFluxNoGamma=(B  
				    * ptpm->epsilon
				    * mu_obs
				    * (1.0-ptpm->density)			       
				    + flc[ilambda]*ptpm->density	   				    
		                    / ptpm->crat[i].projArea);
		  
		  facetBriTemp=NIST_C2/(plambda[ilambda]*log(1.0+NIST_C1*ptpm->epsilon*mu_obs/facetFluxNoGamma/PI/lambda5));
		  //		  printf("F> %d %lf %lf %lf \n", i, facetBriTemp, ptpm->pTemp[i], TnoGamma);
		  // we correct the facetBrightness Temperature because of heat conduction		  
		  //		  if ((ptpm->pTemp[i]/TnoGamma)<1) 
		  facetBriTemp*=(ptpm->pTemp[i]/TnoGamma);
		  // If there is a filter index and a color correction pointer

		  ccfactor=1.0;
		  if ((pfiltind[ilambda] >= 0) && (ptpm->pCCorr))
		    {
		      if((int)facetBriTemp > 999) 
			printf("Warning: Temperature too high!\n");		  
		      else
			ccfactor = ptpm->pCCorr->CCTable[pfiltind[ilambda]][(int)facetBriTemp];
		    }		  

		  pflux[ilambda]+=NIST_C1/lambda5/
		    (exp(NIST_C2/(plambda[ilambda]*facetBriTemp))-1)/PI*ptpm->epsilon*mu_obs*
		    ptpm->mesh.n[i].mod*ccfactor; //in steradians (diveded by pi) FLUX From facet with no Gamma

		  pcflux[ilambda]+=flc[ilambda] / ptpm->crat[i].projArea * ptpm->mesh.n[i].mod;
		}//loop on lambda
	    }	 // if  (mu_sun>0) i.e. facet is illuminated	  
	  //==================================== FACET IS NOT ILLUMINATED BUT STILL VISIBLE ====================
	  // MDB 2012-11-06 Nice
	  // HOWEVER, with non negligible thermal inertia a facet can irradiate heat even when non illiminated. 
	  // In this case, the Lagerros approximation is no longer valid, 
	  // but we need to get the flux from that facet.
	  // We use the planar approximation i.e. NO CRATER
	  // Victor20130319: color correction here
	  else { // if  (mu_sun>0) i.e. facet is illuminated
	    for (ilambda=0;ilambda<nlambda; ilambda++) //loop on lambda 
	      {

		// If there is a filter index and a color correction pointer
		ccfactor=1.0;
		if ((pfiltind[ilambda] >= 0) && (ptpm->pCCorr))
		  {
		    if((int)ptpm->pTemp[i] > 999) 
		      printf("Warning: Temperature too high!\n");		  
		    else
		      ccfactor = ptpm->pCCorr->CCTable[pfiltind[ilambda]][(int)ptpm->pTemp[i]];
		  }

		lambda5=plambda[ilambda]*plambda[ilambda]*plambda[ilambda]*plambda[ilambda]*plambda[ilambda];
		B=NIST_C1/lambda5/
		  (exp(NIST_C2/(plambda[ilambda]*ptpm->pTemp[i]))-1)/PI; //in steradians (diveded by pi)
		pflux[ilambda]+=B * ptpm->epsilon * mu_obs  * ptpm->mesh.n[i].mod*ccfactor;
	      }//loop on lambda	    
	  } 
	  
	}// if  (mu_obs>0) i.e. facet is visible
    } // facet loop

  for (ilambda=0;ilambda<nlambda; ilambda++)
    {
      pflux[ilambda] /=2;
      pflux[ilambda] /= (DeltaQAU);
      pflux[ilambda] *= (plambda[ilambda]*plambda[ilambda]*0.33357e12);		//Jy	
      pcflux[ilambda] /=2;
      pcflux[ilambda] /= (DeltaQAU);
      pcflux[ilambda] *= (plambda[ilambda]*plambda[ilambda]*0.33357e12);		//Jy	
    }

  free(flc);
  return 1; 
}


// this function performs a full rotation of the mesh and calculate the average 
// temperature of each facet on the surface
int inittemps(TPM* ptpm, VECTOR sun0)
{
  long Nstep=360;//one step per degree
  long i,j;
  double *paveT=NULL;
  VECTOR sun;
  double theta=0;

  printf("resetting temperatures for %d facets\n",ptpm->mesh.fn);
  memcpy(&sun, &sun0, sizeof(VECTOR));
  paveT=malloc(sizeof(double)*ptpm->mesh.fn);
  if (!paveT)
    {
      printf("inittemps E1: alloc error\n");
      return -1;
    }
  memset(paveT,0,sizeof(double)*ptpm->mesh.fn);

  for(i=0; i<Nstep; i++)
    {
      calcTemp(ptpm,sun);
      //      for (j=0;j<ptpm->mesh.fn;j++) printf("%d %lf %lf\n",i,NORMA(sun0),ptpm->pTemp[j]);
      for(j=0;j<ptpm->mesh.fn;j++)
	paveT[j]+=ptpm->pTemp[j];

      theta=D2PI*i/Nstep;// angle of rotation

      // we rotate the frame (the asteroid rotates)
      // http://mathworld.wolfram.com/RotationMatrix.html eq (3)
      sun.x=  cos(theta)*sun0.x + sin(theta)*sun0.y;
      sun.y= -sin(theta)*sun0.x + cos(theta)*sun0.y;
    }

  for(j=0; j<ptpm->mesh.fn; j++)
    {

      paveT[j]/=Nstep;
      //      printf("Fat> %ld T=%lf\n",j,paveT[j]);
    }

  // it checks what is allocated
  if (ptpm->Gamma > 0) // if there is thermal inertia
    {
      if (ptpm->pTB)// thermal inertia buffer for heat diffusion is allocated
	{
	  for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
	    for(j=0;j<ptpm->NTslabs; j++)
	      ptpm->pTB[i][j]=paveT[i];
	}

      if (ptpm->crat)// craters are present thermal inertia buffer for heat diffusion is allocated
	{
	  for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
	    resetcratertemp(&ptpm->crat[i], paveT[i]);
 	}
    }
  
  free(paveT);
}

// this function performs a full rotation of the mesh and calculate the average 
// temperature of each facet on the surface
int MeanInsolationTemps(TPM* ptpm, VECTOR *sun, double* pjd, unsigned long Npts) // April 22, 2010: Nice
{
  long i,j, i0=-1, i1=-1, nn=0;
  double *paveT=NULL;
  double theta=0;
  double HelioD, theconst;

  printf("Avaraging insolation.. over %lf - %lf\n", ptpm->JD0, ptpm->JD1);

  // finds the time interval indexes
  for(i=0; i<Npts; i++)
    if (pjd[i]>=ptpm->JD0) break;
  if ((i<0) && (i>Npts))
    {
      printf("MeanInsolationTemps: JD0 before init of ephemerides\n");
      return -1;
    }
  i0=i;// set the initial index

  for(i=0; i<Npts; i++)
    if (pjd[i]>=ptpm->JD1) break;
  if ((i<0) && (i>Npts))
    {
      printf("MeanInsolationTemps: JD1 before init of ephemerides\n");
      return -2;
    }
  i1=i;// set the final index
  nn=i1-i0;
  printf("Avaraging insolation: period index are %ld - %ld total points %ld\n", i0, i1, nn);
  printf("Avaraging insolation: emiss=%lf, eta=%lf SunLum=%lf A=%lf\n", 
	 ptpm->epsilon, ptpm->BeamingParameter, ptpm->SunLuminosity, ptpm->A);
     
  paveT=malloc(sizeof(double)*ptpm->mesh.fn);
  if (!paveT)
    {
      printf("inittemps E1: alloc error\n");
      return -3;
    }
  memset(paveT,0,sizeof(double)*ptpm->mesh.fn);
  /*
  for(i=i0; i<i1; i++)
    {
      calcTemp(ptpm,sun[i]);
      for(j=0;j<ptpm->mesh.fn;j++)
	paveT[j]+=ptpm->pTemp[j];
      if (i % 1000 == 0) 
	printf("completed %ld        \r", 100*(i-i0)/nn);
    }
  */

  for(i=i0; i<i1; i++)
    {
      if (meshillumination(&ptpm->mesh, sun[i])<=0)
	{
	  printf("calcTemp E02:  meshillumination Failed\n");
	  return -1;	// error
	}
      
      HelioD=NORMA(sun[i]);
      theconst=(1.0-ptpm->A)*SOLARC*ptpm->SunLuminosity/HelioD/HelioD;
      //      printf("%lf\t%lf\t\t%lf\t%lf\t%lf\n", HelioD, theconst, sun[i].x, sun[i].y, sun[i].z);
      for (j=0;j<ptpm->mesh.fn;j++)
	paveT[j]+=(theconst*ptpm->mesh.ill[j]);
      //      for (j=0;j<ptpm->mesh.fn;j++) printf("%lf\t",ptpm->mesh.ill[j]); printf("\n");      
    }
 
  printf("\nAvaraging insolation: now storing temperatures\n");

  for(j=0; j<ptpm->mesh.fn; j++)
      paveT[j]/=nn;

  for(j=0; j<ptpm->mesh.fn; j++)
    paveT[j]=sqrt(sqrt(paveT[j]/SIGMA/ptpm->BeamingParameter/ptpm->epsilon));

  printf("Herebelow are facets' average temperatures: facet index within ()\n");
  printf("theconst=%lf\n", theconst);
  for(j=0; j<ptpm->mesh.fn; j++)
    printf("(%4ld) %6.1lf\t", j, paveT[j]);

  // it checks what is allocated
  if (ptpm->Gamma > 0) // if there is thermal inertia
    {
      if (ptpm->pTB)// thermal inertia buffer for heat diffusion is allocated
	{
	  for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
	    for(j=0;j<ptpm->NTslabs; j++)
	      ptpm->pTB[i][j]=paveT[i];
	}

      if (ptpm->crat)// craters are present thermal inertia buffer for heat diffusion is allocated
	{
	  for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
	    resetcratertemp(&ptpm->crat[i], paveT[i]);
 	}
    }
  
  free(paveT);
  printf("\nAvaraging insolation completed\n");
}

int PrintSubSolarSubObserver(OBS* pobs)
{
  VECTOR vSunN, v;
  double lambda, beta, rxy;

  normalize(pobs->vs, &v); 
  memcpy(&vSunN, &v, sizeof(VECTOR));
  rxy=sqrt(v.x*v.x + v.y*v.y); lambda=atan2(v.y, v.x)*RADEG;
  if (lambda<0) lambda+=360.0;
  if (rxy!=0)
    beta=atan(v.z/rxy)*RADEG;
  else 
    { 
      if (v.z>0) beta=90.0;
      else beta=-90.0;
    }
  printf("ss> %18.10lf %6.1lf %6.1lf %14.10lf\t", pobs->JD, lambda, beta, NORMA(pobs->vs));
	  
  normalize(pobs->vo, &v);
  rxy=sqrt(v.x*v.x + v.y*v.y); lambda=atan2(v.y, v.x)*RADEG;
  if (lambda<0) lambda+=360.0;
  if (rxy!=0)
    beta=atan(v.z/rxy)*RADEG;
  else 
    { 
      if (v.z>0) beta=90.0;
      else beta=-90.0;
    }
  printf("so> %6.1lf %6.1lf %14.10lf %14.10lf\n", 
	 lambda, beta, NORMA(pobs->vo), RADEG*acos(SCALAR(v,vSunN)));
  return 0;
}

int GetVecLongLatAsteroid(VECTOR *pVec, double *longitude, double *latitude)
{
  VECTOR v;
  double lambda, beta, rxy;

  normalize(pVec[0], &v);
  rxy=sqrt(v.x*v.x + v.y*v.y); longitude[0]=atan2(v.y, v.x)*RADEG;
  if (longitude[0]<0) longitude[0]+=360.0;
  if (rxy!=0)
    latitude[0]=atan(v.z/rxy)*RADEG;
  else 
    { 
      if (v.z>0) latitude[0]=90.0;
      else latitude[0]=-90.0;
    }
  //  printf("ss> %6.1lf %6.1lf\t", lambda, beta);

  return 0;
}


int TPM_noTI_noCrat_Flux(TPM* ptpm, 
	   EPH* peph, long Neph, 
           OBS* pobs, long Nobs)
{
  double		jds=0, jdsold=0;
  unsigned long		i, j, indf;
  time_t  		init, etime;
  double		*prat=NULL;
  FILE			*pfhtemp=NULL;
  FILE                  *pfhfflux=NULL;
  VECTOR                vSun; // vector to the Sun and the Observer.. 
  long                  ieph, iobs, idata, iStep, nbStep;
  OBS                   *pEphObs=NULL; //used just to simplify the reading
  double                miniStepDays;
  double                dtEph, t;
  double                *pMaxTemp=NULL;
  double                *pMinTemp=NULL;
	
// initialize the pointer to the temperature buffer
// if no temperature buffer, it allocates it
  if (!ptpm->pTemp)
    if ((ptpm->pTemp = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E03: could not allocate pTemp buffer\n");
	return -3;	// error
      }

// initialize the pointer of the illumination vector
// if no illumination vector, it allocates it
  if (!ptpm->mesh.ill)
    if ((ptpm->mesh.ill = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E04: could not allocate pTemp buffer\n");
	return -4;	// error
      }	  

// -------------------- RUN THE TPM HERE -------------------------------------------------
  init=time(NULL);
// START THE TPM: Gamma=0, Crat=0
  // run the case with no thermal inertia and no roughness
  // in this case there is no need to calculate the thermal history,
  // so fluxes are calculated only at the epoch of the observations
  printf("runTPM started with No thermal inertia, no Craters, and N=%ld\n", Nobs);

  for (iobs=0; iobs<Nobs; iobs++) // loops on observations
    {
      // rotate de vSun to the corotating asteroid frame
      Eclip2Ast(&pobs[iobs].vs, ptpm->mesh.lambda, ptpm->mesh.beta, 
		ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pobs[iobs].JD);
      // rotate de vObs to the corotating asteroid frame
      Eclip2Ast(&pobs[iobs].vo, ptpm->mesh.lambda, ptpm->mesh.beta, 
		ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pobs[iobs].JD);
      
      calcTemp(ptpm, pobs[iobs].vs);
      //      CalcFluxDerivatives(ptpm, &pobs[iobs]);
	     calcFlux(ptpm, pobs[iobs].vo, pobs[iobs].lambda, pobs[iobs].ModelFlux, pobs[iobs].CraterFlux, 
	       pobs[iobs].Ndata, pobs[iobs].FilterIndex);	  

	  //	      printf("%lf\n",ptpm->pTemp[0]);
      //      printf("f> iobs JD\t\t idata    wl\tobs.\t   sigmaobs.   model_flux  refl_flux    flux_r\n");
      for(idata=0; idata<pobs[iobs].Ndata; idata++)	
	printf("f> %03ld %18.10lf %03ld %6.2f %11.6f %10.6f %11.6f %11.6f %11.4f\n", iobs,
	       pobs[iobs].JD, idata, pobs[iobs].lambda[idata], pobs[iobs].flux[idata], 
	       pobs[iobs].eflux[idata], pobs[iobs].ModelFlux[idata], pobs[iobs].ReflectedFlux[idata], 
	       pobs[iobs].flux[idata]/pobs[iobs].ModelFlux[idata]);
      
      PrintSubSolarSubObserver(&pobs[iobs]);    
      
      // check if additional flags are present and do some actions accordingly
      // these needs observations to be present
      if ((pobs[iobs].flags & PRINTFACETFLUX)==PRINTFACETFLUX)
	PrintFacetFlux(ptpm, &pobs[iobs], &pfhtemp, 0);          // FLAGS are all zero as there are no craters
      if ((pobs[iobs].flags & DUMPFACETFLUX)==DUMPFACETFLUX)
	DumpFacetFlux(ptpm, &pobs[iobs], &pfhtemp, 0);	         // FLAGS are all zero as there are no craters

      // check if additional flags are present and do some actions accordingly
      if ((pobs[iobs].flags & DUMPTEMP)==DUMPTEMP)
	DumpTemp(ptpm, pobs[iobs].JD, &pfhtemp);
      if ((pobs[iobs].flags & PRINTTEMP)==PRINTTEMP)
	PrintTemp(ptpm, &pobs[iobs], &pfhtemp);      

    }

  printf("\nrunTPM completed\n");
  return 1;
}


int TPM_TI_noCrat_Flux(TPM* ptpm, 
	   EPH* peph, long Neph, 
           OBS* pobs, long Nobs)
{
  double		jds=0, jdsold=0;
  unsigned long		i, j, indf;
  time_t  		init, etime;
  double		*prat=NULL;
  FILE			*pfhtemp=NULL;
  FILE                  *pfhfflux=NULL;
  VECTOR                vSun, vSunTmp; // vector to the Sun and the Observer.. 
  long                  ieph, iobs, idata, iStep, nbStep;
  OBS                   *pEphObs=NULL; //used just to simplify the reading
  double                miniStepDays;
  double                dtEph, t;
  double                *pMaxTemp=NULL;
  double                *pMinTemp=NULL;
  double                dLongitude, dLatitude;
  int                   iEphLoop;
  double                deltaJDeph;
  double                maxTempDiff;
	
// initialize the pointer to the temperature buffer
// if no temperature buffer, it allocates it
  if (!ptpm->pTemp)
    if ((ptpm->pTemp = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E03: could not allocate pTemp buffer\n");
	return -3;	// error
      }

// initialize the pointer of the illumination vector
// if no illumination vector, it allocates it
  if (!ptpm->mesh.ill)
    if ((ptpm->mesh.ill = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E04: could not allocate pTemp buffer\n");
	return -4;	// error
      }	  

  // ------- INITIALIZE TEMPERATURES -------------------------------------------------------------
// It is the place here, where to set up the intial temperatures
// the idea is to set the deep temperature and all initial temperature to
// the value of the mean surface temperature. The average is taken on one rotation.
// In order to do this, we use a special procedure
  printf("initial temp %lf\n", ptpm->ti);
  if (ptpm->ti==-1.0)
    inittemps(ptpm, peph[0].vs);// we pass the thermal model and the initialposition of sun vector
  else if (ptpm->ti==-2.0) 
    LoadTemp(ptpm); // April 22, 2010: Nice
  else if (ptpm->ti==-3.0) 
    printf("TBD: MeanInsolation Has to be changed\n");
  //    MeanInsolationTemps(ptpm, psun, pjd, Npts); // April 22, 2010: Nice
  else
    {
      printf("resetting temperatures to %lf K \n", ptpm->ti);
      if (ptpm->Gamma > 0) // if there is thermal inertia
	{
	  if (ptpm->pTB)// thermal inertia buffer for heat diffusion is allocated
	    {
	      for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
		for(j=0;j<ptpm->NTslabs; j++)
		  ptpm->pTB[i][j]=ptpm->ti;
	    }

	  if (ptpm->crat)// craters are present thermal inertia buffer for heat diffusion is allocated
	    {
	      for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
		resetcratertemp(&ptpm->crat[i], ptpm->ti);
	    }
	}
    }

// RESET THE MAX and MIN TEMPs----------------------
//  MaxTemp(ptpm,&pMaxTemp);
//  MinTemp(ptpm,&pMinTemp);

// -------------------- RUN THE TPM HERE -------------------------------------------------
  init=time(NULL);
  printf("runTPM started with thermal inertia, no Craters, Neph=%ld dtMax=%lf(s)\n", Neph, ptpm->dtMAX);

  /*
  for (iEphLoop=0; iEphLoop<1000; iEphLoop++)
    {
      for (ieph=0; ieph<(Neph); ieph++) // loop on ephemerides
	peph[ieph].JD+=deltaJDeph;
      SaveTemp(ptpm);
  */
  for (ieph=0; ieph<(Neph-1); ieph++) // loop on ephemerides
    {
     // does the ephemeris have a linked observation?
      if (peph[ieph].pobs)
	{ 
	  pEphObs=peph[ieph].pobs; // get the pointer
	  // rotate de vObs to the corotating asteroid frame
	  Eclip2Ast(&pEphObs->vo, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pEphObs->JD);
	  calcFlux(ptpm, pEphObs->vo, pEphObs->lambda, pEphObs->ModelFlux, pEphObs->CraterFlux, 
		   pEphObs->Ndata,pEphObs->FilterIndex); //problem here?
	  //	      printf("%lf\n",ptpm->pTemp[0]);
	  
	  for(idata=0; idata<pEphObs->Ndata; idata++)
	    printf("f> %03ld %18.10lf %03ld %6.2f %11.6f %10.6f %11.6f %11.6f %11.4f\n",ieph,
		   pEphObs->JD, idata, pEphObs->lambda[idata], pEphObs->flux[idata], 
		   pEphObs->eflux[idata], pEphObs->ModelFlux[idata], pEphObs->ReflectedFlux[idata], 
		   pEphObs->flux[idata]/pEphObs->ModelFlux[idata]);

	  // copy the vector of the sun from the observations to a temporary variable
	  memcpy(&vSun, &pEphObs->vs, sizeof(VECTOR));
	  // apply the rotation to the asteroid refrence frame
	  Eclip2Ast(&vSun, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pEphObs->JD);
	  // get coords of the Sun
	  GetVecLongLatAsteroid(&vSun, &dLongitude, &dLatitude);
	  printf("ss> %18.10lf %6.1lf %6.1lf\t", pEphObs->JD, dLongitude,dLatitude);
	  // get coords of the Observer
	  GetVecLongLatAsteroid(&pEphObs->vo, &dLongitude, &dLatitude);
	  printf("so> %6.1lf %6.1lf\n", dLongitude,dLatitude);
	  
	  // check if additional flags are present and do some actions accordingly
	  // these needs observations to be present
	  if ((pEphObs->flags & PRINTFACETFLUX)==PRINTFACETFLUX)
	    PrintFacetFlux(ptpm, pEphObs, &pfhtemp, 0);  // FLAGS to zero as there are no craters
	  if ((pEphObs->flags & DUMPFACETFLUX)==DUMPFACETFLUX)
	    DumpFacetFlux(ptpm, pEphObs, &pfhtemp, 0);	 // FLAGS to zero as there are no craters
	}
        
      // check if additional flags are present and do some actions accordingly
      if ((peph[ieph].flags & DUMPTEMP)==DUMPTEMP)
	DumpTemp(ptpm, peph[ieph].JD, &pfhtemp);
      if ((peph[ieph].flags & PRINTTEMP)==PRINTTEMP)
	  PrintTemp(ptpm, &peph[ieph], &pfhtemp);      

      dtEph=(peph[ieph+1].JD - peph[ieph].JD)*86400.; // dtEPH in seconds

      if (dtEph<=ptpm->dtMAX) // we evolve directly to the next ephemeris point
	{
	  // we set tpm.dt to the dtEph
	  ptpm->dt=dtEph;
	  // copy the Sun position
	  vSun.x = peph[ieph].vs.x;
	  vSun.y = peph[ieph].vs.y;
	  vSun.z = peph[ieph].vs.z;
	  // rotate it to the asteroid frame
	  Eclip2Ast(&vSun, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, peph[ieph].JD);	  
	  calcTempTI(ptpm, vSun); // calc the temperatures with thermal inertia
	} else
	{  // we need to do the fucking miniSteps	 
	  // compute the number of ministeps to reach the next ephemeris point
	  nbStep = (long) ceil(dtEph/ptpm->dtMAX);
	  ptpm->dt=dtEph/nbStep;

	  if ((peph[ieph].flags & EPH_FLAGS_SKIPMINISTEP)==EPH_FLAGS_SKIPMINISTEP)
	    {
	      printf("----Eph: %ld MiniStep:%d JD0=%lf  JD1=%lf :: SKIPPED\n", 
		     ieph, -1, peph[ieph].JD,peph[ieph+1].JD);
	      continue; // skip the ministep if flagged !!!
	    }

	  // does the MiniSTEPS here!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	  for (iStep=0; iStep<nbStep; iStep++)
	    {
	      t=iStep*ptpm->dt;
	  // interpolate the sun position around the asteroid (linear)
	      vSun.x = peph[ieph].vs.x + 
		t * (peph[ieph+1].vs.x - peph[ieph].vs.x) / dtEph;
	      vSun.y = peph[ieph].vs.y + 
		t * (peph[ieph+1].vs.y - peph[ieph].vs.y) / dtEph;
	      vSun.z = peph[ieph].vs.z + 
		t * (peph[ieph+1].vs.z - peph[ieph].vs.z) / dtEph;
	  
	      // rotate it to the asteroid frame
	      Eclip2Ast(&vSun, ptpm->mesh.lambda, ptpm->mesh.beta, 
			ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, peph[ieph].JD+t/86400);
	      calcTempTI(ptpm, vSun); // calc the temperatures with thermal inertia
	    } 
	}
      
      printf("--Eph: %ld NminS:%ld JD0=%lf JD1=%lf NiterHD=%ld EPH.dt=%.1lf TPM.dt=%.1lf TPM.dtMAX=%.1lf T[0]=%.2lf\n", 
	     ieph, nbStep, peph[ieph].JD,peph[ieph+1].JD, dtEph,
	     ptpm->dt, ptpm->dtMAX, ptpm->NiterDiffHeat,ptpm->pTemp[0]);
    }//  loop on ephemerides
  /*
  MaxTempDiff(ptpm, &maxTempDiff);
  printf("Tdiff> %lf \n", maxTempDiff);
  if (maxTempDiff<0.5) break;
    }
  */
  if (pfhtemp) fclose(pfhtemp);
  if (pMaxTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MaxTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i], pMaxTemp[i]);
      free(pMaxTemp);
    }
  if (pMinTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MinTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i],pMinTemp[i]);
      free(pMinTemp);
    }

  printf("\nrunTPM completed\n");
  return 1;
}

int TPM_TI_Crat_Lagerros_Flux(TPM* ptpm, 
			       EPH* peph, long Neph, 
			       OBS* pobs, long Nobs)
{
  double		jds=0, jdsold=0;
  unsigned long		i, j, indf;
  time_t  		init, etime;
  double		*prat=NULL;
  FILE			*pfhtemp=NULL;
  FILE                  *pfhfflux=NULL;
  VECTOR                vSun; // vector to the Sun and the Observer.. 
  long                  ieph, iobs, idata, iStep, nbStep;
  OBS                   *pEphObs=NULL; //used just to simplify the reading
  double                miniStepDays;
  double                dtEph, t;
  double                *pMaxTemp=NULL;
  double                *pMinTemp=NULL;
  double                dLongitude, dLatitude;
	
// initialize the pointer to the temperature buffer
// if no temperature buffer, it allocates it
  if (!ptpm->pTemp)
    if ((ptpm->pTemp = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E03: could not allocate pTemp buffer\n");
	return -3;	// error
      }

// initialize the pointer of the illumination vector
// if no illumination vector, it allocates it
  if (!ptpm->mesh.ill)
    if ((ptpm->mesh.ill = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E04: could not allocate pTemp buffer\n");
	return -4;	// error
      }	  

  // ------- INITIALIZE TEMPERATURES -------------------------------------------------------------
// It is the place here, where to set up the intial temperatures
// the idea is to set the deep temperature and all initial temperature to
// the value of the mean surface temperature. The average is taken on one rotation.
// In order to do this, we use a special procedure
  printf("initial temp %lf\n", ptpm->ti);
  if (ptpm->ti==-1.0)
    inittemps(ptpm, peph[0].vs);// we pass the thermal model and the initialposition of sun vector
  else if (ptpm->ti==-2.0) 
    LoadTemp(ptpm); // April 22, 2010: Nice
  else if (ptpm->ti==-3.0) 
    printf("TBD: MeanInsolation Has to be changed\n");
  //    MeanInsolationTemps(ptpm, psun, pjd, Npts); // April 22, 2010: Nice
  else
    {
      printf("resetting temperatures to %lf K \n", ptpm->ti);
      if (ptpm->Gamma > 0) // if there is thermal inertia
	{
	  if (ptpm->pTB)// thermal inertia buffer for heat diffusion is allocated
	    {
	      for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
		for(j=0;j<ptpm->NTslabs; j++)
		  ptpm->pTB[i][j]=ptpm->ti;
	    }

	  if (ptpm->crat)// craters are present thermal inertia buffer for heat diffusion is allocated
	    {
	      for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
		resetcratertemp(&ptpm->crat[i], ptpm->ti);
	    }
	}
    }

// RESET THE MAX and MIN TEMPs----------------------
//  MaxTemp(ptpm,&pMaxTemp);
//  MinTemp(ptpm,&pMinTemp);

// -------------------- RUN THE TPM HERE -------------------------------------------------
  init=time(NULL);
// LAGERROS MODEL -------------------------------------------------------------------------
// Thermal inertia>0 and CRATERS -----------------------------------------------------------
// START THE TPM: Gamma > 0, Crat>0
// -----------------------------------------------------------------------------------------
  miniStepDays=ptpm->dt/86400.0; // ministeps in days
  printf("runTPM started with thermal inertia, and Craters with Lagerros approx. and N=%ld\n", Neph);
  for (ieph=0; ieph<(Neph-1); ieph++) // loop on ephemerides
    {
      // does the ephemeris have a linked observation?
      if (peph[ieph].pobs)
	{ 
	  pEphObs=peph[ieph].pobs; // get the pointer
	  // rotate de vSun to the corotating asteroid frame
	  Eclip2Ast(&pEphObs->vs, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pEphObs->JD);
	  // rotate de vObs to the corotating asteroid frame
	  Eclip2Ast(&pEphObs->vo, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pEphObs->JD);
	  CalcFluxCratLagerros(ptpm, pEphObs->vo, pEphObs->vs, pEphObs->lambda, 
			       pEphObs->ModelFlux,  pEphObs->CraterFlux, pEphObs->Ndata, 
			       pEphObs->FilterIndex, ptpm->pCCorr);
      
	  for(idata=0; idata<pEphObs->Ndata; idata++)
 	    printf("f> %03ld %18.10lf %03ld %6.2f %11.6f %10.6f %11.6f %11.6f %11.4f\n",ieph,
		   pEphObs->JD, idata, pEphObs->lambda[idata], pEphObs->flux[idata], 
		   pEphObs->eflux[idata], pEphObs->ModelFlux[idata], pEphObs->ReflectedFlux[idata], 
		   pEphObs->flux[idata]/pEphObs->ModelFlux[idata]);
	  
	  // copy the vector of the sun from the observations to a temporary variable
	  memcpy(&vSun, &pEphObs->vs, sizeof(VECTOR));
	  // apply the rotation to the asteroid refrence frame
	  Eclip2Ast(&vSun, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pEphObs->JD);
	  // get coords of the Sun
	  GetVecLongLatAsteroid(&vSun, &dLongitude, &dLatitude);
	  printf("ss> %18.10lf %6.1lf %6.1lf\t", pEphObs->JD, dLongitude,dLatitude);
	  // get coords of the Observer
	  GetVecLongLatAsteroid(&pEphObs->vo, &dLongitude, &dLatitude);
	  printf("so> %6.1lf %6.1lf\n", dLongitude,dLatitude);
	  
	  // check if additional flags are present and do some actions accordingly
	  // these needs observations to be present
	  if ((pEphObs->flags & PRINTFACETFLUX)==PRINTFACETFLUX)
	    PrintFacetFlux(ptpm, pEphObs, &pfhtemp, LAGERROSCRATER);
	  if ((pEphObs->flags & DUMPFACETFLUX)==DUMPFACETFLUX)
	    DumpFacetFlux(ptpm, pEphObs, &pfhtemp, LAGERROSCRATER);	      
	}
      
      // check if additional flags are present and do some actions accordingly
      if ((peph[ieph].flags & DUMPTEMP)==DUMPTEMP)
	DumpTemp(ptpm, peph[ieph].JD, &pfhtemp);
      if ((peph[ieph].flags & PRINTTEMP)==PRINTTEMP)
	PrintTemp(ptpm, &peph[ieph], &pfhtemp);
      if ((peph[ieph].flags & PRINTCRATTEMPSTAT)==PRINTCRATTEMPSTAT)
      	PrintCraterTempStatistics(ptpm, peph[ieph].JD, &pfhtemp);	      

      dtEph=(peph[ieph+1].JD - peph[ieph].JD); // ephemeris lapse in days
      // compute the number of ministeps to reach the next ephemeris point
      nbStep = ceil((peph[ieph+1].JD-peph[ieph].JD)/miniStepDays); 
      //	  printf("Eph: %ld MiniStep:%ld\r", ieph, nbStep);
      if ((peph[ieph].flags & EPH_FLAGS_SKIPMINISTEP)==EPH_FLAGS_SKIPMINISTEP)
	{
	  printf("----Eph: %ld MiniStep:%d JD0=%lf  JD1=%lf :: SKIPPED\n", 
		 ieph, -1, peph[ieph].JD,peph[ieph+1].JD);
	  continue; // skip the ministep if flagged !!!
	}
      printf("----Eph: %ld MiniStep:%ld JD0=%lf  JD1=%lf\n", ieph, nbStep, peph[ieph].JD,peph[ieph+1].JD);
      
      // does the MiniSTEPS here!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
      for (iStep=0; iStep<nbStep; iStep++)
	{
	  // check the MaxTemps /// to be removed if not required
	  // MaxTemp(ptpm, &pMaxTemp);/////////
	  
	  t=iStep*miniStepDays;
	  // interpolate the sun position around the asteroid (linear)
	  vSun.x = peph[ieph].vs.x + t
	    * (peph[ieph+1].vs.x - peph[ieph].vs.x) / dtEph;
	  vSun.y = peph[ieph].vs.y + t
	    * (peph[ieph+1].vs.y - peph[ieph].vs.y) / dtEph;
	  vSun.z = peph[ieph].vs.z + t
	    * (peph[ieph+1].vs.z - peph[ieph].vs.z) / dtEph;
	  
	  // rotate it to the asteroid frame
	  Eclip2Ast(&vSun, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, peph[ieph].JD+t);
	  
	  //		  printf("Eph: %ld iStep:%ld Time=%lf\n", ieph, iStep, peph[ieph].JD+t);
	  calcTempTI(ptpm, vSun); // calc the temperatures with thermal inertia
	}	  
    }//  loop on ephemerides
  
  if (pfhtemp) fclose(pfhtemp);
  if (pMaxTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MaxTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i], pMaxTemp[i]);
      free(pMaxTemp);
    }
  if (pMinTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MinTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i],pMinTemp[i]);
      free(pMinTemp);
    }
  
  printf("\nrunTPM completed\n");
  return 1;
}

// FULL HEAT DIFFUSION IN CRATERS ----------------------------------------------
int TPM_TI_Crat_Flux(TPM* ptpm, 
		     EPH* peph, long Neph, 
		     OBS* pobs, long Nobs)
{
  double		jds=0, jdsold=0;
  unsigned long		i, j, indf;
  time_t  		init, etime;
  double		*prat=NULL;
  FILE			*pfhtemp=NULL;
  FILE                  *pfhfflux=NULL;
  VECTOR                vSun; // vector to the Sun and the Observer.. 
  long                  ieph, iobs, idata, iStep, nbStep;
  OBS                   *pEphObs=NULL; //used just to simplify the reading
  double                miniStepDays;
  double                dtEph=0, t=0;
  double                *pMaxTemp=NULL;
  double                *pMinTemp=NULL;
  double                dLongitude, dLatitude;
	
// initialize the pointer to the temperature buffer
// if no temperature buffer, it allocates it
  if (!ptpm->pTemp)
    if ((ptpm->pTemp = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E03: could not allocate pTemp buffer\n");
	return -3;	// error
      }

// initialize the pointer of the illumination vector
// if no illumination vector, it allocates it
  if (!ptpm->mesh.ill)
    if ((ptpm->mesh.ill = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E04: could not allocate pTemp buffer\n");
	return -4;	// error
      }	  

  // ------- INITIALIZE TEMPERATURES -------------------------------------------------------------
// It is the place here, where to set up the intial temperatures
// the idea is to set the deep temperature and all initial temperature to
// the value of the mean surface temperature. The average is taken on one rotation.
// In order to do this, we use a special procedure
  printf("initial temp %lf\n", ptpm->ti);
  if (ptpm->ti==-1.0)
    inittemps(ptpm, peph[0].vs);// we pass the thermal model and the initialposition of sun vector
  else if (ptpm->ti==-2.0) 
    LoadTemp(ptpm); // April 22, 2010: Nice
  else if (ptpm->ti==-3.0) 
    printf("TBD: MeanInsolation Has to be changed\n");
  //    MeanInsolationTemps(ptpm, psun, pjd, Npts); // April 22, 2010: Nice
  else
    {
      printf("resetting temperatures to %lf K \n", ptpm->ti);
      if (ptpm->Gamma > 0) // if there is thermal inertia
	{
	  if (ptpm->pTB)// thermal inertia buffer for heat diffusion is allocated
	    {
	      for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
		for(j=0;j<ptpm->NTslabs; j++)
		  ptpm->pTB[i][j]=ptpm->ti;
	    }

	  if (ptpm->crat)// craters are present thermal inertia buffer for heat diffusion is allocated
	    {
	      for(i=0;i<ptpm->mesh.fn; i++)//loop on facets
		resetcratertemp(&ptpm->crat[i], ptpm->ti);
	    }
	}
    }

// RESET THE MAX and MIN TEMPs----------------------
//  MaxTemp(ptpm,&pMaxTemp);
//  MinTemp(ptpm,&pMinTemp);

// -------------------- RUN THE TPM HERE -------------------------------------------------
  init=time(NULL);
// FULL HEAT DIFFUSION IN CRATERS MODEL ----------------------------------------------------
// Thermal inertia>0 and CRATERS -----------------------------------------------------------
// START THE TPM: Gamma > 0, Crat>0
// -----------------------------------------------------------------------------------------
  miniStepDays=ptpm->dt/86400.0; // ministeps in days
  printf("runTPM started with thermal inertia, and Craters (heat duffusion in craters) and N=%ld\n", Neph);
  for (ieph=0; ieph<(Neph-1); ieph++) // loop on ephemerides
    {
      // does the ephemeris have a linked observation?
      if (peph[ieph].pobs)
	{ 
	  pEphObs=peph[ieph].pobs; // get the pointer
	  // rotate de vSun to the corotating asteroid frame
	  Eclip2Ast(&pEphObs->vs, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pEphObs->JD);
	  // rotate de vObs to the corotating asteroid frame
	  Eclip2Ast(&pEphObs->vo, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pEphObs->JD);
	  calcFluxCrat(ptpm, pEphObs->vo, pEphObs->lambda, 
		       pEphObs->ModelFlux,  pEphObs->CraterFlux, pEphObs->Ndata, 
		       pEphObs->FilterIndex, ptpm->pCCorr);
	  
	  for(idata=0; idata<pEphObs->Ndata; idata++)
 	    printf("f> %03ld %18.10lf %03ld %6.2f %11.6f %10.6f %11.6f %11.6f %11.4f\n",ieph,
		   pEphObs->JD, idata, pEphObs->lambda[idata], pEphObs->flux[idata], 
		   pEphObs->eflux[idata], pEphObs->ModelFlux[idata], pEphObs->ReflectedFlux[idata], 
		   pEphObs->flux[idata]/pEphObs->ModelFlux[idata]);
	  
	  // copy the vector of the sun from the observations to a temporary variable
	  memcpy(&vSun, &pEphObs->vs, sizeof(VECTOR));
	  // apply the rotation to the asteroid refrence frame
	  Eclip2Ast(&vSun, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pEphObs->JD);
	  // get coords of the Sun
	  GetVecLongLatAsteroid(&vSun, &dLongitude, &dLatitude);
	  printf("ss> %18.10lf %6.1lf %6.1lf\t", pEphObs->JD, dLongitude,dLatitude);
	  // get coords of the Observer
	  GetVecLongLatAsteroid(&pEphObs->vo, &dLongitude, &dLatitude);
	  printf("so> %6.1lf %6.1lf\n", dLongitude,dLatitude);
	  
	  // check if additional flags are present and do some actions accordingly
	  // these needs observations to be present
	  if ((pEphObs->flags & PRINTFACETFLUX)==PRINTFACETFLUX)
	    PrintFacetFlux(ptpm, pEphObs, &pfhtemp, HEATDIFFUSIONCRATER); // heat diffusion is taken into account inside craters
	  if ((pEphObs->flags & DUMPFACETFLUX)==DUMPFACETFLUX)
	    DumpFacetFlux(ptpm, pEphObs, &pfhtemp, HEATDIFFUSIONCRATER); // heat diffusion is taken into account inside craters  
	}
      
      // check if additional flags are present and do some actions accordingly
      if ((peph[ieph].flags & DUMPTEMP)==DUMPTEMP)
	DumpTemp(ptpm, peph[ieph].JD, &pfhtemp);
      if ((peph[ieph].flags & PRINTTEMP)==PRINTTEMP)
	PrintTemp(ptpm, &peph[ieph], &pfhtemp);
      if ((peph[ieph].flags & PRINTCRATTEMPSTAT)==PRINTCRATTEMPSTAT)
      	PrintCraterTempStatistics(ptpm, peph[ieph].JD, &pfhtemp);	      

      dtEph=(peph[ieph+1].JD - peph[ieph].JD); // ephemeris lapse in days
      if (dtEph<miniStepDays) continue;

      // compute the number of ministeps to reach the next ephemeris point
      nbStep = (long)ceil(dtEph/miniStepDays); 
      //	  printf("Eph: %ld MiniStep:%ld\r", ieph, nbStep);
      if ((peph[ieph].flags & EPH_FLAGS_SKIPMINISTEP)==EPH_FLAGS_SKIPMINISTEP)
	{
	  printf("----Eph: %ld MiniStep:%d JD0=%lf  JD1=%lf :: SKIPPED\n", 
		 ieph, -1, peph[ieph].JD,peph[ieph+1].JD);
	  continue; // skip the ministep if flagged !!!
	}
      printf("----Eph: %ld MiniStep:%ld JD0=%lf  JD1=%lf\n", ieph, nbStep, peph[ieph].JD,peph[ieph+1].JD);
      
      // does the MiniSTEPS here!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
      for (iStep=0; iStep<nbStep; iStep++)
	{
	  // check the MaxTemps /// to be removed if not required
	  // MaxTemp(ptpm, &pMaxTemp);/////////
	  
	  t=iStep*miniStepDays;
	  // interpolate the sun position around the asteroid (linear)
	  vSun.x = peph[ieph].vs.x + t
	    * (peph[ieph+1].vs.x - peph[ieph].vs.x) / dtEph;
	  vSun.y = peph[ieph].vs.y + t
	    * (peph[ieph+1].vs.y - peph[ieph].vs.y) / dtEph;
	  vSun.z = peph[ieph].vs.z + t
	    * (peph[ieph+1].vs.z - peph[ieph].vs.z) / dtEph;
	  
	  // rotate it to the asteroid frame
	  Eclip2Ast(&vSun, ptpm->mesh.lambda, ptpm->mesh.beta, 
		    ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, peph[ieph].JD+t);
	  
	  //		  printf("Eph: %ld iStep:%ld Time=%lf\n", ieph, iStep, peph[ieph].JD+t);
	  calcTempTICrat(ptpm, vSun); // calc the temperatures with thermal inertia
	}
      dtEph=0; // reset it
    }//  loop on ephemerides
  
  if (pfhtemp) fclose(pfhtemp);
  if (pMaxTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MaxTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i], pMaxTemp[i]);
      free(pMaxTemp);
    }
  if (pMinTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MinTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i],pMinTemp[i]);
      free(pMinTemp);
    }
  
  printf("\nrunTPM completed\n");
  return 1;
}


int TPM_noTI_Crat_Lagerros_Flux(TPM* ptpm, 
	   EPH* peph, long Neph, 
           OBS* pobs, long Nobs)
{
  double		jds=0, jdsold=0;
  unsigned long		i, j, indf;
  time_t  		init, etime;
  double		*prat=NULL;
  FILE			*pfhtemp=NULL;
  FILE                  *pfhfflux=NULL;
  VECTOR                vSun; // vector to the Sun and the Observer.. 
  long                  ieph, iobs, idata, iStep, nbStep;
  OBS                   *pEphObs=NULL; //used just to simplify the reading
  double                miniStepDays;
  double                dtEph, t;
  double                *pMaxTemp=NULL;
  double                *pMinTemp=NULL;
  double                dLongitude, dLatitude;
	
// initialize the pointer to the temperature buffer
// if no temperature buffer, it allocates it
  if (!ptpm->pTemp)
    if ((ptpm->pTemp = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E03: could not allocate pTemp buffer\n");
	return -3;	// error
      }

// initialize the pointer of the illumination vector
// if no illumination vector, it allocates it
  if (!ptpm->mesh.ill)
    if ((ptpm->mesh.ill = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E04: could not allocate pTemp buffer\n");
	return -4;	// error
      }	  


// -------------------- RUN THE TPM HERE -------------------------------------------------
  init=time(NULL);


// Thermal inertia ZERO and CRATERS  -----------------------------------------------------------
// START THE TPM: Gamma=0, Crat>0, Expansion=0
// LAGERROS MODEL -------------------------------------------------------------------------
  printf("runTPM started with NO thermal inertia, Craters with Lagerros Approximation and N=%ld\n", Neph);
  for (iobs=0; iobs<Nobs; iobs++) // loops on observations
    {
      // rotate de vSun to the corotating asteroid frame
      Eclip2Ast(&pobs[iobs].vs, ptpm->mesh.lambda, ptpm->mesh.beta, 
		ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pobs[iobs].JD);
      // rotate de vObs to the corotating asteroid frame
      Eclip2Ast(&pobs[iobs].vo, ptpm->mesh.lambda, ptpm->mesh.beta, 
		ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pobs[iobs].JD);
      calcTempCrat(ptpm, pobs[iobs].vs);	  
      //      calcFluxCrat(ptpm, pobs[iobs].vo, pobs[iobs].lambda, pobs[iobs].ModelFlux, 
      CalcFluxCratLagerros(ptpm, pobs[iobs].vo, pobs[iobs].vs, pobs[iobs].lambda, 
			   pobs[iobs].ModelFlux, pobs[iobs].CraterFlux, pobs[iobs].Ndata, 
                           pobs[iobs].FilterIndex, ptpm->pCCorr);
      for(idata=0; idata<pobs[iobs].Ndata; idata++)
	printf("f> %03ld %18.10lf %03ld %6.2f %11.6f %10.6f %11.6f %11.6f %11.4f\n", iobs,
	       pobs[iobs].JD, idata, pobs[iobs].lambda[idata], pobs[iobs].flux[idata], 
	       pobs[iobs].eflux[idata], pobs[iobs].ModelFlux[idata], pobs[iobs].ReflectedFlux[idata],
	       pobs[iobs].flux[idata]/pobs[iobs].ModelFlux[idata]);
	  
      // copy the vector of the sun from the observations to a temporary variable
      memcpy(&vSun, &pobs[iobs].vs, sizeof(VECTOR));
      // apply the rotation to the asteroid refrence frame
      Eclip2Ast(&vSun, ptpm->mesh.lambda, ptpm->mesh.beta, 
		ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pobs[iobs].JD);
      // get coords of the Sun
      GetVecLongLatAsteroid(&vSun, &dLongitude, &dLatitude);
      printf("ss> %18.10lf %6.1lf %6.1lf\t", pobs[iobs].JD, dLongitude,dLatitude);
      // get coords of the Observer
      GetVecLongLatAsteroid(&pobs[iobs].vo, &dLongitude, &dLatitude);
      printf("so> %6.1lf %6.1lf\n", dLongitude,dLatitude);

      // check if additional flags are present and do some actions accordingly
      // these needs observations to be present
      if ((pobs[iobs].flags & PRINTFACETFLUX)==PRINTFACETFLUX)
	PrintFacetFlux(ptpm, &pobs[iobs], &pfhtemp, LAGERROSCRATER);
      if ((pobs[iobs].flags & DUMPFACETFLUX)==DUMPFACETFLUX)
	DumpFacetFlux(ptpm, &pobs[iobs], &pfhtemp, LAGERROSCRATER);	      
      if ((pobs[iobs].flags & PRINTCRATTEMPSTAT)==PRINTCRATTEMPSTAT)
      	PrintCraterTempStatistics(ptpm, pobs[iobs].JD, &pfhtemp);	      
  }

  if (pfhtemp) fclose(pfhtemp);
  if (pMaxTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MaxTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i], pMaxTemp[i]);
      free(pMaxTemp);
    }
  if (pMinTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MinTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i],pMinTemp[i]);
      free(pMinTemp);
    }
  
  printf("\nrunTPM completed\n");
  return 1;
}

int TPM_noTI_Crat_Flux(TPM* ptpm, 
	   EPH* peph, long Neph, 
           OBS* pobs, long Nobs)
{
  double		jds=0, jdsold=0;
  unsigned long		i, j, indf;
  time_t  		init, etime;
  double		*prat=NULL;
  FILE			*pfhtemp=NULL;
  FILE                  *pfhfflux=NULL;
  VECTOR                vSun; // vector to the Sun and the Observer.. 
  long                  ieph, iobs, idata, iStep, nbStep;
  OBS                   *pEphObs=NULL; //used just to simplify the reading
  double                miniStepDays;
  double                dtEph, t;
  double                *pMaxTemp=NULL;
  double                *pMinTemp=NULL;
	
// initialize the pointer to the temperature buffer
// if no temperature buffer, it allocates it
  if (!ptpm->pTemp)
    if ((ptpm->pTemp = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E03: could not allocate pTemp buffer\n");
	return -3;	// error
      }

// initialize the pointer of the illumination vector
// if no illumination vector, it allocates it
  if (!ptpm->mesh.ill)
    if ((ptpm->mesh.ill = (double*) malloc(sizeof(double)*ptpm->mesh.fn))==NULL)
      {
	printf("RunTPM E04: could not allocate pTemp buffer\n");
	return -4;	// error
      }	  


// -------------------- RUN THE TPM HERE -------------------------------------------------
  init=time(NULL);


// Thermal inertia ZERO and CRATERS  -----------------------------------------------------------
// START THE TPM: Gamma=0, Crat>0, Expansion=0
  printf("runTPM started with NO thermal inertia, Craters, and N=%ld\n", Neph);
  for (iobs=0; iobs<Nobs; iobs++) // loops on observations
    {
      // rotate de vSun to the corotating asteroid frame
      Eclip2Ast(&pobs[iobs].vs, ptpm->mesh.lambda, ptpm->mesh.beta, 
		ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pobs[iobs].JD);
      // rotate de vObs to the corotating asteroid frame
      Eclip2Ast(&pobs[iobs].vo, ptpm->mesh.lambda, ptpm->mesh.beta, 
		ptpm->mesh.P, ptpm->mesh.phi0, ptpm->mesh.t0, pobs[iobs].JD);
      calcTempCrat(ptpm, pobs[iobs].vs);	  
      calcFluxCrat(ptpm, pobs[iobs].vo, pobs[iobs].lambda, 
		   pobs[iobs].ModelFlux, pobs[iobs].CraterFlux, pobs[iobs].Ndata, 
		   pobs[iobs].FilterIndex, ptpm->pCCorr);

      for(idata=0; idata<pobs[iobs].Ndata; idata++)
	printf("f> %03ld %18.10lf %03ld %6.2f %11.6f %10.6f %11.6f %11.6f %11.4f\n", iobs,
	       pobs[iobs].JD, idata, pobs[iobs].lambda[idata], pobs[iobs].flux[idata], 
	       pobs[iobs].eflux[idata], pobs[iobs].ModelFlux[idata], pobs[iobs].ReflectedFlux[idata],
	       pobs[iobs].flux[idata]/pobs[iobs].ModelFlux[idata]);
	  
      PrintSubSolarSubObserver(&pobs[iobs]);    

      // check if additional flags are present and do some actions accordingly
      // these needs observations to be present
      if ((pobs[iobs].flags & PRINTFACETFLUX)==PRINTFACETFLUX)
	PrintFacetFlux(ptpm, &pobs[iobs], &pfhtemp, HEATDIFFUSIONCRATER);
      if ((pobs[iobs].flags & DUMPFACETFLUX)==DUMPFACETFLUX)
	DumpFacetFlux(ptpm, &pobs[iobs], &pfhtemp, HEATDIFFUSIONCRATER);	      
      if ((pobs[iobs].flags & PRINTCRATTEMPSTAT)==PRINTCRATTEMPSTAT)
      	PrintCraterTempStatistics(ptpm, pobs[iobs].JD, &pfhtemp);
     if ((pobs[ieph].flags & PRINTTEMP)==PRINTTEMP)
	PrintTemp(ptpm, pobs[iobs].peph, &pfhtemp);

      dumpcrater(ptpm->crat,"pippo.crat",pobs[iobs].vs,pobs[iobs].vo,ptpm->LUT); // REMOVE
  }

  if (pfhtemp) fclose(pfhtemp);
  if (pMaxTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MaxTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i], pMaxTemp[i]);
      free(pMaxTemp);
    }
  if (pMinTemp)
    {
      for (i=0; i<ptpm->mesh.fn; i++)
	printf("MinTemp> %5d %6.1lf %6.1lf\t%7.2lf\n", i, 
	       ptpm->mesh.Longitude[i], ptpm->mesh.Latitude[i],pMinTemp[i]);
      free(pMinTemp);
    }
  
  printf("\nrunTPM completed\n");
  return 1;
}

