#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef max
	#define max( a, b ) ( ((a) > (b)) ? (a) : (b) )
#endif

#ifndef min
	#define min( a, b ) ( ((a) < (b)) ? (a) : (b) )
#endif

#define  RECLEN 512

#define USAGE "\n\
  -h this help\n\
  -f <filename> change the filename of the TMAP file\n\
  -n <ntslabs> number of depth slabs e.g. 32, 64 [default=32] \n\
  -v Set to VERBOSE output mode \n"
 
typedef struct tagTdata
{
  double  *pjd;
  long    *pindex;
  double  *plambda;
  double  *pbeta;
  double  **pt;
  long     nt;
  long     ntslabs;
} Tdata;


int ReadTemps(char * fname,  Tdata *ptdata)
{
	FILE * file_in;
	double	*pjd=NULL, *plambda=NULL, *pbeta=NULL, **pt;
	char linebuf[RECLEN];
	long npt=0, i, j, *pindex=NULL;

// try to open the file...
	if ((file_in = fopen( fname, "r"))== NULL ) 
	  {
	  printf("E TSTAT_C 00 Temperature File not found\n");
	  return 0; //if does not succed, return 0
	  }

	// count number of lines in the file: only temperature data must be present
	while (1)
		{
		if (NULL==fgets(linebuf,RECLEN,file_in)) break;
			npt++;
		}

	fclose(file_in);//close the file in order to rewind the pointer
	//	printf("%s",linebuf);

		// allocate the memory for JD
	pjd=malloc(npt*sizeof(double)); 
	if (pjd==NULL) 
		{
		printf("E TSTAT_C 01 Memory not allocated\n");
		return 0; //failed
		}

	pindex=malloc(npt*sizeof(long));
	if (pindex==NULL) 
		{
		if (pjd!=NULL) free(pjd);
		printf("E TSTAT_C 02 Memory not allocated\n");
		return 0; //failed
		}

	plambda=malloc(npt*sizeof(double));
	if (plambda==NULL) 
		{
		if (pjd!=NULL) free(pjd);
		if (pindex!=NULL) free(pindex);
		printf("E TSTAT_C 03 Memory not allocated\n");
		return 0; //failed
		}

	pbeta=malloc(npt*sizeof(double));
	if (pbeta==NULL) 
		{
		if (pjd!=NULL) free(pjd);
		if (pindex!=NULL) free(pindex);
		if (plambda!=NULL) free(plambda);
		printf("E TSTAT_C 04 Memory not allocated\n");
		return 0; //failed
		}
    
	pt=malloc(npt*sizeof(double*));
	if (pt==NULL) 
		{
		if (pjd!=NULL) free(pjd);
		if (pindex!=NULL) free(pindex);
		if (plambda!=NULL) free(plambda);
		if (pbeta!=NULL) free(pbeta);
		printf("E TSTAT_C 04 Memory not allocated\n");
		return 0; //failed
		}
	for (i=0; i<npt; i++)
	  {
          pt[i]=malloc(ptdata->ntslabs*sizeof(double));
          if (pt[i]==NULL) break;
	  }
	  
	if (i<(npt-1)) //roll back
	  {
	    for (j=i; j>=0; j--)
	      if (pt[j]!=NULL) free(pt[j]);

      	    if (pjd!=NULL) free(pjd);
	    if (pindex!=NULL) free(pindex);
	    if (plambda!=NULL) free(plambda);
       	    if (pbeta!=NULL) free(pbeta);
       	    if (pt!=NULL) free(pt);
	  }
      
	// pass back the addresses of the vectors

	ptdata->nt=npt;        // pass back the number of points
	ptdata->pjd=pjd;
	ptdata->pindex=pindex;
	ptdata->plambda=plambda;
	ptdata->pbeta=pbeta;
	ptdata->pt=pt;
 
	// reopen the file
	if ((file_in = fopen( fname, "r"))== NULL ) 
	{
	  printf("E TSTAT_C 05 File not open\n");
	  for (j=0; j<npt; j++)
	      if (pt[j]) free(pt[j]);

      	    if (pjd) free(pjd);
	    if (pindex) free(pindex);
	    if (plambda) free(plambda);
       	    if (pbeta) free(pbeta);
       	    if (pt) free(pt);
		return 0;
	}
    
	for(i=0; i<npt; i++)
		{
		  //		printf("%ld..",i);
		if ((fscanf(file_in, "%lf %ld %lf %lf", &pjd[i], &pindex[i], &plambda[i], &pbeta[i]))!=4)
		  {
		    printf("E TSTAT_C 06 Error reading file\n");
		    for (j=0; j<npt; j++)
		      if (pt[j]!=NULL) free(pt[j]);

		    if (pjd) free(pjd);
		    if (pindex) free(pindex);
		    if (plambda) free(plambda);
		    if (pbeta) free(pbeta);
		    if (pt) free(pt);
		    return 0;
		  }
  		for (j=0; j<ptdata->ntslabs; j++)
		  if (fscanf(file_in,"%lf",&pt[i][j])!=1)		
		    {
		      printf("E TSTAT_C 07 Error reading file\n");
		      for (j=0; j<npt; j++)
			if (pt[j]) free(pt[j]);

		      if (pjd!=NULL) free(pjd);
		      if (pindex!=NULL) free(pindex);
		      if (plambda!=NULL) free(plambda);
		      if (pbeta!=NULL) free(pbeta);
		      if (pt!=NULL) free(pt);
		      return 0;
		    }
		}

	fclose(file_in);// close the file and..
		// returns
	return 1;
	}

double meantemp(Tdata *ptdata)
{
  long i;
  double aT=0.0;
  
  for (i=0; i<ptdata->nt; i++)
    aT+=ptdata->pt[i][0];//surface temperature

  aT/=ptdata->nt;
  return aT;
}

double maxtemp(Tdata *ptdata)
{
  long i;
  double T=0.0;
  
  for (i=0; i<ptdata->nt; i++)
    T = max(T, ptdata->pt[i][0]);

  return T;
}

double mintemp(Tdata *ptdata)
{
  long i;
  double T=1.e7;
  
  for (i=0; i<ptdata->nt; i++)
    T = min(T, ptdata->pt[i][0]);

  return T;
}

main(int argc, char *argv[])
{
  FILE *fn=NULL;
  unsigned int output = 0;
  char filename[256];
  double *pjd=NULL;
  int  ntslabs=32;
  double **ppt=NULL;
  double *plambda=NULL, *pbeta=NULL;
  long   *pindex=NULL;
  Tdata  tdata={NULL,NULL,NULL,NULL,NULL,0,32};

  strcpy(filename, "tmap_temps.dat");

        while (argc>1)
        {
                if (argv[1][0]=='-')
                {
		  switch (argv[1][1])
		    {
                      case 'v': output+=1;    
                        break;
		      case 'n': 
			  argc--;
			  argv++;
			  tdata.ntslabs=atol(argv[1]);
			break;
		      case 'h': printf("%s\n",USAGE); exit(0);
                        break;
		      case 'f': 
			argc--;
			argv++;
			strcpy(filename, argv[1]);
                        break; 
                    }
                } 
                argc--;
                argv++;
        }
  
	if ((output & 1)==1)
          printf("Number of slabs=%ld\n",tdata.ntslabs);
	ReadTemps(filename, &tdata);
	if ((output & 1)==1)
	  printf("Red %ld temps\n",tdata.nt);
	if ((output & 1)==1)
	  printf("Tmax %lf\t<T> %lf\tTmin %lf\n",  maxtemp(&tdata), meantemp(&tdata), mintemp(&tdata));
	else
	  printf("%lf\t%lf\t%lf\n",  maxtemp(&tdata), meantemp(&tdata), mintemp(&tdata));
}
