#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MODE_TSURF 1
#define MODE_TMAX  2
#define MODE_TMUL  4

typedef struct Tfacet
{
  int      index;
  float    lambda;
  float    beta;
  float    area;
  float    *pT;
} TFACET;

typedef struct Tmesh
{
  int     fn;//number of facets
  int     NTslabs;// number of slabs
  double  JD;//julian date 
  float   dx; 
  float   dt;
  TFACET  *pTf;
}TMESH;

#ifndef max
	#define max( a, b ) ( ((a) > (b)) ? (a) : (b) )
#endif

#ifndef min
	#define min( a, b ) ( ((a) < (b)) ? (a) : (b) )
#endif

#define  RECLEN 512

#define USAGE "\n\
 enter <temp.file name> <number of observations> <command> \n\
 \n\
 optional paramaters \n\ 
  -h this help\n\
  -f <filename> change the filename of the TMAP file\n\
  -n <ntslabs> number of depth slabs e.g. 32, 64 [default=32] \n\
  -v Set to VERBOSE output mode \n\
LIST OF COMMANDS \n\
  tsurf \n\
  tmax  \n\
  tmul  \n"


main(int argc, char *argv[])
{
  FILE *fn=NULL;
  char filename[256];
  char command[256];
  int mode;
  int verbose=0;
  float  ls;
  TMESH tmesh;
  int i, index, j, io, dummyi, k;
  float *pT;
  int Nobs;
  float ttr;
  float area=0, totarea=0;
  float volume=0;
  float tmax=0, tmin=1.e6;
  float temp=0, a=0, depth=0;

  while (argc>1)
  {
          if (argv[1][0]=='-')
          {
  	  switch (argv[1][1])
  	    {
	          case 'v': verbose=1;
                  break;
  	        case 'n': printf("%s\n",USAGE); exit(0);
  		  argc--;
  		  argv++;
  		break;
  	        case 'h': printf("%s\n",USAGE); exit(0);
                  break;
  	        case 'f': printf("%s\n",USAGE); exit(0);
                  break; 
              }
          } 
          argc--;
          argv++;
  }

    scanf("%s",filename);//read the filename
    scanf("%d",&Nobs);//read the number of observation per file
    scanf("%s",command);//read the command

    if (strcmp(command, "tsurf")==0) mode=MODE_TSURF;
    if (strcmp(command, "tmax")==0) mode=MODE_TMAX;
    if (strcmp(command, "tmul")==0) mode=MODE_TMUL;

    switch (mode)
      {
      case MODE_TSURF:
	scanf("%f",&ls);//read the heat length scale
	scanf("%f",&depth);//read the depth
	scanf("%f",&ttr);//read temp. threshold
	break;
      }

    if ( (fn=(FILE*)fopen(filename,"rb"))==NULL)
      {
	printf("tsurf: cannot open inputfile!!!\n");
	exit(0);
      }

    fread(&tmesh.fn,sizeof(tmesh.fn),1,fn);
    fread(&tmesh.NTslabs,sizeof(tmesh.NTslabs),1,fn);

    if (verbose) printf("mesh has %d facets and %d slabs\n", tmesh.fn, tmesh.NTslabs);

    if ((tmesh.pTf=(TFACET*)malloc(sizeof(TFACET)*tmesh.fn))==NULL)
      {
	printf("tsurf: Alloc error\n");
	exit(0);
      }
	
    if ((pT=(float*)malloc(sizeof(float)*tmesh.NTslabs))==NULL)
      {
	printf("tsurf: Alloc error N1\n");
	exit(0);
      }

    for(i=0; i<tmesh.fn; i++)
      {
	if ((tmesh.pTf[i].pT=(float*)malloc(sizeof(float)*tmesh.NTslabs))==NULL)
          {
	    printf("tsurf: Alloc error N2\n");
	    exit(0);
	  }
	tmesh.pTf[i].index=i;
	tmesh.pTf[i].beta=0;
	tmesh.pTf[i].lambda=0;		
	tmesh.pTf[i].area=0;
	memset(tmesh.pTf[i].pT,0,sizeof(float)*tmesh.fn);
      }

    for (io=0; io<Nobs; io++)
      {
	if (io>0)
	  {
	    fread(&dummyi,sizeof(tmesh.fn),1,fn);
	    fread(&dummyi,sizeof(tmesh.NTslabs),1,fn);
	  }
	//read JD
	fread(&tmesh.JD,sizeof(tmesh.JD),1,fn);
	//	printf("%lf\t",tmesh.JD);
	// read dt dx
	fread(&tmesh.dt,sizeof(tmesh.dt),1,fn);
	fread(&tmesh.dx,sizeof(tmesh.dx),1,fn);

	// for each facet reads for each facet reads facet_index lambda, beta, area;
	for(i=0; i<tmesh.fn; i++)
	  {
	    fread(&index,sizeof(index),1,fn);
	    //	    printf("%d %d>>", i, index);
	    fread(&tmesh.pTf[index].lambda,sizeof(tmesh.pTf[index].lambda),1,fn);
	    fread(&tmesh.pTf[index].beta,sizeof(tmesh.pTf[index].beta),1,fn);
	    fread(&tmesh.pTf[index].area,sizeof(tmesh.pTf[index].area),1,fn);
	    // read temps
	    //	    printf("%f %f %f>>", tmesh.pTf[index].lambda, tmesh.pTf[index].beta, tmesh.pTf[index].area);
	    fread(pT,sizeof(pT[0]),tmesh.NTslabs,fn);

	    for(j=0; j<tmesh.NTslabs; j++)
	      {
		tmesh.pTf[index].pT[j]=max(pT[j],tmesh.pTf[index].pT[j]);
		//		printf("%f ",pT[j]);
	      }
	    //	    printf("\n..");
	  }
	//	printf("obs %d completed\n",io);
      }

    if (mode==MODE_TSURF)
      {
	for(i=0; i<tmesh.fn; i++)
	  {

	    // calc the temp at the subsurface
	    // find nearest subsurface slab
	    k=(int)(depth/(tmesh.dx*ls));
	    if (depth==0) 
	      {
		temp=tmesh.pTf[i].pT[0]; 
	      }
	    else if (depth>=tmesh.NTslabs*tmesh.dx*ls)
	      {
		temp=tmesh.pTf[i].pT[tmesh.NTslabs-1]; 
	      }
	    else 
	      {
		a=(tmesh.pTf[i].pT[k+1]-tmesh.pTf[i].pT[k])/(tmesh.dx*ls);
		temp=tmesh.pTf[i].pT[k] + a*(depth-tmesh.dx*ls*k);
	      }

	    if (temp >= ttr)
	      {
		area+=tmesh.pTf[i].area;
		volume+=tmesh.pTf[i].area*tmesh.dx*ls;
	      }
	    for (j=1; j<tmesh.NTslabs; j++)
	      if (temp >= ttr) volume+=tmesh.pTf[i].area*tmesh.dx*ls;
	  }
	
	totarea=0;
	for(i=0; i<tmesh.fn; i++)
	  totarea+=tmesh.pTf[i].area;

	printf("o> %f %f %f %f\n", area, totarea, 
	       volume, totarea*tmesh.dx*ls*tmesh.NTslabs);
      }

    if (mode==MODE_TMAX)
      {
	for(i=0; i<tmesh.fn; i++)
	  {
	    tmax=max(tmax,tmesh.pTf[i].pT[0]);
	    tmin=min(tmin,tmesh.pTf[i].pT[0]);
	  }
	printf("tm> %s\t%f\t%f\n", filename, tmax, tmin); 	
      }

    // deallocate all
   for(i=0; i<tmesh.fn; i++) 
      free(tmesh.pTf[i].pT);
   free(tmesh.pTf);
   free(pT);
   fclose(fn);  
}
