// MDBOrbit.c : This is the implementation of library functions
//
#include <stdio.h>
#include <math.h>

#include "SLALIB/slalib.h"
#include "SLALIB/slamac.h"

#define GK		0.01720209895e0                         // to calculate the mean motion in rad/days
#define PI		3.1415926535897932384626433832795
#define RAD		0.017453292519943295769236907684886 	// PI/180
#define OBL		0.4090928042223289096135375d0		// J2000 obliquity in radians, IAU
#define COBL		0.91748206206918183682560225347235	// cosine of the J2000 obliquity in radians, IAU
#define SOBL		0.39777715593191367603721308989317	// sine of the J2000 obliquity in radians, IAU


extern	int		kepler(double *E, double M, double e, double derror, int method);
void			orb2inertial(double *x, double *r, double omega, double Omega, double i);


// FUNCTIONs PROTOTYPEs for DLL exports --------------------------------------------------------------
#ifdef MDBORBIT_DLL

#include <windows.h>

BOOL APIENTRY DllMain(HINSTANCE hinstDLL,
                      DWORD  ul_reason_for_call,
                      LPVOID lpReserved);

__declspec ( dllexport ) int WINAPI LIBelem2xyz(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBelem2xyzVEC(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBelem2xyzN(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBEarthxyz(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBEarthxyzVEC(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBelemda2xyz(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBelemda2xyzVEC(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBnormalizeVEC(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBnormaVECN(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBxyz2equator(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBxyz2ecliptic(int argc,void* argv[]);
__declspec ( dllexport ) int WINAPI LIBxy2pha(int argc,void* argv[]);
#endif

#ifndef MDBORBIT_DLL
  #define WINAPI
int WINAPI LIBelem2xyz(int argc,void* argv[]);
int WINAPI LIBelem2xyzVEC(int argc,void* argv[]);
int WINAPI LIBelem2xyzN(int argc,void* argv[]);
int WINAPI LIBEarthxyz(int argc,void* argv[]);
int WINAPI LIBEarthxyzVEC(int argc,void* argv[]);
int WINAPI LIBelemda2xyz(int argc,void* argv[]);
int WINAPI LIBelemda2xyzVEC(int argc,void* argv[]);
int WINAPI LIBnormalizeVEC(int argc,void* argv[]);
int WINAPI LIBnormaVECN(int argc,void* argv[]);
int WINAPI LIBxyz2equator(int argc,void* argv[]);
int WINAPI LIBxy2pha(int argc,void* argv[]);
#endif


// Global variables ------------------------------------------------------------------------
long	SizeV = sizeof(double)*3;


// FUNCTIONs Implementataion ------------------------------------------------------------------------
#ifdef MDBORBIT_DLL
BOOL APIENTRY DllMain( HMODULE hModule,
                       DWORD  ul_reason_for_call,
                       LPVOID lpReserved
					 )
{
	SizeV=sizeof(double)*3;
    return TRUE;
}
#endif

int WINAPI LIBEarthxyz(int argc,void* argv[])
{
	double	*x	= (double*)	argv[0];
	double	JD	= *(double*)	argv[1];
/*
	float	pv[6];
	slaEarthMDB(JD,pv);
	x[0]=(double)pv[0];
	x[1]=(double)pv[1];
        x[2]=(double)pv[2];
*/
	double dvh[3], dph[3], dvb[3], dpb[3];

	slaEvp (JD-2400000.5, 2000.0, dvb,  dpb, dvh,  dph );
	x[0]=dph[0];
	x[1]=dph[1];
	x[2]=dph[2];

	return 1;
}

int WINAPI LIBEarthxyzVEC(int argc,void* argv[])
{
	double	*x	= (double*)		argv[0];
	double	*JD	= (double*)		argv[1];
	long	Nt	= *(long*)		argv[2];
	long	j;
	double* px=x;
/*
	float	pv[6];
	for (j=0; j<Nt; j++)
	{
		slaEarthMDB(JD[j],pv);
		px[0]=(double)pv[0];
		px[1]=(double)pv[1];
		px[2]=(double)pv[2];
		px+=3;
	}
*/
	double dvh[3], dph[3], dvb[3], dpb[3];

	for (j=0; j<Nt; j++)
	{
		slaEvp (JD[j]-2400000.5, 2015.0, dvb,  dpb, dvh,  dph );
		px[0]=dph[0];
		px[1]=dph[1];
		px[2]=dph[2];
		px+=3;
	}

	return j;
}

int WINAPI LIBelem2xyz(int argc,void* argv[])
{
	double *x		=  (double*) argv[0];	// position vector in the inertial reference frame
							// external 3 elem vector allocated by the calling procedure
	double M		= *(double*) argv[1];
	double omega	        = *(double*) argv[2];
	double Omega	        = *(double*) argv[3];
	double inc		= *(double*) argv[4];
	double e		= *(double*) argv[5];
	double a		= *(double*) argv[6];
	double n		= *(double*) argv[7];
	double t0		= *(double*) argv[8];   // mean motion in rad/days
	double t		= *(double*) argv[9];   // reference epoch JD (days)
	double Mt;					// new anomaly: Mt=M+n(t-t0)
	double E;					// eccentric anomaly
	double r[3];					// position vector in the orbital reference frame

//    printf("%lf %lf %lf\n", x[0], x[1], x[2]);
//    printf("%lf %lf %lf\n", M, omega, a);

	if (n<=0)								// if mean motion is zero
		n=GK/a/sqrt(a);						// calculates mean motion

	Mt = M + n*(t-t0);						// calculates the new mean anomaly
	kepler(&E, Mt, e, .0000001, 1);			// solve the kepler equation

	r[0]=a*(cos(E)-e);
	r[1]=a*sqrt(1-e*e)*sin(E);
	r[2]=0.0;

//    printf("%lf %lf %lf\n", omega, Omega, inc);
//    printf("%lf %lf %lf\n", r[0], r[1], r[2]);
//    printf("%lf %lf %lf\n", x[0], x[1], x[2]);
	orb2inertial(x,r,omega,Omega,inc);
//    printf("%lf %lf %lf\n", x[0], x[1], x[2]);

//    x[0]=1; x[1]=2; x[2]=3;
	return 1;
}

/*
   N asteroids
   1 epoch
*/

int WINAPI LIBelem2xyzN(int argc,void* argv[])
{
	double *xv		=  (double*) argv[0];	// position vector in the inertial reference frame
								 			// external 3 elem vector allocated by the calling procedure
	double *M		= (double*) argv[1];
	double *omega	= (double*) argv[2];
	double *Omega	= (double*) argv[3];
	double *inc		= (double*) argv[4];
	double *e		= (double*) argv[5];
	double *a		= (double*) argv[6];
	double *n		= (double*) argv[7];
	double *t0		= (double*) argv[8];
	double t		= *(double*) argv[9];
	long   Nast		= *(long*)	 argv[10];
	double Mt;								// new anomaly: Mt=M+n(t-t0)
	double E;								// eccentric anomaly
	double r[3];							// position vector in the orbital reference frame
	long   j;
	double* px=xv;							// used for pointer algebra. Point to xv
	double np;


	for (j=0; j<Nast; j++)					// loop on objects
	{
		if (n[j]<=0)								// if mean motion is zero
		  np=GK/a[j]/sqrt(a[j]);					// calculates mean motion
		else
		  np=n[j];

		Mt = M[j] + np*(t-t0[j]);						// calculates the new mean anomaly
		kepler(&E, Mt, e[j], .0000001, 1);			// solve the kepler equation

		r[0]=a[j]*(cos(E)-e[j]);
		r[1]=a[j]*sqrt(1-e[j]*e[j])*sin(E);
		r[2]=0.0;

		orb2inertial(px,r,omega[j],Omega[j],inc[j]);
		px+=3;								// increment the pointer
	}
	return 1;
}
/*
	1	ASTEROID
	Nt	Epochs
*/

int WINAPI LIBelem2xyzVEC(int argc,void* argv[])
{
	double *xv		=  (double*) argv[0];	// position vector in the inertial reference frame
								 			// external 3 elem vector allocated by the calling procedure
	double M		= *(double*) argv[1];
	double omega	= *(double*) argv[2];
	double Omega	= *(double*) argv[3];
	double inc		= *(double*) argv[4];
	double e		= *(double*) argv[5];
	double a		= *(double*) argv[6];
	double n		= *(double*) argv[7];
	double t0		= *(double*) argv[8];
	double* t		= (double*)	 argv[9];
	long   Nt		= *(long*)	 argv[10];
	double Mt;								// new anomaly: Mt=M+n(t-t0)
	double E;								// eccentric anomaly
	double r[3];							// position vector in the orbital reference frame
	long   j;
	double* px=xv;							// used for pointer algebra. Point to xv

	if (n<=0)								// if mean motion is zero
		n=GK/a/sqrt(a);						// calculates mean motion

	for (j=0; j<Nt; j++)
	{
		Mt = M + n*(t[j]-t0);						// calculates the new mean anomaly
		kepler(&E, Mt, e, .0000001, 1);			// solve the kepler equation

		r[0]=a*(cos(E)-e);
		r[1]=a*sqrt(1-e*e)*sin(E);
		r[2]=0.0;

		orb2inertial(px,r,omega,Omega,inc);
		px+=3;								// increment the pointer
	}
	return 1;
}

void orb2inertial(double *x, double *r, double omega, double Omega, double i)
{
	double	co=cos(omega),
			so=sin(omega),
			cO=cos(Omega),
			sO=sin(Omega),
			ci=cos(i),
			si=sin(i);
	double  x1tmp;

	x[0]= co*cO*r[0] - so*sO*ci*r[0] 	//R11*r1
		 -so*cO*r[1] - co*sO*ci*r[1] 	//R12*r2
		 +sO*si*r[2];					//R13*r3

	x[1]= co*sO*r[0] + so*cO*ci*r[0] 	//R21*r1
		 -so*sO*r[1] + co*cO*ci*r[1]    //R22*r2
		 -cO*si*r[2];					//R23*r3

	x[2]= so*si*r[0]					//R31*r1
		 +co*si*r[1]					//R32*r2
		 +ci*r[2];					//R33*r3

// ------ rotate to equatorial reference frame ---------------------------------
// remove this lines if you want ECLIPTIC heliocentric coordinates
//	x1tmp = x[1]*COBL - x[2]*SOBL;
//	x[2]  = x[1]*SOBL + x[2]*COBL;
//	x[1]  = x1tmp;
// ----------------------------------------------ROTATE to EQUATORIAL ----------
}



int WINAPI LIBelemda2xyzVEC(int argc,void* argv[])
{
	double *xv		=  (double*) argv[0];	// position vector in the inertial reference frame
								 			// external 3 elem vector allocated by the calling procedure
	double M		= *(double*) argv[1];
	double omega	= *(double*) argv[2];
	double Omega	= *(double*) argv[3];
	double inc		= *(double*) argv[4];
	double e		= *(double*) argv[5];
	double a0		= *(double*) argv[6];
	double n		= *(double*) argv[7];
	double t0		= *(double*) argv[8];
	double da_dt	= *(double*) argv[9];
	double *t		=  (double*) argv[10];
	long   Nt		= *(long*)   argv[11];
	double Mt;								// new anomaly: Mt=M+n(t-t0)
	double E;								// eccentric anomaly
	double r[3];							// position vector in the orbital reference frame
	double dt;
	long	j;
	double* px=xv;							// used for pointer algebra. Point to xv
	double	a;								// varied a

	for (j=0; j<Nt; j++)
	{
		dt=t[j]-t0;

		a=a0+da_dt*dt/365.25;

		if (n<=0)								// if mean motion is zero
			n=GK/a/sqrt(a);						// calculates mean motion

		Mt = M + n*dt							// calculates the new mean anomaly
//			-1.5*PI/sqrt(a*a*a*a*a)*da_dt*dt*dt  // including its variation due to da/dt [my formula]
//			/133407.5625;						// 365.25^2
			-0.75 * (n/a) * da_dt * dt*dt / 365.25; // Vokrouhlicky et al, 2000 Icarus 148, 118

		kepler(&E, Mt, e, .0000001, 1);			// solve the kepler equation

		r[0]=a*(cos(E)-e);
		r[1]=a*sqrt(1-e*e)*sin(E);
		r[2]=0.0;

		orb2inertial(px,r,omega,Omega,inc);
		px+=3;
	}
/*
	x[0]=r[0];
	x[1]=r[1];
	x[2]=r[2];
*/
	return 1;
}

int WINAPI LIBnormalizeVEC(int argc,void* argv[])
{
	double	*v		= (double*)	argv[0];	// dimension of vector along x
	long	Ndim	= *(long*)	argv[1];
	long	Nvect	= *(long*)	argv[2];	// number of vectors along y

	long	xi, yi;
	double  norm;

	for (yi=0; yi<Nvect; yi++)
	{
		norm=0.0;
		for (xi=0; xi<Ndim; xi++)
			norm+=v[xi+yi*Ndim]*v[xi+yi*Ndim];	// sum values^2
		norm=sqrt(norm);
		for (xi=0; xi<Ndim; xi++)
			v[xi+yi*Ndim]/=norm;
	}

}

int WINAPI LIBnormaVECN(int argc,void* argv[])
{
	double	*v		= (double*)	argv[0];
	double  *norm	= (double*)	argv[1];
	long	Nvect	= *(long*)	argv[2];
	long	i;

	for (i=0; i<Nvect; i++)
	{
		norm[i]=sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);	// sum values^2
		v+=3;
	}
	return 1;
}

/* given the array of N vectors r in the equatorial reference frame
   return the RA and the DE of each one
   Tested against HORIZON on Aug 7, 2007
*/
int	WINAPI LIBxyz2equator(int argc,void* argv[])
{
	double *r ;
	double *ra;
	double *de;
	long    N ;
	long	i;
	double	xynorm;
	double	*x;

	if (argc<3) return -1;
	if (argc<4) N=1; else N = *(long*)	argv[3];

	r  = (double*)	argv[0];
	ra = (double*)	argv[1];
	de = (double*)	argv[2];

	x=r;
	for (i=0; i<N; i++)
	{
		xynorm=sqrt(x[0]*x[0] + x[1]*x[1]);
		ra[i]=atan2(x[1],x[0]);
		if (ra[i] < 0) ra[i]+=D2PI;
		de[i]=atan2(x[2],xynorm);
		x+=3;
	}
	return 1;
}

/* given the array of N vectors r in the equatorial reference frame
   return the ecliptc lambda and beta of each one
   Tested against HORIZON on Aug 7, 2007
*/
int	WINAPI LIBxyz2ecliptic(int argc,void* argv[])
{
	double 	*r ;
	double 	*lambda;
	double 	*beta;
	long   	N ;
	long	i;
	double	xynorm;
	double	*x;
	double  xe[3];

	if (argc<3) return -1;
	if (argc<4) N=1; else N = *(long*)	argv[3];

	r  		= (double*)	argv[0];
	lambda	= (double*)	argv[1];
	beta 	= (double*)	argv[2];

	x=r;
	for (i=0; i<N; i++)
	{
      	xe[1] =  x[1]*COBL + x[2]*SOBL;
      	xe[2] = -x[1]*SOBL + x[2]*COBL;
      	xe[0] = x[0];
		xynorm=sqrt(xe[0]*xe[0] + xe[1]*xe[1]);
		lambda[i]=atan2(xe[1],xe[0]);
		if (lambda[i] < 0) lambda[i]+=D2PI;
		beta[i]=atan2(xe[2],xynorm);
		x+=3;
	}
	return 1;
}


/* given the array of N vectors x and y returns the phase angle
   Tested against HORIZON on Aug 7, 2007
*/
int	WINAPI LIBxy2pha(int argc,void* argv[])
{
	double *x;
	double *y;
	double *pha;
	long    N ;
	long	i;
	double	xnorm;
	double  ynorm;
	double	ps;

	if (argc<4) return -1;

	x   = (double*)	argv[0];
	y   = (double*)	argv[1];
	pha = (double*)	argv[2];
	N   = *(long*)	argv[3];

	for (i=0; i<N; i++)
	{
		xnorm=sqrt(x[0]*x[0] + x[1]*x[1] + x[2]*x[2]);
		ynorm=sqrt(y[0]*y[0] + y[1]*y[1] + y[2]*y[2]);
		pha[i]=acos((x[0]*y[0] + x[1]*y[1] + x[2]*y[2])/xnorm/ynorm);
		x+=3;
		y+=3;
	}
	return 1;
}
