# Find observations of a single numbered asteroid from the MPCOBS
# database and print out an IPAC style table of the designation, MJD,
# Ra, Dec.
#
# Requires AstroLibCoords for date to JD conversion and
# find-numbered.py to search the database.
#
# - Michael S. Kelley, UMD, 2011-05-18
#
# 2011-05-19 - Fixed so that RA is in degrees like it is supposed to
#              be.  Tweaked the formatting.  MSK.
#
# - Victor M. Ali-Lagoa, IAC, 2011-07
#
# 2011-07-03 - Fixed ZeroDivisionError in 'def conv(s):'
#              Need a platform that supports zeros! VAL

import sys
import subprocess
from datetime import datetime
import coords
import math

# IPAC table format: 
#  http://irsa.ipac.caltech.edu/applications/DDGEN/Doc/ipac_tbl.html

######################################################################
# s2i = string to integer
s2i = lambda x: int(float(x))  

######################################################################=
# convert find-numbered output to object, MJD, ra, dec where ra, dec
# are decimal degrees
def conv(s):
    ss = s.split()
    y, m, d = s2i(ss[1]), s2i(ss[2]), s2i(ss[3])
    f = (float(ss[3]) - d) * 24
    hr = int(f)
    f = (f - hr) * 60
    mm = int(f)
    f = (f - mm) * 60
    sec = int(f)
    f = (f - sec) * 1e6
    um = int(f)
    jd = coords.astrodate.utc2jd(datetime(y, m, d, hr, mm, sec, um))
    mjd = jd - coords.astrodate.MJD_0

    ra = float(ss[4]) + float(ss[5]) / 60.0 + float(ss[6]) / 3600.0
    ra *= 15.0
    sgn = math.copysign(1.0, float(ss[7]))
#    sgn = float(ss[7]) / float(ss[7])
    dec = sgn * (abs(float(ss[7])) + float(ss[8]) / 60.0 +
                 float(ss[9]) / 3600.0)
    return "  {0:10}   {1:11.5f}   {2:10.6f}   {3:10.6f}".format(
        ss[0], mjd, ra, dec)

######################################################################
if len(sys.argv) < 2:
    print """
Find observations of a single numbered asteroid from an MPCOBS file
and print out an IPAC style table of the designation, MJD, Ra, Dec.

Usage:
  {0} asteroid_number [mpcobs_file]
""".format(sys.argv[0])
    sys.exit(1)

aster = sys.argv[1]
if len(sys.argv) > 2:
    obsfile = sys.argv[2]
else:
    obsfile = 'total_wise_mpc_list.txt'

cmd = "python find-numbered.py {0} {1}".format(aster, obsfile)
pipe = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE).stdout

print "|    desg    |    mjd      |    ra      |    dec     |"
print "|char        |double       |double      |double      |"
try:
    print "\n".join(map(conv, pipe.readlines()))
except IndexError:
    pass

pipe.close()
