# Find observations of a single numbered asteroid from the MPCOBS
# database and print out an IPAC style table of the designation, MJD,
# Ra, Dec.
#
# Requires AstroLibCoords for date to JD conversion and
# find-numbered.py to search the database.
#
# - Michael S. Kelley, UMD, 2011-05-18
#
# 2011-05-19 - Fixed so that RA is in degrees like it is supposed to
#              be.  Tweaked the formatting.  MSK.
# 
# 2011-07-04 - Kept Michael Kelley's code and comments format.
#              Extended the script to work either with an asteroid 
#              or with a list input from a file containing a single
#              column of asteroid numbers. VMAL
# 
#            - Fixed ZeroDivisionError in 'def conv(s):'
#               Needs a platform that supports signed zeros! VMAL
# 
# 2011-07-07 - Extended to use also lists of any combination of numbered
#              and unnumbered designations. VMAL
#
# - Victor M. Ali-Lagoa, IAC, 2011-07-08
#
# 2011-09-08 - Extended for use with packed asteroid numbers as well as 
#              unnumbered designations with underscore. VMAL
# 




import sys
import subprocess
from datetime import datetime
import coords
import math

# IPAC table format: 
# http://irsa.ipac.caltech.edu/applications/DDGEN/Doc/ipac_tbl.html

######################################################################
# s2i = string to integer
s2i = lambda x: int(float(x))  

######################################################################
# convert find-numbered output to object, MJD, ra, dec where ra, dec
# are decimal degrees
def conv(s):
    ss = s.split()
    y, m, d = s2i(ss[1]), s2i(ss[2]), s2i(ss[3])
    f = (float(ss[3]) - d) * 24
    hr = int(f)
    f = (f - hr) * 60
    mm = int(f)
    f = (f - mm) * 60
    sec = int(f)
    f = (f - sec) * 1e6
    um = int(f)
    jd = coords.astrodate.utc2jd(datetime(y, m, d, hr, mm, sec, um))
    mjd = jd - coords.astrodate.MJD_0

    ra = float(ss[4]) + float(ss[5]) / 60.0 + float(ss[6]) / 3600.0
    ra *= 15.0
    sgn = math.copysign(1.0, float(ss[7]))
    dec = sgn * (abs(float(ss[7])) + float(ss[8]) / 60.0 +
                 float(ss[9]) / 3600.0)
    return "  {0:10}   {1:11.5f}   {2:10.6f}   {3:10.6f}".format(
        ss[0], mjd, ra, dec)

######################################################################

if len(sys.argv) < 2:
    print """
Find WISE observations of a single asteroid (or a file containing a list
in column form) from an MPCOBS file and print out an IPAC style table of 
the designation, MJD, Ra, Dec. Compound names for single-asteroid input 
in the command-line should be included in quotes when a space is used to 
separate the year of discovery from the letters (if it is in a list, 
this is not required). It is also possible to use an underscore: "_". See 
an example below. In all cases, the letters in the input designation must 
be capital letters.

It makes use of 'find-numbered.py' and 'find-unnumbered.py', 
which must be kept in the same folder.

i)  Single-asteroid input:
    {0} asteroid_number [mpcobs_file]
e.g.
    {0} '2005 XD'
    {0} K05X00D 
    {0} 2005_XD
are equivalent.
    
ii) With a file containing a column of asteroid designations (numbered
    or unnumbered):
    {0} file_name [mpcobs_file] 
    
NB: The first column in [mpcobs_file] must not have any indentation 
(first caracter of each line must be a number or a letter, never a blank space)

NB2: There must be no empty rows in the file_name and none or one single 
carriage return at the bottom of the list. If it had more than one, the output
would still be correct, but until we improve on this, it is nicer to stick with 
the rules.

NB3: option ii) requires a platform that supports signed zeros for the math.copysign() function to work.
""".format(sys.argv[0])
    sys.exit(1)

if len(sys.argv) > 2:
    obsfile = sys.argv[2]
else:
    obsfile = 'WISEUnnObs.txt'

sys_argv1 = sys.argv[1]

######################################################################
# Define the dictionaries that will do the conversion from unnumbered 
# designation to the MPC's unnumbered designation
dicc = dict([('18', 'I'), ('19', 'J'), ('20', 'K')])
packed_dicc = dict([('10', 'A'),('11', 'B'),('12', 'C'),('13', 'D'),('14', 'E'),('15', 'F'),('16', 'G'),('17', 'H'),('18', 'I'),('19', 'J'),('20', 'K'),('21', 'L'),('22', 'M'),('23', 'N'),('24', 'O'),('25', 'P'),('26', 'Q'),('27', 'R'),('28', 'S'),('29', 'T'),('30', 'U'),('31', 'V'),('32', 'W'),('33', 'X'),('34', 'Y'),('35', 'Z'),('36', 'a'),('37', 'b'),('38', 'c'),('39', 'd'),('40', 'e'),('41', 'f'),('42', 'g'),('43', 'h'),('44', 'i'),('45', 'j'),('46', 'k'),('47', 'l'),('48', 'm'),('49', 'n'),('50', 'o'),('51', 'p'),('52', 'q'),('53', 'r'),('54', 's'),('55', 't'),('56', 'u'),('57', 'v'),('58', 'w'),('59', 'x'),('60', 'y'),('61', 'z')])
######################################################################

if sys_argv1.isdigit():
    # if the input in the command line is a number everything works as 
    # in mpc2ipac.py

    obsfile = 'total_wise_mpc_list.txt'
    aster = int(sys_argv1)
    cmd = "python find-numbered.py {0} {1}".format(aster, obsfile)
    pipe = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE).stdout   
    print "|    desg    |    mjd      |    ra      |    dec     |"
    print "|char        |double       |double      |double      |"
    pipe = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE).stdout
    try:
        print "\n".join(map(conv, pipe.readlines()))
    except IndexError:
        pass
    
    pipe.close()

else:
    # If it is not a number it might be a file, so we try to open a file.
    # In case it is, we iteratively work with the each row - check the 
    # enumerate() procedure.
    print "|    desg    |    mjd      |    ra      |    dec     |"
    print "|char        |double       |double      |double      |"  
    try:
        file_with_aster = open(sys_argv1)
        for i, aster_line in enumerate(file_with_aster):
#            print "{0}".format(i)
            obsfile = 'WISEUnnObs.txt'
            aster = aster_line.split()
            if len(aster) > 1:
                uno = dicc[aster[0][0:2]]
                dyt = aster[0][2:]
                cuat = aster[1][0]
                siet = aster[1][1]
                middle = aster[1][2:]
                try:
                    mi = int(middle)
                    if mi > 99:
                        # get the packed designation
                        pmid = packed_dicc[middle[0:2]]
                        medio  = pmid + middle[2:]                   
                    else:
                        # pad with zeros
                        medio = "{0:02d}".format(int(mi))
      
                except ValueError:
                    medio = '00'

                finally:
                    cadena = uno+dyt+cuat+medio+siet
                cmd = "python find-unnumbered.py {0} {1}".format(cadena, obsfile)
               
            else:
                if aster[0].isdigit():
                    obsfile = 'total_wise_mpc_list.txt'
                    cmd = "python find-numbered.py {0} {1}".format(int(aster[0]), obsfile)
                elif len(aster[0]) < 6:
                    obsfile = 'total_wise_mpc_list.txt'
                    cmd = "python find-numbered.py {0} {1}".format(aster[0], obsfile)
                else:
                    if aster[0].find("_") > 0:
                        aster0 = aster[0].replace("_"," ").split()
                        uno = dicc[aster0[0][0:2]]
                        dyt = aster0[0][2:]
                        cuat = aster0[1][0]
                        siet = aster0[1][1]
                        middle = aster0[1][2:]
                        try:
                            mi = int(middle)
                            if mi > 99:
                        # get the packed designation
                                pmid = packed_dicc[middle[0:2]]
                                medio  = pmid + middle[2:]                   
                            else:
                        # pad with zeros
                                medio = "{0:02d}".format(int(mi))
      
                        except ValueError:
                            medio = '00'

                        finally:
                            cadena = uno+dyt+cuat+medio+siet
                        cmd = "python find-unnumbered.py {0} {1}".format(cadena, obsfile)
                    else:
                        cmd = "python find-unnumbered.py {0} {1}".format(aster[0],obsfile)
#            print "Ejecutando:\n{0}".format(cmd)
            pipe = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE).stdout
            try:
                print "\n".join(map(conv, pipe.readlines()))
            except IndexError:
                pass
    
            pipe.close()


        file_with_aster.close()
    except IOError:

        # If we made it here then it means the input is not a file
        aster = sys_argv1.split()
        if len(aster) > 1:
            uno = dicc[aster[0][0:2]]
            dyt = aster[0][2:]
            cuat = aster[1][0]
            siet = aster[1][1]
            middle = aster[1][2:]
            try:
                mi = int(middle)
                if mi > 99:
                    # get the packed designation
                    pmid = packed_dicc[middle[0:2]]
                    medio  = pmid + middle[2:]                   
                else:
                    # pad with zeros
                    medio = "{0:02d}".format(int(mi))
            except ValueError:
                medio = '00'

            finally:
                cadena = uno+dyt+cuat+medio+siet
                cmd = "python find-unnumbered.py {0} {1}".format(cadena, obsfile)
            
        else:
            if aster[0].isdigit():
                obsfile = 'total_wise_mpc_list.txt'
                cmd = "python find-numbered.py {0} {1}".format(int(aster[0]), obsfile)
            elif len(aster[0]) < 6:
                obsfile = 'total_wise_mpc_list.txt'
                cmd = "python find-numbered.py {0} {1}".format(aster[0], obsfile)
            else:
                if aster[0].find("_") > 0:
                    aster0 = aster[0].replace("_"," ").split()
                    uno = dicc[aster0[0][0:2]]
                    dyt = aster0[0][2:]
                    cuat = aster0[1][0]
                    siet = aster0[1][1]
                    middle = aster0[1][2:]
                    try:
                        mi = int(middle)
                        if mi > 99:
                        # get the packed designation
                            pmid = packed_dicc[middle[0:2]]
                            medio  = pmid + middle[2:]                   
                        else:
                        # pad with zeros
                            medio = "{0:02d}".format(int(mi))
      
                    except ValueError:
                        medio = '00'

                    finally:
                        cadena = uno+dyt+cuat+medio+siet
                    cmd = "python find-unnumbered.py {0} {1}".format(cadena, obsfile)
                else:
                    cmd = "python find-unnumbered.py {0} {1}".format(aster[0],obsfile)

#        print "Ejecutando:\n{0}".format(cmd)
        pipe = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE).stdout
        try:
            print "\n".join(map(conv, pipe.readlines()))
        except IndexError:
            pass
        
        pipe.close()
